/****************************************************************************

  fixstub - Reads and manipulates DOS exe (executable) file headers.
  (fixes (updates) an old style DOS executable so can be used as the
   DOS stub in one of the new style executables)

  Written to read in and update to latest format (compatible
  to be stub within PE (and probably LE and NE) executables)
  of the DOS exe header.

  Version 0.00
  Written by: Kenneth J. Davis
  Date:       Novemeber 07, 2000, Feb 2001
  Contact:    jeremyd@computer.org


Copyright (c): Public Domain [United States Definition]

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR AUTHORS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT
OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

****************************************************************************/

typedef unsigned short WORD;  /* 2 bytes */
typedef unsigned long DWORD;   /* 4 bytes */

#include "exehdr.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>


/* Verifies base exe header, 0=ok */
int testValidDosHdr(EXEHEADER_BASE *baseHdr);

/* updates sizes to match new exehdr sizes */
void updateExeHeader(EXEHEADER_BASE *baseHdr, EXEHEADER_BASE *newBaseHdr);


int main(int argc, char *argv[])
{
  FILE *fin, *fout;
  EXEHEADER_BASE baseHdr, newBaseHdr;
  EXEHEADER_LINKINFO unusedHdr;
  EXEHEADER_EXT extendedHdr;
  unsigned char relocData[4];
  long i;
  int c;


  if (argc != 3)
  {
    printf("Usage: fixstub source.exe out.exe\n");
    printf("\nfixstub will take a DOS only executable and update its exe header\n"
           "to be compatible with the new exe header (PE, possibly LE/NE)\n"
           "The primary purpose is to compile a program with an older DOS compiler\n"
           "and use the resulting executable as the stub of a win32 program.\n"
           "\nA few simple tests are done and if the file does not seem to contain\n"
           "a valid DOS executable or it appears to already be a new exe header\n"
           "then no action is taken.\n\n"
           "source.exe is the original unmodified DOS executable\n"
           "out.exe is the name to save the modified DOS exe to.\n\n");
    return 1;
  }

  if ((fin = fopen(argv[1], "rb")) == NULL)
  {
    printf("Error opening source exe: %s\n", argv[1]);
    return 1;
  }

  if ((fout = fopen(argv[2], "wb")) == NULL)
  {
    printf("Error opening output exe: %s\n", argv[2]);
    fclose(fin);
    return 1;
  }


  if (fread(&baseHdr, sizeof(struct EXEHEADER_BASE), 1, fin) != 1)
  {
    printf("Error reading in base exe header.\n");
    fclose(fin);
    fclose(fout);
    return 1;
  }

  if (testValidDosHdr(&baseHdr))
  {
    printf("Does not appear to be a valid DOS exe header.\n");
    fclose(fin);
    fclose(fout);
    return 1;
  }

  if (baseHdr.relocTable >= 0x40)
  {
    printf("Dos exe header seems updated already.\n");
    fclose(fin);
    fclose(fout);
    return 1;
  }

  memset(&unusedHdr, 0, sizeof(struct EXEHEADER_LINKINFO));
  memset(&extendedHdr, 0, sizeof(struct EXEHEADER_EXT));
  memset(&newBaseHdr, 0, sizeof(struct EXEHEADER_BASE));

  updateExeHeader(&baseHdr, &newBaseHdr);

  /* write out new header */
  if (fwrite(&newBaseHdr, sizeof(struct EXEHEADER_BASE), 1, fout) != 1)
  {
    printf("Error writing new base exe header to %s\n", argv[2]);
    fclose(fin);
    fclose(fout);
    return 1;
  }
  if (fwrite(&unusedHdr, sizeof(struct EXEHEADER_LINKINFO), 1, fout) != 1)
  {
    printf("Error writing unused/linker portion of exe header to %s\n", argv[2]);
    fclose(fin);
    fclose(fout);
    return 1;
  }
  if (fwrite(&extendedHdr, sizeof(struct EXEHEADER_EXT), 1, fout) != 1)
  {
    printf("Error writing extended portion of exe header to %s\n", argv[2]);
    fclose(fin);
    fclose(fout);
    return 1;
  }


  /* write out relocation data, currently only handles *
   * no reloc table or reloc table fully in header     */
  if (baseHdr.relocCnt)
  {
    if (fseek(fin, baseHdr.relocTable, SEEK_SET)) /* seek to start of reloc data */
    {
      printf("Error seeking to start of relocation data in source file.\n");
      fclose(fin);
      fclose(fout);
      return 1;
    }
  
    for (i = 0; i < baseHdr.relocCnt; i++)
    {
      if (fread(relocData, 4, 1, fin) != 1) /* read in one reloc entry */
      {
        printf("Error unexepected EOF on source exe: %s\n", argv[1]);
        fclose(fin);
        fclose(fout);
        return 1;
      }
      if (fwrite(relocData, 4, 1, fout) != 1)
      {
        printf("Error unable to write %ith relocation entry to %s\n",
          (i+1), argv[2]);
        fclose(fin);
        fclose(fout);
        return 1;
      }
    } /* end for() */
  }

  /* write out header padding */
  i = (newBaseHdr.hdrSize * 16) -         /* calc pad bytes needed */
      (sizeof(struct EXEHEADER_BASE) +
       sizeof(struct EXEHEADER_LINKINFO) +
       sizeof(struct EXEHEADER_EXT) +
       newBaseHdr.relocCnt * 4);
  if (i < 0)
  {
    printf("INTERNAL Error: unexpected negative padding length!\n");
    fclose(fin);
    fclose(fout);
    return 2;
  }
  while (i)
  {
    fputc(0, fout);   /* write out 0s for padding bytes */
    i--;
  }



  /* write out the rest of the image, until eof */
  if (fseek(fin, (baseHdr.hdrSize * 16),SEEK_SET)) /* seek to end of exe header */
  {
    printf("Error seeking to end of exe header in source file.\n");
    fclose(fin);
    fclose(fout);
    return 1;
  }

  c = fgetc(fin);
  while (!feof(fin))
  {
    fputc(c, fout);
    c = fgetc(fin);
  }

  fclose(fin);
  fclose(fout);

  printf("Done.\n");
  return 0;
}


/* Verifies base exe header, 0=ok */
int testValidDosHdr(EXEHEADER_BASE *baseHdr)
{
  if (baseHdr->magic != MAGIC_DOS) return 1;
  if (baseHdr->pages == 0) return 1;
  if (baseHdr->hdrSize == 0) return 1;
  return 0;
}

/**
 * updates sizes to match new exehdr sizes 
 * Assumes no extra information in header
 * other than base header and possibly relocation
 * data.  All information after header is kept,
 * but if must not make assumptions about its
 * absolute location in file (assumptions from
 * end of header should be fine).
 */
void updateExeHeader(EXEHEADER_BASE *baseHdr, EXEHEADER_BASE *newBaseHdr)
{
  unsigned long temp;
  unsigned long relocTableSize;
  WORD extendedHdrSize = sizeof(struct EXEHEADER_BASE) + 
                                  sizeof(struct EXEHEADER_LINKINFO) +
                                  sizeof(struct EXEHEADER_EXT);


  newBaseHdr->magic = MAGIC_DOS;
  newBaseHdr->relocCnt = baseHdr->relocCnt;
  newBaseHdr->minAlloc = baseHdr->minAlloc;
  newBaseHdr->maxAlloc = baseHdr->maxAlloc;
  newBaseHdr->SS = baseHdr->SS;
  newBaseHdr->SP = baseHdr->SP;
  newBaseHdr->checksum = 0;
  newBaseHdr->IP = baseHdr->IP;
  newBaseHdr->CS = baseHdr->CS;
  newBaseHdr->overlayNumber = baseHdr->overlayNumber;

  relocTableSize = baseHdr->relocCnt * 4;

  /* see if there is a relocation table */
  if (baseHdr->relocCnt != 0)
  {
    /* see if contained within the header (starts inside header) */
    if (baseHdr->relocTable < (baseHdr->hdrSize * 16))
    {
      /* new header size is size of extended header + relocation data. */
      temp = extendedHdrSize + relocTableSize;

      newBaseHdr->hdrSize = (WORD)((temp+15) / 16); /* round up */
      newBaseHdr->relocTable = extendedHdrSize;
    }
    else
    {
      printf("Error: currently unable to handle cases that have relocation table\n"
             "outside of exe header.\n");
      exit(2);
    }
  }
  else /* no relocation table */
  {
      /* new header size is size of extended header. */
      newBaseHdr->hdrSize = (extendedHdrSize+15) / 16; /* round up */
      newBaseHdr->relocTable = 0;
  }


  /* update image size, assume old-size = old-header-size + rest, and
     new-size = old-size - old-header-size + new-header-size 
              = new-header-size + rest */
  temp = ((baseHdr->pages-1) * 512) + baseHdr->lastPageSize -
         (baseHdr->hdrSize * 16) + (newBaseHdr->hdrSize * 16);
  newBaseHdr->lastPageSize = (WORD)(temp % 512UL);
  newBaseHdr->pages = (WORD)((temp+512UL) / 512UL); /* round up */
}
