REM  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
REM  *  MiNES - Mateusz's iNES Editor                                *
REM  *  Written by Mateusz Viste <mateusz@viste-family.net>...       *
REM  *   ...using FreeBASIC v0.20.0                                  *
REM  *  Website: http://www.viste-family.net/mateusz/software/mines/ *
REM  *  Licence: GNU/GPLv3                                           *
REM  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
REM
REM This program is free software: you can redistribute it and/or modify
REM it under the terms of the GNU General Public License as published by
REM the Free Software Foundation, either version 3 of the License, or
REM (at your option) any later version.
REM
REM This program is distributed in the hope that it will be useful,
REM but WITHOUT ANY WARRANTY; without even the implied warranty of
REM MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
REM GNU General Public License for more details.
REM
REM You should have received a copy of the GNU General Public License
REM along with this program.  If not, see <http://www.gnu.org/licenses/>.
REM

#INCLUDE ONCE "file.bi"      ' Needed to use the 'FileExists' function.
#INCLUDE ONCE "dir.bi"       ' Needed when listing ROMs across directories.

#IFNDEF __FB_WIN32__
 REM The line below is needed because FBC v0.18.3 hasn't correct headers.
 REM Beside that, the truncate function doesn't work in the Windows port.
 #INCLUDE ONCE "crt.bi"       ' Needed to use the 'Truncate' function.
 DECLARE FUNCTION Truncate CDECL ALIAS "truncate" (BYVAL __file AS ZSTRING PTR, BYVAL __length AS INTEGER) AS INTEGER
#ENDIF

REM  **** ** * USER-DEFINIED TYPES * ** ****
TYPE BOOL AS BYTE        ' Creating the BOOL type, as it
CONST TRUE AS BOOL = 1   ' is not supported natively by
CONST FALSE AS BOOL = 0  ' the FreeBASIC compiler.

TYPE iNES
  ROMbanks AS UBYTE
  VROMbanks AS UBYTE
  Mirroring AS BOOL
  BattRAM AS BOOL
  Trainer AS BOOL         ' The header's setting
  TrainerDetected AS BYTE ' Header presence assumption from the ROM's size [0 = No trainer ; 1 = Trainer ; -1 = Can't guess]
  FourScreen AS BOOL
  VS AS BYTE              '0 = Normal NES ; 1 = VS ; 2 = PC10)
  Mapper AS UBYTE
  RAMbanks AS UBYTE
  PAL AS BOOL
  Title AS STRING
  TitleLen AS INTEGER ' Title presence assumption from the ROM's size [0 = No title ; 128 or 127 = Title ; -1 = Can't guess]
  Header(1 TO 16) AS UBYTE
  FileSize AS INTEGER ' Used for checking if ROM/VROM numbers of banks aren't too high.
  CRC32 AS UINTEGER
  GoodNES AS BOOL
END TYPE

REM  **** ** * FUNCTIONS/SUBS DECLARATIONS * ** ****
DECLARE SUB About()
DECLARE SUB ReadHeader(NESfile AS STRING)
DECLARE SUB DecodeHeader()
DECLARE SUB EncodeHeader(CleanFlag AS BYTE = 0)
DECLARE SUB KeybFlush()
DECLARE SUB DrawScreen()
DECLARE SUB HeaderPreview()
DECLARE SUB DisplayComment(Komentarz AS STRING)
DECLARE SUB DisplayStatusLine()
DECLARE SUB LoadMapperList()
DECLARE SUB GetNESTitle(NESfile AS STRING)
DECLARE SUB GetNesCRC(NESfile AS STRING)
DECLARE SUB DisplayHelp()
DECLARE SUB GetRomInformations(RomFile AS STRING)
DECLARE SUB ParseRoms(BYVAL StartPath AS STRING)
DECLARE SUB CheckRomWarnings()
DECLARE FUNCTION SaveNESTitle(NESfile AS STRING) AS BOOL
DECLARE FUNCTION SaveHeader(NESfile AS STRING) AS BOOL
DECLARE FUNCTION GetGoodInfos(RomCRC AS UINTEGER) AS BYTE
DECLARE FUNCTION AskForTitle(Proposition AS STRING = "") AS STRING
DECLARE FUNCTION AskWarning(Ostrz1 AS STRING, Ostrz2 AS STRING = "", Ostrz3 AS STRING = "", Ostrz4 AS STRING = "", Ostrz5 AS STRING = "", Ostrz6 AS STRING = "", Ostrz7 AS STRING = "", Ostrz8 AS STRING = "", Ostrz9 AS STRING = "", Ostrz10 AS STRING = "") AS STRING
DECLARE FUNCTION CheckNESFile(NESfile AS STRING) AS BYTE
DECLARE FUNCTION CRC32(BYVAL lpBuffer AS ZSTRING PTR, BYVAL BufferSize AS INTEGER) AS UINTEGER
DECLARE FUNCTION XorString(StringToXor AS STRING) AS STRING
DECLARE FUNCTION SelectFile() AS STRING
DECLARE FUNCTION HeaderGarbage(Header() AS UBYTE) AS BOOL

REM  **** ** * CONSTANTS * ** ****
CONST pVer AS STRING = "0.96"
CONST pDate AS STRING = "2009"

REM  **** ** * SHARED VARIABLES * ** ****
DIM SHARED HorVer(0 TO 1) AS STRING*11 => {"Horizontal ", "Vertical   "}
DIM SHARED NtscPal(0 TO 1) AS STRING*11 => {"NTSC       ", "PAL        "}
DIM SHARED NoYes(0 TO 1) AS STRING*11 => {"No         ", "Yes        "}
DIM SHARED DontknowNoYes(-1 TO 1) AS STRING*11 => {"Can't guess", "No         ", "Yes        "}
DIM SHARED ClassVS(0 TO 2) AS STRING*11 => {"Normal NES ", "VS-System  ", "PC 10      "}
DIM SHARED AS iNES OrgROM, ModROM, GoodROM  ' Original ROM, Modified ROM and Recommended GoodNES settings
DIM SHARED AS BOOL ChangedROM, ParamHelp, ParamGui, ParamFix, ParamList, ParamAudit
DIM SHARED MapperList(0 TO 255) AS STRING
DIM SHARED DirDelimiter AS STRING*1
DIM SHARED Warnings(1 TO 20) AS STRING

REM  **** ** * LOCAL VARIABLES * ** ****
DIM AS STRING RomFile, LastKey
DIM AS BYTE Choice, x, ParamCount, ParamFile

REM  **** ** * HERE WE GO! * ** ****

DirDelimiter = "/"
ParamGui = FALSE

#IFDEF __FB_DOS__
    DirDelimiter = "\"
    ParamGui = FALSE
#ENDIF
#IFDEF __FB_WIN32__
    DirDelimiter = "\"
    ParamGui = TRUE
#ENDIF
#IFDEF __FB_LINUX__
    DirDelimiter = "/"
    ParamGui = TRUE
#ENDIF

ParamCount = 1
ParamHelp = FALSE
ParamFile = -1
ParamFix = FALSE
ParamAudit = FALSE
ParamList = FALSE
DO
  IF COMMAND(ParamCount) = "" THEN ParamCount -= 1 : EXIT DO
  IF UCASE(COMMAND(ParamCount)) = "/?" OR UCASE(COMMAND(ParamCount)) = "--HELP" OR UCASE(COMMAND(ParamCount)) = "-H" THEN ParamHelp = TRUE
  IF UCASE(COMMAND(ParamCount)) = "-GUI" THEN ParamGui = TRUE
  IF UCASE(COMMAND(ParamCount)) = "-TUI" THEN ParamGui = FALSE
  IF UCASE(COMMAND(ParamCount)) = "-LIST" THEN ParamList = TRUE
  IF UCASE(COMMAND(ParamCount)) = "-AUDIT" THEN ParamAudit = TRUE
  IF UCASE(COMMAND(ParamCount)) = "-FIX" THEN ParamFix = TRUE
  IF MID(COMMAND(ParamCount), 1, 1) <> "-" THEN ParamFile = ParamCount
  ParamCount += 1
LOOP

IF ParamFile < 0 AND (ParamList = TRUE OR ParamFix = TRUE OR ParamAudit = TRUE) THEN RomFile = CURDIR

IF ParamFile > 0 THEN
    RomFile = COMMAND(ParamFile)
  ELSEIF RomFile = "" THEN
    RomFile = SelectFile
    IF RomFile = CHR(27) THEN CLS: END(9)
END IF

IF ParamList = TRUE OR ParamFix = TRUE OR ParamAudit = TRUE THEN ParseRoms(RomFile)

IF ParamGui = TRUE THEN
    SCREEN 17,,2   ' Allocating two video pages
    SETMOUSE ,,0
  ELSE
    WIDTH 80, 25
END IF

IF ParamHelp = TRUE THEN About
Choice = 1
ChangedROM = FALSE

IF FileExists(EXEPATH + DirDelimiter + "mines.dat") = 0 THEN PRINT "The MINES.DAT file is missing!": END(4)

SELECT CASE CheckNESFile(RomFile)
  CASE 0
    PRINT "The file "; CHR(34); RomFile; CHR(34); " is not a valid iNES ROM."
    IF ParamGui = TRUE THEN SLEEP
    END(3)
  CASE -1
    PRINT "The file "; CHR(34); RomFile; CHR(34); " can't be found."
    IF ParamGui = TRUE THEN SLEEP
    END(2)
END SELECT

LoadMapperList
PRINT "Browsing the database... ";
GetRomInformations(RomFile)
PRINT "Done!";

WindowTitle("MiNES v" + pVer + " Copyright (C) Mateusz Viste " + pDate)
LOCATE ,,0
DrawScreen

OrgROM = ModROM  ' Backup copy

COLOR 3, 0
IF LEN(RomFile) <= 65 THEN
    LOCATE 3, 36 - LEN(RomFile)\2 : PRINT "ROM File: "; RomFile;
  ELSE
    LOCATE 3, 2: PRINT "ROM File: (...)"; RIGHT(RomFile, 63);
END IF
COLOR 11, 0
LOCATE 4, 33 - LEN(GoodROM.Title)\2: PRINT "GoodNES title: "; GoodROM.Title;

KeybFlush

DO
  IF Choice = 1 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.ROMbanks <> OrgROM.ROMbanks THEN COLOR 14
  LOCATE 7, 44: PRINT " 16kB ROM banks....: " & LEFT(STR(ModROM.ROMbanks) + SPACE(11), 11);
  IF Choice = 2 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.VROMbanks <> OrgROM.VROMbanks THEN COLOR 14
  LOCATE 8, 44: PRINT " 8kB VROM banks....: " & LEFT(STR(ModROM.VROMbanks) + SPACE(11), 11);
  IF Choice = 3 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.RAMbanks <> OrgROM.RAMbanks THEN COLOR 14
  LOCATE 9, 44: PRINT " 8kB RAM banks.....: ";
  IF ModROM.RAMbanks > 0 THEN PRINT LEFT(STR(ModROM.RAMbanks) + SPACE(11), 11); ELSE PRINT "Undefined  ";
  IF Choice = 4 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.Mirroring <> OrgROM.Mirroring THEN COLOR 14
  LOCATE 10, 44: PRINT " Mirroring.........: " & HorVer(ModROM.Mirroring);
  IF Choice = 5 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.BattRAM <> OrgROM.BattRAM THEN COLOR 14
  LOCATE 11, 44: PRINT " Battery-backed RAM: " & NoYes(ModROM.BattRAM);
  IF Choice = 6 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.Trainer <> OrgROM.Trainer THEN COLOR 14
  LOCATE 12, 44: PRINT " Trainer presence..: " & NoYes(ModROM.Trainer);
  IF Choice = 7 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.FourScreen <> OrgROM.FourScreen THEN COLOR 14
  LOCATE 13, 44: PRINT " Four-Screen layout: " & NoYes(ModROM.FourScreen);
  IF Choice = 8 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.VS <> OrgROM.VS THEN COLOR 14
  LOCATE 14, 44: PRINT " NES type..........: " & ClassVS(ModROM.VS);
  IF Choice = 9 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.Mapper <> OrgROM.Mapper THEN COLOR 14
  LOCATE 15, 44: PRINT " Memory mapper type: " & LEFT(STR(ModROM.Mapper) + SPACE(11), 11);
  IF Choice = 10 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.PAL <> OrgROM.PAL THEN COLOR 14
  LOCATE 16, 44: PRINT " TV encoding system: " & NtscPal(ModROM.PAL);
  IF Choice = 11 THEN COLOR 7, 1 ELSE COLOR 7, 0
  IF ModROM.Title <> OrgROM.Title OR ModROM.TitleLen <> OrgROM.TitleLen THEN COLOR 14
  LOCATE 17, 44: PRINT " ROM title.........: " & DontknowNoYes(SGN(ModROM.TitleLen));

  CheckRomWarnings
  HeaderPreview
  DisplayStatusLine
 REM ** Checking for any comments... ***
  SELECT CASE Choice
    CASE 1
      IF ModROM.ROMbanks = 0 THEN DisplayComment("This setting is wrong! A NES ROM needs at least 1 ROM bank to work.")
      IF ModROM.FileSize - 16 < ModROM.ROMbanks * 16384 THEN DisplayComment("This setting is wrong! The file is too short to contain so many ROM banks!")
    CASE 2
      IF ModROM.FileSize - 16 < ModROM.VROMbanks * 8192 THEN DisplayComment("This setting is wrong! The file is too short to contain so many VROM banks!")
    CASE 6
      IF ModROM.TrainerDetected = 1 AND ModROM.Trainer = FALSE THEN DisplayComment("This setting may be wrong. The file's size is suggesting a trainer there.")
      IF ModROM.TrainerDetected = 0 AND ModROM.Trainer = TRUE THEN DisplayComment("This setting may be wrong. The file's size don't suggest any trainer there.")
    CASE 9
      DisplayComment("Chipset: " + MapperList(ModROM.Mapper))
    CASE 11
      IF ModROM.TitleLen > 0 THEN DisplayComment(ModROM.Title)
      IF ModROM.TitleLen < 0 THEN DisplayComment("The file's size is awkward. Therefore, MiNES can't locate the 'title' area.")
  END SELECT
 REM ** Checking for comments done... ***
  DO: SLEEP: LastKey = INKEY: LOOP UNTIL LastKey <> ""
  KeybFlush
  IF LastKey = CHR(255) + "H" AND Choice > 1 THEN Choice -= 1
  IF LastKey = CHR(255) + "P" AND Choice < 11 THEN Choice += 1
  IF LastKey = CHR(255) + "I" THEN Choice = 1
  IF LastKey = CHR(255) + "Q" THEN Choice = 11
  IF LastKey = CHR(255) + "<" AND Warnings(1) <> "" THEN LastKey = AskWarning(Warnings(1), Warnings(2), Warnings(3), Warnings(4), Warnings(5), Warnings(6), Warnings(7), Warnings(8), Warnings(9), Warnings(10)) : LastKey = "" : KeybFlush 'F2
  IF LastKey = CHR(255) + CHR(59) THEN DisplayHelp   ' F1
  IF UCASE(LastKey) = "U" THEN ModROM = OrgROM
  IF UCASE(LastKey) = "R" THEN
    ModROM.ROMbanks = GoodROM.ROMbanks
    ModROM.VROMbanks = GoodROM.VROMbanks
    ModROM.Mirroring = GoodROM.Mirroring
    ModROM.BattRAM = GoodROM.BattRAM
    ModROM.Trainer = GoodROM.Trainer
    ModROM.FourScreen = GoodROM.FourScreen
    ModROM.Mapper = GoodROM.Mapper
  END IF
  IF UCASE(LastKey) = "C" THEN                     ' "C" pressed
    IF AskWarning("This will zeroes all unused header's bits. Be aware, that", "the iNES format could have evolved since the MiNES creation,", "and such 'unused' bits may contain some informations now.", "", "Do you want to proceed? [Y/N]") = "Y" THEN EncodeHeader(1)   ' CleanFlag ON
  END IF                                           '
  IF Lastkey = CHR(13) OR LastKey = "+" OR LastKey = "-" THEN
    SELECT CASE Choice
      CASE 1
        IF LastKey = "-" THEN
            IF ModROM.ROMbanks > 0 THEN ModROM.ROMbanks -= 1 ELSE ModROM.ROMbanks = 255
          ELSE
            IF ModROM.ROMbanks < 255 THEN ModROM.ROMbanks += 1 ELSE ModROM.ROMbanks = 0
        END IF
      CASE 2
        IF LastKey = "-" THEN
            IF ModROM.VROMbanks > 0 THEN ModROM.VROMbanks -= 1 ELSE ModROM.VROMbanks = 255
          ELSE
            IF ModROM.VROMbanks < 255 THEN ModROM.VROMbanks += 1 ELSE ModROM.VROMbanks = 0
        END IF
      CASE 3
        IF LastKey = "-" THEN
            IF ModROM.RAMbanks > 0 THEN ModROM.RAMbanks -= 1 ELSE ModROM.RAMbanks = 255
          ELSE
            IF ModROM.RAMbanks < 255 THEN ModROM.RAMbanks += 1 ELSE ModROM.RAMbanks = 0
        END IF
      CASE 4
        ModROM.Mirroring = 1 - ModROM.Mirroring
      CASE 5
        ModROM.BattRAM = 1 - ModROM.BattRAM
      CASE 6
        ModROM.Trainer = 1 - ModROM.Trainer
      CASE 7
        ModROM.FourScreen = 1 - ModROM.FourScreen
      CASE 8
        IF LastKey = "-" THEN
            IF ModROM.VS > 0 THEN ModROM.VS -= 1 ELSE ModROM.VS = 2
          ELSE
            IF ModROM.VS < 2 THEN ModROM.VS += 1 ELSE ModROM.VS = 0
        END IF
      CASE 9
        IF LastKey = "-" THEN
            IF ModROM.Mapper > 0 THEN ModROM.Mapper -= 1 ELSE ModROM.Mapper = 255
          ELSE
            IF ModROM.Mapper < 255 THEN ModROM.Mapper += 1 ELSE ModROM.Mapper = 0
        END IF
      CASE 10
        ModROM.PAL = 1 - ModROM.PAL
      CASE 11
        SELECT CASE ModROM.TitleLen
          CASE 0
            IF ModROM.GoodNES = TRUE THEN ModROM.Title = AskForTitle(GoodROM.Title) ELSE ModROM.Title = AskForTitle()
            IF ModROM.Title <> CHR(27) THEN
                ModROM.TitleLen = 128
                ModROM.FileSize += 128
              ELSE
                ModROM.Title = ""
            END IF
          CASE IS > 0
            ModROM.Title = ""
            ModROM.FileSize -= ModROM.TitleLen
            ModROM.TitleLen = 0
        END SELECT
    END SELECT
  END IF
  EncodeHeader
  DecodeHeader
LOOP UNTIL LastKey = CHR(27) OR LastKey = CHR(255) + "k"

FOR x = 1 TO 16
  IF ModROM.Header(x) <> OrgROM.Header(x) THEN ChangedROM = TRUE
NEXT x
IF ModROM.TitleLen <> OrgROM.TitleLen THEN ChangedROM = TRUE
IF ModROM.Title <> OrgROM.Title THEN ChangedROM = TRUE

IF ChangedROM = TRUE THEN
  IF AskWarning(" The file has been modified! ", "Do you want to save it? [Y/N]") = "Y" THEN
      IF SaveHeader(RomFile) = FALSE OR SaveNESTitle(RomFile) = FALSE THEN LastKey = AskWarning("An error occured when saving the file!", "", "Press any key..."): ChangedROM = FALSE
    ELSE
      ChangedROM = FALSE
  END IF
END IF

COLOR 7, 0
CLS
IF ChangedROM = TRUE THEN PRINT "File saved." ELSE PRINT "No changes made to the file."
PRINT
END


REM **** * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * ****
REM *** **  END OF THE MAIN PROGRAM  *  BELOW ARE FUNCTIONS AND SUBS   ** ***
REM **** * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * ****


SUB About()
PRINT
PRINT "MiNES v"; pVer; " - Mateusz's iNES editor, Copyright (C) Mateusz Viste "; pDate
PRINT
PRINT "MiNES is a handy tool which allows you to edit header's properties of an iNES"
PRINT "ROM file. MiNES acts accordingly to the Marat Fayzullin's specification, and"
PRINT "take care to not change any unused (reserved for future use) bits. That way you"
PRINT "can be sure that it will not broke your ROMs (unless you do something stupid),"
PRINT "and preserve compatibility with future versions of the iNES format."
PRINT
PRINT " Usage:  mines [rom.nes | path] [-h] [-gui | -tui] [-list] [-audit]"
PRINT
PRINT "Switches:"
PRINT " -h     - Displays this help screen"
PRINT " -gui   - Switches to graphic mode (default in the Windows and Linux versions)"
PRINT " -tui   - Switches to text mode (default in the DOS version)"
PRINT " -list  - Lists all iNES ROMs (*.nes) in the given directory"
PRINT " -audit - Checks iNES ROMs in the given directory and displays a detailed log"
'PRINT " -fix   - Massively (try to) fix all broken headers of ROMs at the given path"
PRINT
PRINT "Examples:    mines asterix.nes"
PRINT "             mines smb3.nes -gui"
PRINT "             mines c:\myroms\ -list -audit"
PRINT "             mines -list > my_roms.txt"
PRINT
IF ParamGui = TRUE THEN SLEEP
END(1)
END SUB


FUNCTION SaveHeader(NESfile AS STRING) AS BOOL
  DIM AS INTEGER FileHandler
  DIM AS BOOL SaveResult
  FileHandler = FREEFILE
  OPEN NESfile FOR BINARY AS #FileHandler
  IF PUT(#FileHandler, 1, ModRom.Header()) <> 0 THEN SaveResult = FALSE ELSE SaveResult = TRUE
  CLOSE #FileHandler
  RETURN SaveResult
END FUNCTION


SUB ReadHeader(NESfile AS STRING)
  DIM AS INTEGER FileHandler
  FileHandler = FREEFILE
  OPEN NESfile FOR BINARY AS #FileHandler
  GET #FileHandler, 1, ModROM.Header()
  CLOSE #FileHandler
END SUB


SUB EncodeHeader(CleanFlag AS BYTE = 0)
  DIM ByteBuffer AS UBYTE
 REM *** Byte #5 ***
  ModROM.Header(5) = ModROM.ROMbanks      ' Number of ROM banks
 REM *** Byte #6 ***
  ModROM.Header(6) = ModROM.VROMbanks     ' Number of VROM banks
 REM *** Byte #7 ***
  ByteBuffer = 0
  IF ModROM.Mirroring = TRUE THEN ByteBuffer = BITSET(ByteBuffer, 0) ELSE ByteBuffer = BITRESET(ByteBuffer, 0)
  IF ModROM.BattRAM = TRUE THEN ByteBuffer = BITSET(ByteBuffer, 1) ELSE ByteBuffer = BITRESET(ByteBuffer, 1)
  IF ModROM.Trainer = TRUE THEN ByteBuffer = BITSET(ByteBuffer, 2) ELSE ByteBuffer = BITRESET(ByteBuffer, 2)
  IF ModROM.FourScreen = TRUE THEN ByteBuffer = BITSET(ByteBuffer, 3) ELSE ByteBuffer = BITRESET(ByteBuffer, 3)
  IF BIT(ModROM.Mapper, 0) <> 0 THEN ByteBuffer = BITSET(ByteBuffer, 4) ELSE ByteBuffer = BITRESET(ByteBuffer, 4)
  IF BIT(ModROM.Mapper, 1) <> 0 THEN ByteBuffer = BITSET(ByteBuffer, 5) ELSE ByteBuffer = BITRESET(ByteBuffer, 5)
  IF BIT(ModROM.Mapper, 2) <> 0 THEN ByteBuffer = BITSET(ByteBuffer, 6) ELSE ByteBuffer = BITRESET(ByteBuffer, 6)
  IF BIT(ModROM.Mapper, 3) <> 0 THEN ByteBuffer = BITSET(ByteBuffer, 7) ELSE ByteBuffer = BITRESET(ByteBuffer, 7)
  ModROM.Header(7) = ByteBuffer
 REM *** Byte #8 ***
  IF CleanFlag = 0 THEN ByteBuffer = ModROM.Header(8) ELSE ByteBuffer = 0
  IF ModROM.VS = 1 THEN ByteBuffer = BITSET(ByteBuffer, 0) ELSE ByteBuffer = BITRESET(ByteBuffer, 0)
  IF ModROM.VS = 2 THEN ByteBuffer = BITSET(ByteBuffer, 1) ELSE ByteBuffer = BITRESET(ByteBuffer, 1)
  IF BIT(ModROM.Mapper, 4) <> 0 THEN ByteBuffer = BITSET(ByteBuffer, 4) ELSE ByteBuffer = BITRESET(ByteBuffer, 4)
  IF BIT(ModROM.Mapper, 5) <> 0 THEN ByteBuffer = BITSET(ByteBuffer, 5) ELSE ByteBuffer = BITRESET(ByteBuffer, 5)
  IF BIT(ModROM.Mapper, 6) <> 0 THEN ByteBuffer = BITSET(ByteBuffer, 6) ELSE ByteBuffer = BITRESET(ByteBuffer, 6)
  IF BIT(ModROM.Mapper, 7) <> 0 THEN ByteBuffer = BITSET(ByteBuffer, 7) ELSE ByteBuffer = BITRESET(ByteBuffer, 7)
  ModROM.Header(8) = ByteBuffer
 REM *** Byte #9 ***
  ModROM.Header(9) = ModROM.RAMbanks     ' Number of RAM banks
 REM *** Byte #10 ***
  IF CleanFlag = 0 THEN ByteBuffer = ModROM.Header(10) ELSE ByteBuffer = 0
  IF ModROM.PAL = TRUE THEN ByteBuffer = BITSET(ByteBuffer, 0) ELSE ByteBuffer = BITRESET(ByteBuffer, 0)
  ModROM.Header(10) = ByteBuffer
 REM *** Bytes #11 - #16 ***
  IF CleanFlag <> 0 THEN
    ModROM.Header(11) = 0
    ModROM.Header(12) = 0
    ModROM.Header(13) = 0
    ModROM.Header(14) = 0
    ModROM.Header(15) = 0
    ModROM.Header(16) = 0
  END IF
END SUB


SUB DecodeHeader()
  DIM ByteBuffor AS STRING*1
  ModROM.ROMbanks = ModROM.Header(5)          ' Number of ROM banks
  ModROM.VROMbanks = ModROM.Header(6)         ' Number of VROM banks
  IF BIT(ModROM.Header(7), 0) = 0 THEN ModROM.Mirroring = FALSE ELSE ModROM.Mirroring = TRUE
  IF BIT(ModROM.Header(7), 1) = 0 THEN ModROM.BattRAM = FALSE ELSE ModROM.BattRAM = TRUE
  IF BIT(ModROM.Header(7), 2) = 0 THEN ModROM.Trainer = FALSE ELSE ModROM.Trainer = TRUE
  IF BIT(ModROM.Header(7), 3) = 0 THEN ModROM.FourScreen = FALSE ELSE ModROM.FourScreen = TRUE
  REM *** Calculating the mapper ***
  ModROM.Mapper = 0
  IF BIT(ModROM.Header(7), 4) <> 0 THEN ModROM.Mapper = BITSET(ModROM.Mapper, 0)
  IF BIT(ModROM.Header(7), 5) <> 0 THEN ModROM.Mapper = BITSET(ModROM.Mapper, 1)
  IF BIT(ModROM.Header(7), 6) <> 0 THEN ModROM.Mapper = BITSET(ModROM.Mapper, 2)
  IF BIT(ModROM.Header(7), 7) <> 0 THEN ModROM.Mapper = BITSET(ModROM.Mapper, 3)
  IF BIT(ModROM.Header(8), 4) <> 0 THEN ModROM.Mapper = BITSET(ModROM.Mapper, 4)
  IF BIT(ModROM.Header(8), 5) <> 0 THEN ModROM.Mapper = BITSET(ModROM.Mapper, 5)
  IF BIT(ModROM.Header(8), 6) <> 0 THEN ModROM.Mapper = BITSET(ModROM.Mapper, 6)
  IF BIT(ModROM.Header(8), 7) <> 0 THEN ModROM.Mapper = BITSET(ModROM.Mapper, 7)
  REM *** Mapper calculated ***
  IF BIT(ModROM.Header(8), 0) = 0 THEN ModROM.VS = 0 ELSE ModROM.VS = 1  'VS
  IF BIT(ModROM.Header(8), 1) <> 0 THEN ModROM.VS = 2    ' PC10

  ModROM.RAMbanks = ModROM.Header(9)
  IF BIT(ModROM.Header(10), 0) = 0 THEN ModROM.PAL = FALSE ELSE ModROM.PAL = TRUE
END SUB


FUNCTION CheckNESFile(NESfile AS STRING) AS BYTE
  REM The CheckNESFile function opens the specified file and checks if its
  REM header is iNES format compliant. If the file is a valid iNES ROM, it
  REM returns 1. If not, 0. If the file doesn't exist, it returns -1.
  REM --------------------------------------------------------------------
  DIM SigNES AS STRING * 4
  DIM AS INTEGER FileHandler
  DIM AS BYTE Wynik
  Wynik = 1
  FileHandler = FREEFILE
  IF FileExists(NESfile) = 0 THEN
      Wynik = -1
    ELSE
      OPEN NESfile FOR BINARY AS #FileHandler
      IF LOF(FileHandler) <= 16 THEN Wynik = 0 ELSE GET #FileHandler, 1, SigNES
      CLOSE #FileHandler
      IF SigNES <> "NES" + CHR(26) THEN Wynik = 0
  END IF
  RETURN Wynik
END FUNCTION


SUB KeybFlush
  DO: LOOP UNTIL INKEY = ""
END SUB


SUB DrawScreen
  COLOR 7,0
  CLS
  COLOR 15, 2
  PRINT SPACE(18); "MiNES v"; pVer; " Copyright (C) Mateusz Viste "; pDate; SPACE(18);
  COLOR 8, 0
  LOCATE 6, 5: PRINT "[ Recommended settings ]ͻ";
  LOCATE 7, 5: PRINT " 16kB ROM banks....:            ";
  LOCATE 8, 5: PRINT " 8kB VROM banks....:            ";
  LOCATE 9, 5: PRINT " 8kB RAM banks.....:            ";
  LOCATE 10, 5: PRINT " Mirroring.........:            ";
  LOCATE 11, 5: PRINT " Battery-backed RAM:            ";
  LOCATE 12, 5: PRINT " Trainer presence..:            ";
  LOCATE 13, 5: PRINT " Four-screen layout:            ";
  LOCATE 14, 5: PRINT " NES type..........:            ";
  LOCATE 15, 5: PRINT " Memory mapper type:            ";
  LOCATE 16, 5: PRINT " TV encoding system:            ";
  LOCATE 17, 5: PRINT " ROM title.........:            ";
  LOCATE 18, 5: PRINT "ͼ";

  IF ModROM.GoodNES = TRUE THEN
    LOCATE 7, 27: PRINT LEFT(STR(GoodROM.ROMbanks) + SPACE(11), 11);
    LOCATE 8, 27: PRINT LEFT(STR(GoodROM.VROMbanks) + SPACE(11), 11);
    LOCATE 9, 27: PRINT "-          ";
    LOCATE 10, 27: PRINT HorVer(GoodROM.Mirroring);
    LOCATE 11, 27: PRINT NoYes(GoodROM.BattRAM);
    LOCATE 12, 27: PRINT NoYes(GoodROM.Trainer);
    LOCATE 13, 27: PRINT NoYes(GoodROM.FourScreen);
    LOCATE 14, 27: PRINT "-          ";
    LOCATE 15, 27: PRINT LEFT(STR(GoodROM.Mapper) + SPACE(11), 11);
    LOCATE 16, 27: PRINT "-          ";
    LOCATE 17, 27: PRINT "-          ";
   ELSE
    LOCATE 7, 27: PRINT "-          ";
    LOCATE 8, 27: PRINT "-          ";
    LOCATE 9, 27: PRINT "-          ";
    LOCATE 10, 27: PRINT "-          ";
    LOCATE 11, 27: PRINT "-          ";
    LOCATE 12, 27: PRINT "-          ";
    LOCATE 13, 27: PRINT "-          ";
    LOCATE 14, 27: PRINT "-          ";
    LOCATE 15, 27: PRINT "-          ";
    LOCATE 16, 27: PRINT "-          ";
    LOCATE 17, 27: PRINT "-          ";
  END IF

  COLOR 7, 0
  LOCATE 19, 5: PRINT "[ File's informations ]ͻ";
  LOCATE 20, 5: PRINT " CRC32:                         ";
  LOCATE 21, 5: PRINT " Size:                          ";
  LOCATE 22, 5: PRINT "                                ";
  LOCATE 23, 5: PRINT "ͼ";

  LOCATE 6, 43: PRINT "[ Current settings ]ͻ";
  LOCATE 7, 43: PRINT "                                ";
  LOCATE 8, 43: PRINT "                                ";
  LOCATE 9, 43: PRINT "                                ";
  LOCATE 10, 43: PRINT "                                ";
  LOCATE 11, 43: PRINT "                                ";
  LOCATE 12, 43: PRINT "                                ";
  LOCATE 13, 43: PRINT "                                ";
  LOCATE 14, 43: PRINT "                                ";
  LOCATE 15, 43: PRINT "                                ";
  LOCATE 16, 43: PRINT "                                ";
  LOCATE 17, 43: PRINT "                                ";
  LOCATE 18, 43: PRINT "ͼ";

  LOCATE 19, 43: PRINT "[ Header preview ]ͻ";
  LOCATE 20, 43: PRINT " ASCII:                         ";
  LOCATE 21, 43: PRINT " HEX..:                         ";
  LOCATE 22, 43: PRINT "                                ";
  LOCATE 23, 43: PRINT "ͼ";
END SUB


SUB HeaderPreview
  DIM AS BYTE x
  DIM AS INTEGER AuditCount
  COLOR 7, 0
  LOCATE 20, 14: PRINT HEX(ModROM.CRC32, 8);
  LOCATE 21, 13: PRINT ModROM.FileSize; " bytes";
 REM * ** *** Checking warnings *** ** *
  AuditCount = 0
  DO
    IF Warnings(AuditCount + 1) = "" OR AuditCount = 19 THEN EXIT DO
    AuditCount += 1
  LOOP
  LOCATE 22, 7: PRINT SPACE(30);
  LOCATE 22, 7
  IF AuditCount = 0 THEN
      COLOR 2, 0: PRINT " No warnings to display.";
    ELSEIF AuditCount = 1 THEN
      COLOR 4, 0: PRINT AuditCount; " warning [F2 to check]";
    ELSE
      COLOR 4, 0: PRINT AuditCount; " warnings [F2 to check]";
  END IF
 REM * ** *** Warnings checked *** ** *
  COLOR 7, 0
  FOR x = 1 TO 16
    LOCATE 20, 51 + x
    IF ModROM.Header(x) >= 32 AND ModROM.Header(x) < 128 THEN PRINT CHR(ModROM.Header(x)); ELSE PRINT ".";
    IF x < 9 THEN LOCATE 21, 49 + x * 3 ELSE LOCATE 22, 49 + (x - 8) * 3
    PRINT HEX(ModROM.Header(x), 2);
  NEXT x
END SUB


SUB DisplayComment(Komentarz AS STRING)
  COLOR 0, 6
  IF LEN(Komentarz) < 6 THEN Komentarz = Komentarz + SPACE(6 - LEN(Komentarz))
  IF LEN(Komentarz) > 76 THEN Komentarz = LEFT(Komentarz, 74) + ".."
  LOCATE 23, 39 - LEN(Komentarz)\2: PRINT ""; STRING(LEN(Komentarz) + 2, ""); "";
  LOCATE 23, 37: PRINT "[ Note ]";
  LOCATE 24, 39 - LEN(Komentarz)\2: PRINT " "; Komentarz; " ";
  LOCATE 25, 39 - LEN(Komentarz)\2: PRINT ""; STRING(LEN(Komentarz) + 2, ""); "";
END SUB


SUB DisplayStatusLine
  COLOR 7, 0
  LOCATE 23, 1: PRINT "    ͼ    ͼ    ";
  LOCATE 24, 1: PRINT SPACE(80);
  COLOR 0, 3
  LOCATE 25, 1: PRINT " Up/Down=Select ; +/-=Change ; Esc=Quit ; U=Undo ; R=Recom. ; F1=Help ; C=Clean ";
END SUB


SUB GetNESTitle(NESfile AS STRING)
  DIM AS INTEGER FileHandler, x
  DIM ByteBuff AS UBYTE
  DIM RestSize AS SHORT
  ModROM.TitleLen = 0
  ModROM.Title = ""
  FileHandler = FREEFILE
  OPEN NESfile FOR BINARY AS #FileHandler
  ModROM.FileSize = LOF(FileHandler)
  RestSize = LOF(FileHandler) - (INT(LOF(FileHandler)/8192) * 8192) - 16
  IF RestSize = 127 OR RestSize = 128 OR RestSize = 639 OR RestSize = 640 THEN
      IF RestSize > 512 THEN ModROM.TitleLen = RestSize - 512 ELSE ModROM.TitleLen = RestSize
      FOR x = ModROM.TitleLen - 1 TO 0 STEP -1
        GET #FileHandler, LOF(FileHandler) - x, ByteBuff
        IF ByteBuff >= 32 THEN ModROM.Title += CHR(ByteBuff)
      NEXT x
  END IF
  CLOSE #FileHandler
  SELECT CASE RestSize
    CASE 512, 639, 640
      ModROM.TrainerDetected = 1
    CASE 0, 127, 128
      ModROM.TrainerDetected = 0
    CASE ELSE
      ModROM.TrainerDetected = -1
      ModROM.TitleLen = -1
  END SELECT
END SUB


FUNCTION SaveNESTitle(NESfile AS STRING) AS BOOL
  DIM FileHandler AS INTEGER
  DIM BinTitle(1 TO 128) AS UBYTE  ' Ready to write title-buffer
  DIM AS UBYTE x
  DIM AS STRING TempString
  DIM AS BOOL SaveResult
  SaveResult = TRUE
  FOR x = 1 TO 128
    IF x <= LEN(ModROM.Title) THEN BinTitle(x) = ASC(MID(ModROM.Title, x, 1)) ELSE BinTitle(x) = 0
  NEXT x
  FileHandler = FREEFILE
  IF ModROM.TitleLen = 0 AND OrgROM.TitleLen > 0 THEN
    #IFNDEF __FB_WIN32__  ' Truncate doesn't work in Windows
      IF Truncate(NESfile, OrgROM.FileSize - OrgROM.TitleLen) <> 0 THEN SaveResult = FALSE
    #ELSE
      OPEN NESfile FOR BINARY AS #FileHandler
      DIM FileBuffer(1 TO LOF(FileHandler) - OrgROM.TitleLen) AS UBYTE
      GET #FileHandler, 1, FileBuffer()
      CLOSE #FileHandler
      OPEN NESfile FOR OUTPUT AS #FileHandler
      PUT #FileHandler, 1, FileBuffer()
      CLOSE #FileHandler
    #ENDIF
  END IF
  IF ModROM.TitleLen > 0 AND OrgROM.TitleLen > 0 AND ModROM.Title <> OrgROM.Title THEN
    IF OrgROM.TitleLen = 127 THEN
      OPEN NESfile FOR APPEND AS #FileHandler
      PRINT #FileHandler, " ";
      CLOSE #FileHandler
    END IF
    OPEN NESfile FOR BINARY AS #FileHandler
    IF PUT(#FileHandler, LOF(FileHandler)-127, BinTitle()) <> 0 THEN SaveResult = FALSE
    CLOSE #FileHandler
  END IF
  IF ModROM.TitleLen > 0 AND OrgROM.TitleLen = 0 THEN
    OPEN NESfile FOR APPEND AS #FileHandler
    PRINT #FileHandler, SPACE(ModROM.TitleLen);
    CLOSE #FileHandler
    OPEN NESfile FOR BINARY AS FileHandler
    IF PUT(#FileHandler, LOF(FileHandler) - ModROM.TitleLen + 1, BinTitle()) <> 0 THEN SaveResult = FALSE
    CLOSE #FileHandler
  END IF
  RETURN SaveResult
END FUNCTION


FUNCTION AskForTitle(Proposition AS STRING = "") AS STRING
  DIM AS STRING Wynik, LastKey
  DIM FillChar AS STRING*1
  IF ParamGui = TRUE THEN FillChar = "_" ELSE FillChar = " "
  PCOPY 0, 1
  LOCATE ,,1
  Wynik = Proposition
  COLOR 15, 1
  LOCATE 11, 2: PRINT "[Type in the title and press <Enter>]Ŀ";
  LOCATE 12, 2: PRINT "                                                                            ";
  LOCATE 13, 2: PRINT "";
  DO
    LOCATE 12, 4: PRINT Wynik + STRING(74 - LEN(Wynik), FillChar);
    LOCATE 12, 4 + LEN(Wynik)
    KeybFlush
    DO: SLEEP: LastKey = INKEY: LOOP UNTIL LastKey <> ""
    SELECT CASE LastKey
      CASE CHR(8)
        IF LEN(Wynik) > 0 THEN
           Wynik = MID(Wynik, 1, LEN(Wynik) - 1)
        END IF
      CASE CHR(13)
        EXIT DO
      CASE CHR(27)
        Wynik = CHR(27)
        EXIT DO
      CASE ELSE
        IF LEN(Wynik) < 74 AND ASC(LastKey) >= 28 AND LEN(LastKey) = 1 AND ASC(LastKey) >= 32 AND ASC(LastKey) < 127 THEN
            Wynik += LastKey
        END IF
    END SELECT
  LOOP
  LOCATE ,,0
  PCOPY 1, 0
  RETURN Wynik
END FUNCTION


FUNCTION AskWarning(Ostrz1 AS STRING, Ostrz2 AS STRING = "", Ostrz3 AS STRING = "", Ostrz4 AS STRING = "", Ostrz5 AS STRING = "", Ostrz6 AS STRING = "", Ostrz7 AS STRING = "", Ostrz8 AS STRING = "", Ostrz9 AS STRING = "", Ostrz10 AS STRING = "") AS STRING
  DIM AS STRING LastKey
  DIM AS BYTE LiczLine, Szerokosc, x, StartX, StartY
  DIM Ostrz(1 TO 10) AS STRING = {Ostrz1, Ostrz2, Ostrz3, Ostrz4, Ostrz5, Ostrz6, Ostrz7, Ostrz8, Ostrz9, Ostrz10}
  PCOPY 0, 1
  Szerokosc = 0
  LiczLine = 0
  FOR x = 1 TO 10
    IF Ostrz(x) <> "" THEN LiczLine = x
    IF LEN(Ostrz(x)) > Szerokosc THEN Szerokosc = LEN(Ostrz(x))
  NEXT x
  IF Szerokosc < 10 THEN Szerokosc = 10
  FOR x = 1 TO LiczLine
    IF LEN(Ostrz(x)) < Szerokosc THEN Ostrz(x) += SPACE(Szerokosc - LEN(Ostrz(x)))
  NEXT x

  StartX = 39 - (Szerokosc \ 2)
  StartY = 11 - (LiczLine \ 2)

  COLOR 15, 4
  LOCATE StartY, StartX: PRINT ""; STRING(Szerokosc, ""); "Ŀ";
  LOCATE StartY, 35: PRINT "[ Warning! ]";
  FOR x = 1 TO LiczLine
    LOCATE StartY + x, StartX: PRINT " "; Ostrz(x);" ";
  NEXT x
  LOCATE StartY + LiczLine + 1, StartX: PRINT ""; STRING(Szerokosc, ""); "";

  DO: SLEEP: LastKey = UCASE(INKEY): LOOP UNTIL LastKey <> ""
  PCOPY 1, 0
  KeybFlush
  RETURN LastKey
END FUNCTION


SUB LoadMapperList
  DIM x AS SHORT
  FOR x = 0 TO 255
    MapperList(x) = "Unknown mapper"
  NEXT x
  MapperList(0) = "NROM (No mapper)"
  MapperList(1) = "Nintendo MMC1 Chipset"
  MapperList(2) = "UNROM (ROM Switch)"
  MapperList(3) = "CNROM (VROM Switch)"
  MapperList(4) = "Nintendo MMC3"
  MapperList(5) = "Nintendo MMC5"
  MapperList(6) = "Front Far East (FFE) F4XXX Games"
  MapperList(7) = "AOROM (32kb ROM Switch)"
  MapperList(8) = "Front Far East (FFE) F3XXX Games"
  MapperList(9) = "Nintendo MMC2"
  MapperList(10) = "Nintendo MMC4"
  MapperList(11) = "Color Dreams"
  MapperList(12) = "Front Far East (FFE) F6XXX Games"
  MapperList(13) = "CPROM switch"
  MapperList(15) = "100-in-1 Cart Switch"
  MapperList(16) = "Bandai"
  MapperList(17) = "Front Far East (FFE) F8XXX Games"
  MapperList(18) = "Jaleco SS8806"
  MapperList(19) = "Namcot 106"
  MapperList(20) = "Famicom Disk System"
  MapperList(21) = "Konami VRC4A/C"
  MapperList(22) = "Konami VRC2A"
  MapperList(23) = "Konami VRC2B/VRC4E"
  MapperList(24) = "Konami VRC6A"
  MapperList(25) = "Konami VRC4B/D"
  MapperList(26) = "Konami VRC6B"
  MapperList(32) = "Irem G-101"
  MapperList(33) = "Taito TC0190/TC0350"
  MapperList(34) = "Nina 1 (PRG) Switch"
  MapperList(40) = "FDS-Port"
  MapperList(41) = "Caltron 6-in-1"
  MapperList(42) = "FDS-Port"
  MapperList(43) = "X-in-1"
  MapperList(44) = "7-in-1 MMC3 Port A001h"
  MapperList(45) = "X-in-1 MMC3 Port 6000hx4"
  MapperList(46) = "15-in-1 Color Dreams"
  MapperList(47) = "2-in-1 MMC3 Port 6000h"
  MapperList(48) = "Taito TC1090/RC0350"
  MapperList(49) = "4-in-1 MMC3 Port 6xxxh"
  MapperList(50) = "FDS-Port"
  MapperList(51) = "11-in-1"
  MapperList(52) = "7-in-1 MMC3 Port 6800h with SRAM"
  MapperList(56) = "Pirate SMB3"
  MapperList(57) = "6-in-1"
  MapperList(58) = "X-in-1"
  MapperList(61) = "20-in-1"
  MapperList(62) = "X-in-1"
  MapperList(64) = "Tengen Rambo-1"
  MapperList(65) = "Irem H3001 Chipset"
  MapperList(66) = "74161/32 (GNROM switch)"
  MapperList(67) = "Sunsoft Mapper 3"
  MapperList(68) = "Sunsoft Mapper 4"
  MapperList(69) = "Sunsoft Mapper 5 (FME-7 chip)"
  MapperList(70) = "74161/32 Chipset"
  MapperList(71) = "Camerica Mapper"
  MapperList(72) = "Jaleco Early Mapper #0"
  MapperList(73) = "Konami VRC3"
  MapperList(74) = "Taiwan MMC3 -Varient"
  MapperList(75) = "Jaleco SS805/Konami VRC1"
  MapperList(76) = "Namco 109 Chipset"
  MapperList(77) = "Irem Early Mapper #0"
  MapperList(78) = "Irem 74HC161/32 Chipset"
  MapperList(79) = "AVE Nina-3 board"
  MapperList(80) = "Taito X-005 Chipset"
  MapperList(81) = "AVE Nina-6 board"
  MapperList(82) = "Taito X1-17"
  MapperList(83) = "Cony Mapper"
  MapperList(84) = "PasoFami Mapper!"
  MapperList(85) = "Konami VRC7A/B"
  MapperList(86) = "Jaleco Early Mapper #2"
  MapperList(87) = "74161/32"
  MapperList(88) = "Namco 118"
  MapperList(89) = "SunSoft Early Mapper"
  MapperList(90) = "HK-TK2 (Pirate Cart Switch)"
  MapperList(91) = "HK-SF3 (Pirate Cart Switch)"
  MapperList(92) = "Jaleco Early Mapper #1"
  MapperList(93) = "74161/32"
  MapperList(94) = "74161/32"
  MapperList(95) = "Namcot MMC3-Style"
  MapperList(96) = "74161/32"
  MapperList(97) = "Irem 74161/32"
  MapperList(99) = "VS Unisystem Port 4016h"
  MapperList(100) = "MMC3/Trainer/Buggy Mode (used for hacked ROMs!)"
  MapperList(112) = "Asder"
  MapperList(113) = "Sachen/Hacker/Nina"
  MapperList(114) = "Super Games"
  MapperList(115) = "MMC3 Cart Saint"
  MapperList(116) = "PC-Reserved"
  MapperList(117) = "Future"
  MapperList(118) = "MMC3 with different Names Tables"
  MapperList(119) = "MMC3 TQROM with VROM+VRAM Pattern Tables"
  MapperList(122) = "74161/32"
  MapperList(133) = "Sachen"
  MapperList(151) = "VS Unisystem"
  MapperList(160) = "Pirate MMC5-Style (same as mapper #90)"
  MapperList(161) = "Nintendo MMC1 (same as mapper #1)"
  MapperList(180) = "Nihon Bussan"
  MapperList(182) = "Super Games (same as mapper #114)"
  MapperList(184) = "Sunsoft"
  MapperList(185) = "VROM-disable"
  MapperList(188) = "UNROM-reversed"
  MapperList(189) = "MMC3 Variant"
  MapperList(222) = "Dragon Ninja (Pirate)"
  MapperList(225) = "58/64-in-1 Cart Switch"
  MapperList(226) = "72-in-1 Cart Switch"
  MapperList(227) = "1200-in-1 Cart Switch"
  MapperList(228) = "Action 54 Cart Switch"
  MapperList(229) = "31-in-1"
  MapperList(230) = "X-in-1 plus Contra"
  MapperList(231) = "20-in-1"
  MapperList(232) = "4-in-1 Quattro Camerica"
  MapperList(233) = "X-in-1 plus Reset"
  MapperList(234) = "Maxi-15"
  MapperList(240) = "C&E/Supertone"
  MapperList(241) = "X-in-1 Education"
  MapperList(242) = "Waixing"
  MapperList(243) = "Sachen Poker"
  MapperList(244) = "C&E"
  MapperList(246) = "C&E"
  MapperList(255) = "X-in-1 (same as mapper #225)"
END SUB


FUNCTION CRC32(BYVAL lpBuffer AS ZSTRING PTR, BYVAL BufferSize AS INTEGER) AS UINTEGER
Static Table(255) As Uinteger => { _
&H00000000, &H77073096, &HEE0E612C, &H990951BA, _
&H076DC419, &H706AF48F, &HE963A535, &H9E6495A3, _
&H0EDB8832, &H79DCB8A4, &HE0D5E91E, &H97D2D988, _
&H09B64C2B, &H7EB17CBD, &HE7B82D07, &H90BF1D91, _
&H1DB71064, &H6AB020F2, &HF3B97148, &H84BE41DE, _
&H1ADAD47D, &H6DDDE4EB, &HF4D4B551, &H83D385C7, _
&H136C9856, &H646BA8C0, &HFD62F97A, &H8A65C9EC, _
&H14015C4F, &H63066CD9, &HFA0F3D63, &H8D080DF5, _
&H3B6E20C8, &H4C69105E, &HD56041E4, &HA2677172, _
&H3C03E4D1, &H4B04D447, &HD20D85FD, &HA50AB56B, _
&H35B5A8FA, &H42B2986C, &HDBBBC9D6, &HACBCF940, _
&H32D86CE3, &H45DF5C75, &HDCD60DCF, &HABD13D59, _
&H26D930AC, &H51DE003A, &HC8D75180, &HBFD06116, _
&H21B4F4B5, &H56B3C423, &HCFBA9599, &HB8BDA50F, _
&H2802B89E, &H5F058808, &HC60CD9B2, &HB10BE924, _
&H2F6F7C87, &H58684C11, &HC1611DAB, &HB6662D3D, _
&H76DC4190, &H01DB7106, &H98D220BC, &HEFD5102A, _
&H71B18589, &H06B6B51F, &H9FBFE4A5, &HE8B8D433, _
&H7807C9A2, &H0F00F934, &H9609A88E, &HE10E9818, _
&H7F6A0DBB, &H086D3D2D, &H91646C97, &HE6635C01, _
&H6B6B51F4, &H1C6C6162, &H856530D8, &HF262004E, _
&H6C0695ED, &H1B01A57B, &H8208F4C1, &HF50FC457, _
&H65B0D9C6, &H12B7E950, &H8BBEB8EA, &HFCB9887C, _
&H62DD1DDF, &H15DA2D49, &H8CD37CF3, &HFBD44C65, _
&H4DB26158, &H3AB551CE, &HA3BC0074, &HD4BB30E2, _
&H4ADFA541, &H3DD895D7, &HA4D1C46D, &HD3D6F4FB, _
&H4369E96A, &H346ED9FC, &HAD678846, &HDA60B8D0, _
&H44042D73, &H33031DE5, &HAA0A4C5F, &HDD0D7CC9, _
&H5005713C, &H270241AA, &HBE0B1010, &HC90C2086, _
&H5768B525, &H206F85B3, &HB966D409, &HCE61E49F, _
&H5EDEF90E, &H29D9C998, &HB0D09822, &HC7D7A8B4, _
&H59B33D17, &H2EB40D81, &HB7BD5C3B, &HC0BA6CAD, _
&HEDB88320, &H9ABFB3B6, &H03B6E20C, &H74B1D29A, _
&HEAD54739, &H9DD277AF, &H04DB2615, &H73DC1683, _
&HE3630B12, &H94643B84, &H0D6D6A3E, &H7A6A5AA8, _
&HE40ECF0B, &H9309FF9D, &H0A00AE27, &H7D079EB1, _
&HF00F9344, &H8708A3D2, &H1E01F268, &H6906C2FE, _
&HF762575D, &H806567CB, &H196C3671, &H6E6B06E7, _
&HFED41B76, &H89D32BE0, &H10DA7A5A, &H67DD4ACC, _
&HF9B9DF6F, &H8EBEEFF9, &H17B7BE43, &H60B08ED5, _
&HD6D6A3E8, &HA1D1937E, &H38D8C2C4, &H4FDFF252, _
&HD1BB67F1, &HA6BC5767, &H3FB506DD, &H48B2364B, _
&HD80D2BDA, &HAF0A1B4C, &H36034AF6, &H41047A60, _
&HDF60EFC3, &HA867DF55, &H316E8EEF, &H4669BE79, _
&HCB61B38C, &HBC66831A, &H256FD2A0, &H5268E236, _
&HCC0C7795, &HBB0B4703, &H220216B9, &H5505262F, _
&HC5BA3BBE, &HB2BD0B28, &H2BB45A92, &H5CB36A04, _
&HC2D7FFA7, &HB5D0CF31, &H2CD99E8B, &H5BDEAE1D, _
&H9B64C2B0, &HEC63F226, &H756AA39C, &H026D930A, _
&H9C0906A9, &HEB0E363F, &H72076785, &H05005713, _
&H95BF4A82, &HE2B87A14, &H7BB12BAE, &H0CB61B38, _
&H92D28E9B, &HE5D5BE0D, &H7CDCEFB7, &H0BDBDF21, _
&H86D3D2D4, &HF1D4E242, &H68DDB3F8, &H1FDA836E, _
&H81BE16CD, &HF6B9265B, &H6FB077E1, &H18B74777, _
&H88085AE6, &HFF0F6A70, &H66063BCA, &H11010B5C, _
&H8F659EFF, &HF862AE69, &H616BFFD3, &H166CCF45, _
&HA00AE278, &HD70DD2EE, &H4E048354, &H3903B3C2, _
&HA7672661, &HD06016F7, &H4969474D, &H3E6E77DB, _
&HAED16A4A, &HD9D65ADC, &H40DF0B66, &H37D83BF0, _
&HA9BCAE53, &HDEBB9EC5, &H47B2CF7F, &H30B5FFE9, _
&HBDBDF21C, &HCABAC28A, &H53B39330, &H24B4A3A6, _
&HBAD03605, &HCDD70693, &H54DE5729, &H23D967BF, _
&HB3667A2E, &HC4614AB8, &H5D681B02, &H2A6F2B94, _
&HB40BBE37, &HC30C8EA1, &H5A05DF1B, &H2D02EF8D }

  DIM AS UINTEGER PTR t=@Table(0)
  IF BufferSize < 1 THEN RETURN 0
  asm
    mov edi,[lpBuffer]
    mov esi,[t]
    mov ecx,[BufferSize]
    mov eax,&HFFFFFFFF
    mov edx,&HFF
    push ebp
    mov ebp,edx
    Xor edx,edx
    loop_it:
      mov dl,[edi]
      mov ebx,eax
      Xor eax,edx
      And eax,ebp
      Shr ebx,8
      mov eax,[esi+eax*4]
      inc edi
      Xor eax,ebx
    Loop loop_it
    pop ebp
    Xor eax,&HFFFFFFFF
    mov [Function],eax
  END asm
END FUNCTION


SUB GetNesCRC(NESfile AS STRING)
  DIM AS INTEGER FileHandler, BuffSize
  DIM FileBuff AS STRING
  FileHandler = FREEFILE
  OPEN NESfile FOR BINARY AS #FileHandler
  IF ModROM.TitleLen > 0 THEN BuffSize = LOF(FileHandler) - 16 - ModROM.TitleLen ELSE BuffSize = LOF(FileHandler) - 16
  FileBuff = SPACE(BuffSize)
  GET #FileHandler, 17, FileBuff
  ModROM.CRC32 = CRC32(FileBuff, BuffSize)
  CLOSE #FileHandler
END SUB


FUNCTION GetGoodInfos(RomCRC AS UINTEGER) AS BYTE
  REM GetGoodInfos returns 1 if GoodInfos found, 0 otherwise.
  REM -------------------------------------------------------
  DIM AS INTEGER FileHandler, Adresacja
  DIM IndexBuff AS STRING * 17
  DIM LineBuff AS STRING
  DIM AS BYTE Wynik
  DIM AS UBYTE RekordLen
  Wynik = 0
  FileHandler = FREEFILE
  OPEN EXEPATH + DirDelimiter + "mines.dat" FOR BINARY AS #FileHandler
  DO
    GET #FileHandler, , IndexBuff
    IF UCASE(MID(IndexBuff, 1, 8)) = HEX(RomCRC, 8) THEN
      Adresacja = VAL(MID(IndexBuff, 9, 8))
      RekordLen = ASC(MID(IndexBuff, 17, 1))
      Wynik = 1
      LineBuff = SPACE(RekordLen)
      GET #FileHandler, Adresacja, LineBuff
      LineBuff = XorString(LineBuff)
      GoodROM.Title = TRIM(MID(LineBuff, 43))
      IF MID(LineBuff, 9, 1) = "V" THEN GoodROM.Mirroring = TRUE ELSE GoodROM.Mirroring = FALSE
      IF MID(LineBuff, 10, 1) = "B" THEN GoodROM.BattRAM = TRUE ELSE GoodROM.BattRAM = FALSE
      IF MID(LineBuff, 11, 1) = "T" THEN GoodROM.Trainer = TRUE ELSE GoodROM.Trainer = FALSE
      IF MID(LineBuff, 12, 1) = "4" THEN GoodROM.FourScreen = TRUE ELSE GoodROM.FourScreen = FALSE
      GoodROM.Mapper = VAL(MID(LineBuff, 13, 3))
      GoodROM.ROMbanks = VAL(MID(LineBuff, 35, 3))
      GoodROM.VROMbanks = VAL(MID(LineBuff, 39, 3))
    END IF
  LOOP UNTIL EOF(FileHandler) OR Wynik = 1 OR IndexBuff = "#EndFile        "
  CLOSE #FileHandler
  RETURN Wynik
END FUNCTION


SUB DisplayHelp
  PCOPY 0, 1
  COLOR 0, 7
  LOCATE 6, 16: PRINT "Ŀ";
  LOCATE 7, 16: PRINT "                      HELP                      ";
  LOCATE 8, 16: PRINT "                     ======                     ";
  LOCATE 9, 16: PRINT "                                                ";
  LOCATE 10, 16: PRINT " Up & Down - Select an item.                    ";
  LOCATE 11, 16: PRINT " +/- - Change value.                            ";
  LOCATE 12, 16: PRINT " C - Clean header. This option will zeroes all  ";
  LOCATE 13, 16: PRINT "     header's reserved bits.                    ";
  LOCATE 14, 16: PRINT " U - Undo any changes (reload the file).        ";
  LOCATE 15, 16: PRINT " R - Set recommended (GoodNES compliant) header ";
  LOCATE 16, 16: PRINT "     settings.                                  ";
  LOCATE 17, 16: PRINT " F1 - Displays this help screen.                ";
  LOCATE 18, 16: PRINT " F2 - Displays ROM's warnings (if any).         ";
  LOCATE 19, 16: PRINT " Esc - Exit to the operating system.            ";
  LOCATE 20, 16: PRINT "                                                ";
  LOCATE 21, 16: PRINT "[Press any key]";
  SLEEP
  KeybFlush
  PCOPY 1, 0
END SUB


FUNCTION XorString(StringToXor AS STRING) AS STRING
  DIM x AS INTEGER
  DIM ByteBuff AS UBYTE
  DIM Wynik AS STRING
  Wynik = ""
  FOR x = 1 TO LEN(StringToXor)
    ByteBuff = ASC(MID(StringToXor, x, 1))
    Wynik += CHR(ByteBuff XOR 255)
  NEXT x
  RETURN Wynik
END FUNCTION


SUB GetRomInformations(RomFile AS STRING)
  ReadHeader(RomFile)
  DecodeHeader
  GetNESTitle(RomFile)
  GetNesCRC(RomFile)
  IF GetGoodInfos(ModROM.CRC32) = 1 THEN ModROM.GoodNES = TRUE ELSE ModROM.GoodNES = FALSE: GoodROM.Title = "Unknown"
END SUB


SUB ParseRoms(BYVAL StartPath AS STRING)
  REDIM p(1 TO 1) AS STRING
  DIM AS INTEGER i = 1, n = 1, attrib, AuditCount = 0, FixCount, ConHandler
  DIM AS STRING d, RomFile

  ConHandler = FREEFILE
  OPEN CONS FOR OUTPUT AS #ConHandler
  p(1) = StartPath    '' Starting directory

  IF RIGHT(p(1), 1) <> DirDelimiter THEN
    d = dir(p(1), fbNormal Or fbDirectory, @attrib)
    IF (attrib AND fbDirectory) THEN
      p(1) += DirDelimiter
    END IF
  END IF

  WHILE i <= n
    d = DIR(p(i) + "*.nes" , fbNormal OR fbDirectory, @attrib)

    WHILE d > ""
      IF (attrib AND fbDirectory) THEN
          IF d <> "." AND d <> ".." THEN
            n += 1
            REDIM preserve p(1 To n)
            p(n) = p(i) + d + DirDelimiter
          END IF
        ELSE
          REM * ** ***  Here begins the file's analysis  *** ** *
          RomFile = p(i) & d
          GetRomInformations(RomFile)
          IF ParamList = TRUE THEN
            PRINT #ConHandler, GoodROM.Title; " <"; RomFile; ">"
          END IF
          IF ParamAudit = TRUE THEN
            DIM AS UBYTE Counter = 0
            CheckRomWarnings
            DO
              IF Warnings(Counter + 1) = "" OR Counter = 19 THEN EXIT DO
              AuditCount += 1
              Counter += 1
              PRINT #ConHandler, RomFile; ": "; Warnings(Counter)
            LOOP
          END IF
          IF ParamFix = TRUE THEN
            ' Here will come all "FIX" rules.
          END IF
          REM * ** ***  Analysis done  *** ** *
      END IF
      d = DIR(@attrib)
    WEND
    i += 1
  WEND
  IF ParamAudit = TRUE THEN PRINT #ConHandler, AuditCount; " warning(s)."
  IF ParamFix = TRUE THEN PRINT #ConHandler, FixCount; " ROMs fixed."
  CLOSE #ConHandler
  END(0)
END SUB


FUNCTION SelectFile() AS STRING
  DIM TempVar AS STRING
  PRINT "What iNES file would you like to edit? >";
  LINE INPUT, TempVar
  RETURN TempVar
END FUNCTION


FUNCTION HeaderGarbage(Header() AS UBYTE) AS BOOL
  DIM AS BOOL Wynik = FALSE
  DIM AS STRING TempVar = ""
  DIM AS BYTE x
  FOR x = 8 TO 16
    IF Header(x) > 0 THEN TempVar += CHR(Header(x))
  NEXT x
  IF RIGHT(TempVar, 5) = "Dude!" THEN Wynik = TRUE
  IF RIGHT(TempVar, 3) = "MJR" THEN Wynik = TRUE
  IF RIGHT(TempVar, 5) = "NI1.3" THEN Wynik = TRUE
  IF RIGHT(TempVar, 6) = "NI 1.3" THEN Wynik = TRUE
  IF RIGHT(TempVar, 5) = "NI2.1" THEN Wynik = TRUE
  IF RIGHT(TempVar, 6) = "Ni0330" THEN Wynik = TRUE
  IF RIGHT(TempVar, 5) = "aster" THEN Wynik = TRUE
  RETURN Wynik
END FUNCTION


SUB CheckRomWarnings
  DIM AS INTEGER CorrectSize, VromRomSize, WarnCount
  FOR WarnCount = 1 TO 20
    Warnings(WarnCount) = ""
  NEXT WarnCount
  WarnCount = 1
  VromRomSize = ModROM.FileSize - 16
  IF ModROM.TitleLen > 0 THEN VromRomSize -= ModROM.TitleLen
  IF ModROM.Trainer = TRUE THEN VromRomSize -= 512
  CorrectSize = ModROM.ROMbanks * 16384 + ModROM.VROMbanks * 8192 + 512 * ModROM.Trainer + 16
  IF ModROM.VS = 2 THEN CorrectSize += 8192
  IF ModROM.TitleLen > 0 THEN CorrectSize += ModROM.TitleLen
  REM * ** *** Checks against the GoodNES database *** ** *
  IF ModROM.GoodNES = TRUE THEN
    IF ModROM.ROMbanks <> GoodROM.ROMbanks THEN Warnings(WarnCount) = "The number of ROM banks may be incorrect." : WarnCount += 1
    IF ModROM.VROMbanks <> GoodROM.VROMbanks THEN Warnings(WarnCount) = "The number of VROM banks may be incorrect." : WarnCount += 1
    IF ModROM.Mirroring <> GoodROM.Mirroring THEN Warnings(WarnCount) = "The mirroring may be incorrect." : WarnCount += 1
    IF ModROM.BattRAM <> GoodROM.BattRAM THEN Warnings(WarnCount) = "The SRAM presence flag may be incorrect." : WarnCount += 1
    IF ModROM.Trainer <> GoodROM.Trainer THEN Warnings(WarnCount) = "The trainer presence flag may be incorrect." : WarnCount += 1
    IF ModROM.FourScreen <> GoodROM.FourScreen THEN Warnings(WarnCount) = "The four-screen flag may be incorrect." : WarnCount += 1
    IF ModROM.Mapper <> GoodROM.Mapper THEN Warnings(WarnCount) = "The mapper may be incorrect." : WarnCount += 1
  END IF
  REM * ** *** Various consistency checks *** ** *
  IF ModROM.Trainer <> ModROM.TrainerDetected THEN Warnings(WarnCount) = "The trainer presence flag is incorrect." : WarnCount += 1
  IF ModROM.FileSize <> CorrectSize THEN Warnings(WarnCount) = "The file's size isn't in adequacy with header's values." : WarnCount += 1
  IF VromRomSize MOD 8192 <> 0 THEN Warnings(WarnCount) = "The file's size is awkward. It may indicate a corrupted ROM." : WarnCount += 1
  IF HeaderGarbage(ModROM.Header()) = TRUE THEN Warnings(WarnCount) = "The ROM contains garbage in its header. Needs cleaning!" : WarnCount += 1
  IF ModROM.TitleLen = 127 THEN Warnings(WarnCount) = "The title tag is 127 bytes long. Its size should be 128 bytes." : WarnCount += 1
END SUB
