-- bmp.e - BMP files reading/writing - HIP 2.1
-- Copyright (C) 2001  Davi Tassinari de Figueiredo
--
-- This program is distributed under the terms of the GNU General
-- Public License. Please read the documentation for more information.
--
-- This file has routines for reading the headers of Windows and OS/2
-- bitmaps, loading them into the memory and writing the bitmaps to
-- disk.

constant include_name = "bmp.e"

include machine.e
include fileutil.e
include constant.e

function load_bmp(atom fn, atom bpp, atom width, atom height, atom compression)

    -- Reads a bitmap from disk and writes it in memory.

    atom address, line_size, bytes_read, byte
    sequence line

    -- log_info ("load_bmp: start", LOG_IMPORTANT)

    if compression then return ERROR_UNSUPPORTED end if

    --    if bpp < 8 then return ERROR_UNSUPPORTED end if
    -- only 8- and 24-bpp bitmaps are supported at this time
    if bpp != 8 and bpp != 24 then return ERROR_UNSUPPORTED end if

    line_size = width * (bpp / 8)
    address = allocate( height * line_size)

    -- log_info_s ("load_bmp: allocated in pos %d",address, LOG_DETAIL)

    if address = 0 then     -- Unable to allocate enough space for the image
	return ERROR_OUTOFMEMORY
    end if

    -- Move to beginning of picture

    bytes_read = 0

    line = repeat(0, line_size)

    -- log_info ("load_bmp: starting for", LOG_DETAIL)
    for current_line = height-1 to 0 by -1 do
	-- Read each line and store it
	for char = 1 to line_size do
	    line[char] = getc(fn)
	end for

	bytes_read = bytes_read + line_size

	while remainder(bytes_read, 4) do   -- line starts at 4-byte block
	    byte = getc(fn)
	    bytes_read = bytes_read + 1
	end while

	poke (address + current_line * line_size, line)

    end for

    -- log_info ("load_bmp: ending for", LOG_DETAIL)

    return address
end function



global function bmp_read (atom fn)
    -- Reads the bitmap header.

    atom bmp_size, data_offset, bmp_header_size,
	 width, height, bpp, compression, hres, vres, colors,
	 bytes_per_pal, address
    sequence pal, pal_entry, img_info
    object temp


    -- Move to the beginning of the file
    if seek(fn, 0) then
	return ERROR_FILEERROR
    end if

    -- Bitmap ID - 2 bytes
    temp = read_bytes(fn, 2)

    -- log_info ("get_img_info: analysing header", LOG_DETAIL)
    if find(temp, {"BM", "BA"})=0 then
	-- Not a known bitmap file header
	-- log_info ("get_img_info: invalid header, exiting with ERROR_INVALIDBMP", LOG_IMPORTANT)
	return ERROR_INVALIDBMP
    end if

    -- log_info ("get_img_info: reading size", LOG_DETAIL)

    -- File size - 4 bytes
    bmp_size = read_dword (fn, LITTLE_ENDIAN)
    if bmp_size = -1 then
	-- Unexpected end of file
	-- log_info_s ("get_img_info: eof - bmp_size = %d",bmp_size, LOG_IMPORTANT)
	return ERROR_CORRUPTED
    elsif file_size(fn) < bmp_size then
	-- Not a complete file
	-- log_info_s ("get_img_info: incomplete - bmp_size = %d",bmp_size, LOG_IMPORTANT)
	return ERROR_CORRUPTED
    end if

    -- log_info ("get_img_info: reading reserved", LOG_DETAIL)

    -- Reserved - 4 bytes
    temp = read_dword (fn, LITTLE_ENDIAN)
    if temp = -1 then
	-- Unexpected end of file
	return ERROR_CORRUPTED
    end if

    -- log_info ("get_img_info: reading data offset", LOG_DETAIL)

    -- Bitmap data offset - 4 bytes
    data_offset = read_dword (fn, LITTLE_ENDIAN)
    if data_offset = -1 then
	-- Unexpected end of file
	-- log_info_s ("get_img_info: bmp_size = %d",bmp_size, LOG_DETAIL)
	return ERROR_CORRUPTED
    elsif data_offset > bmp_size then
	-- BMP starts after end of file, error
	return ERROR_CORRUPTED
    end if

    -- log_info ("get_img_info: reading header size", LOG_DETAIL)

    -- Bitmap header size - 4 bytes
    bmp_header_size = read_dword(fn, LITTLE_ENDIAN)
    if bmp_header_size = -1 then
	-- Unexpected end of file
	return ERROR_CORRUPTED
    elsif bmp_header_size != 40 and bmp_header_size != 12 then
	-- Unsupported header size
	return ERROR_UNSUPPORTED
    end if

    -- The next fields depend on the header size
    if bmp_header_size = 40 then    -- Windows bitmap

	-- log_info ("get_img_info: Windows header", LOG_IMPORTANT)

	-- log_info ("get_img_info: reading size", LOG_DETAIL)

	-- Read width and height - 4 + 4 bytes
	width = read_dword(fn, LITTLE_ENDIAN)
	height = read_dword(fn, LITTLE_ENDIAN)

	if height = -1 then
	    -- Unexpected end of file
	    return ERROR_CORRUPTED
	elsif height = 0  or  width = 0  then
	    -- An empty bitmap, invalid
	    return ERROR_CORRUPTED
	end if


	-- log_info ("get_img_info: reading number of planes", LOG_DETAIL)

	-- Number of planes - 2 bytes - ignored
	temp = read_word(fn, LITTLE_ENDIAN)
	if temp = -1 then
	    -- Unexpected end of file
	    return ERROR_CORRUPTED
	end if

	-- log_info ("get_img_info: reading bpp", LOG_DETAIL)

	-- Bits per pixel - 2 bytes
	bpp = read_word(fn, LITTLE_ENDIAN)

	-- log_info_s ("get_img_info: bpp = %d", bpp, LOG_DETAIL)

	if bpp=-1 then
	    -- Unexpected end of file
	    -- log_info ("get_img_info: Unexpected EOF, exiting with ERROR_CORRUPTED", LOG_IMPORTANT)
	    return ERROR_CORRUPTED
	elsif find( bpp, {1, 2, 4, 8, 16, 24, 32}) = 0 then
	    -- Unknown BPP
	    -- log_info ("get_img_info: Unsupported BPP, exiting with ERROR_UNSUPPORTED", LOG_IMPORTANT)
	    return ERROR_UNSUPPORTED
	end if


	-- log_info ("get_img_info: reading compression", LOG_DETAIL)

	-- Compression - 4 bytes
	compression = read_dword(fn, LITTLE_ENDIAN)
	if compression = -1 then
	    -- Unexpected end of file
	    -- log_info ("get_img_info: Unexpected EOF, exiting with ERROR_CORRUPTED", LOG_IMPORTANT)
	    return ERROR_CORRUPTED
	end if


	-- log_info ("get_img_info: reading image size", LOG_DETAIL)

	-- Image size in bytes - 4 bytes - ignored
	temp = read_dword(fn, LITTLE_ENDIAN)
	if temp = -1 then
	    -- Unexpected end of file
	    -- log_info ("get_img_info: Unexpected EOF, exiting with ERROR_CORRUPTED", LOG_IMPORTANT)
	    return ERROR_CORRUPTED
	end if


	-- log_info ("get_img_info: reading resolution", LOG_DETAIL)

	-- Horizontal and vertical resolutions in pixels per metre - 4 + 4 bytes
	hres = read_dword(fn, LITTLE_ENDIAN)
	vres =  read_dword(fn, LITTLE_ENDIAN)
	if vres = -1 then
	    -- Unexpected end of file
	    return ERROR_CORRUPTED
	end if

	-- log_info ("get_img_info: reading number of colors", LOG_DETAIL)

	-- Number of colors used - 4 bytes
	colors = read_dword(fn, LITTLE_ENDIAN)
	if colors = -1 then
	    -- Unexpected end of file
	    -- log_info ("get_img_info: Unexpected EOF, exiting with ERROR_CORRUPTED", LOG_IMPORTANT)
	    return ERROR_CORRUPTED
	elsif colors = 0 then       -- All colors are important
	    colors = power(2, bpp)
	elsif colors > power(2, bpp) then   -- Impossible, error
	    return ERROR_CORRUPTED
	end if


	-- log_info ("get_img_info: reading important colors", LOG_DETAIL)

	-- Number of important colors - 4 bytes - ignored
	temp = read_dword(fn, LITTLE_ENDIAN)
	if temp = -1 then
	    -- Unexpected end of file
	    -- log_info ("get_img_info: Unexpected EOF, exiting with ERROR_CORRUPTED", LOG_IMPORTANT)
	    return ERROR_CORRUPTED
	end if

	-- Bytes per palette entry
	bytes_per_pal = 4


    elsif bmp_header_size = 12 then -- OS/2 bitmap

	-- log_info ("get_img_info: OS/2 header", LOG_DETAIL)

	-- Read width and height - 2 + 2 bytes
	width = read_word(fn, LITTLE_ENDIAN)
	height = read_word(fn, LITTLE_ENDIAN)


	if height = -1 then
	    -- Unexpected end of file
	    -- log_info ("get_img_info: Unexpected EOF, exiting with ERROR_CORRUPTED", LOG_IMPORTANT)
	    return ERROR_CORRUPTED
	elsif height = 0  or  width = 0  then
	    -- An empty bitmap? Weird...
	    return ERROR_CORRUPTED
	end if


	-- Number of planes - 2 bytes - ignored
	temp = read_word(fn, LITTLE_ENDIAN)
	if temp = -1 then
	    -- Unexpected end of file
	    -- log_info ("get_img_info: Unexpected EOF, exiting with ERROR_CORRUPTED", LOG_IMPORTANT)
	    return ERROR_CORRUPTED
	end if


	-- Bits per pixel - 2 bytes
	bpp = read_word(fn, LITTLE_ENDIAN)

	if bpp=-1 then
	    -- Unexpected end of file
	    -- log_info ("get_img_info: Unexpected EOF, exiting with ERROR_CORRUPTED", LOG_IMPORTANT)
	    return ERROR_CORRUPTED
	elsif find( bpp, {8, 24}) = 0 then
	    -- Unknown BPP
	    return ERROR_UNSUPPORTED
	end if


	compression = 0
	hres = 0
	vres = 0
	colors = power(2, bpp)
	bytes_per_pal = 3
    end if

    -- Read palette


    if bpp <= 8 then
	-- log_info ("get_img_info: read palette", LOG_DETAIL)

	pal = repeat(-1, colors)
	pal_entry = {-1, -1, -1}

	for index = 1 to colors do
	    -- Read palette entry
	    pal_entry[PAL_BLUE] = getc(fn)
	    pal_entry[PAL_GREEN] = getc(fn)
	    pal_entry[PAL_RED] = getc(fn)

	    if pal_entry[PAL_RED] = -1 then -- End of file
		return ERROR_CORRUPTED
	    end if

	    if bytes_per_pal = 4 then       -- Windows BMP, skip one byte
		temp = getc(fn)
		if temp=-1 then return ERROR_CORRUPTED end if   -- EOF
	    end if

	    pal [index] = pal_entry
	end for

    else
	pal = {}
    end if


    -- log_info ("get_img_info: calling load_bmp", LOG_DETAIL)

    address = load_bmp (fn, bpp, width, height, compression)
    if address < 0 then
	return address      -- error
    end if

    -- log_info ("get_img_info: write info in img_info", LOG_DETAIL)

    -- Write information in img_info structure
    img_info = repeat(-1, IMG_INFO_SIZE)

    img_info [IMG_WIDTH] = width
    img_info [IMG_HEIGHT] = height
    img_info [IMG_BPP] = bpp
    img_info [IMG_HRES] = hres
    img_info [IMG_VRES] = vres
    img_info [IMG_PAL] = pal
    img_info [IMG_COMPRESSION] = compression
--    img_info [IMG_BACKGROUNDCOLOR]

--     if bmp_header_size = 40 then
--         img_info [IMG_TYPE] = BT_WIN
--     elsif bmp_header_size = 12 then
--         img_info [IMG_TYPE] = BT_OS2
--     end if

    img_info [IMG_INTERLACED] = 0
    img_info [IMG_ADDRESS] = address

--    img_info [IMG_FORMAT] = FORMAT_BMP

    -- log_info ("get_img_info: calling set_transparent_color", LOG_IMPORTANT)

    -- set transparent color to none
    img_info [IMG_TRANSPARENT] = -1
--    img_info = set_transparent_color (img_info, -1)

    -- log_info ("get_img_info: exiting", LOG_DETAIL)

    return img_info
end function


global function bmp_save (atom fn, sequence img_info)
    -- Saves a bitmap from memory to disk in Windows format.

    atom size, line_size, bytes_per_line, address
    sequence pal


    -- log_info_s ("save_bmp: start   fn = %d", fn, LOG_IMPORTANT)


    if find (img_info [IMG_BPP], {8,24}) = 0 then
    -- log_info_s ("save_bmp: unsupported bpp: %d", img_info [IMG_BPP], LOG_IMPORTANT)
	return ERROR_UNSUPPORTED
    end if


    address = img_info [IMG_ADDRESS]

    -- Get palette

    if img_info [IMG_BPP] <= 8 then
	pal = img_info [IMG_PAL]
    else
	pal = {}
    end if

    line_size = img_info [IMG_WIDTH] * img_info [IMG_BPP] / 8

    bytes_per_line = line_size

    if remainder (bytes_per_line, 4) then
	bytes_per_line = bytes_per_line + 4 - remainder(bytes_per_line, 4)
    end if

    -- Total file size (header + palette + image)
    size = 54 + length(pal) * 4 +
	   bytes_per_line * img_info [IMG_HEIGHT]


    -- Write header

    -- ID
    puts(fn, "BM")

    -- File size
    write_dword(fn, size, LITTLE_ENDIAN)

    -- Reserved
    write_dword(fn, 0, LITTLE_ENDIAN)

    -- Picture offset
    write_dword(fn, 54 + length(pal) * 4, LITTLE_ENDIAN)

    -- Header size
    write_dword(fn, 40, LITTLE_ENDIAN)

    -- Picture width
    write_dword(fn, img_info [IMG_WIDTH], LITTLE_ENDIAN)

    -- Picture height
    write_dword(fn, img_info [IMG_HEIGHT], LITTLE_ENDIAN)

    -- Number of planes
    write_word(fn, 1, LITTLE_ENDIAN)

    -- Bits per pixel
    write_word(fn, img_info [IMG_BPP], LITTLE_ENDIAN)

    -- Compression - uncompressed file = 0
    write_dword(fn, 0, LITTLE_ENDIAN)

    -- Bitmap data size
    write_dword(fn, bytes_per_line * img_info [IMG_HEIGHT], LITTLE_ENDIAN)

    -- Horizontal resolution
    write_dword(fn, img_info [IMG_HRES], LITTLE_ENDIAN)

    -- Vertical resolution
    write_dword(fn, img_info [IMG_VRES], LITTLE_ENDIAN)

    -- Number of colors
    if img_info [IMG_BPP] <= 8 and     -- Palette picture
      length(pal) != power(2, img_info [IMG_BPP]) then -- Not all colors used
	write_dword(fn, length(pal), LITTLE_ENDIAN)
    else
	write_dword(fn, 0, LITTLE_ENDIAN)
    end if

    -- Number of important colors; 0 = all are important
    write_dword(fn, 0, LITTLE_ENDIAN)

    -- Write the palette
    for index = 1 to length(pal)  do
	puts(fn, pal [index][PAL_BLUE])
	puts(fn, pal [index][PAL_GREEN])
	puts(fn, pal [index][PAL_RED])
	puts(fn, 0) -- Extra empty byte
    end for

    -- Write the bitmap
    for line = img_info [IMG_HEIGHT]-1 to 0 by -1 do
	-- Write one line at a time
	puts(fn, peek({address + line * line_size, line_size}))
	-- Write extra bytes at the end of the line
	puts(fn, repeat(0, bytes_per_line - line_size))
    end for


    return 0    -- no errors
end function

global constant FORMAT_BMP = register_img_format ("BMP","bmp","bmp",{"BM","BA"},{8,24})

