;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; `eidl.asm' -- EIDL main source code.
;
; This file is part of EIDL.
;
; Author: Robert Riebisch <rr@bttr-software.de>.
;
; Copyright (c) 2020 BTTR Software.
;
; This program is free software; you can redistribute it and/or modify it under
; the terms of the MIT LICENSE. -- See `license.txt' for details.
;
; Build instructions for NASM version 2.14.02:
;   nasm.exe -o ..\..\bin\eidl.com eidl.asm
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

   ; --- Ought to be enough.
   CPU   8086

   ; --- .COM program.
   ORG   100h

   ; --- Let's start.
   jmp   main

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; === Our INT 28h handler.
newint28h:
   ; === The actual idle "intelligence".
   ; --- Make sure CPU will be notified.
   sti
   ; --- Halt CPU until an interrupt occurs.
   hlt
   ; --- Don't interrupt now.
   cli
   ; --- Go to previous handler in chain.
   DB 0EAh  ; = JMP ptr16:16.
oldint28h:
   ; Nothing here. See below.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; === Installation part.
main:
   ; === Save address of "DOS IDLE INTERRUPT" handler (INT 28h).
   ; --- Point ES to interrupt vector table (IVT) at address 0:xxxx.
   xor   ax, ax   ; (  2 bytes.)
   mov   es, ax   ; (+ 2 bytes.)
                  ; (= Doubleword for storing `oldint28h'.)

   ; --- Keep in RAM until here.
   TSR_END \
   EQU   100h + $ - $$ + 15
   ;      |     |   |    \--- Round up to next memory paragraph.
   ;      |     |   \--- Start of the current section.
   ;      |     \--- Current assembly position.
   ;      \--- Length of program segment prefix (PSP).

   ; --- Save offset part.
   mov   ax, [es:4 * 28h]
   mov   [oldint28h], ax
   ; --- Save segment part.
   mov   ax, [es:4 * 28h + 2]
   mov   [oldint28h + 2], ax

   ; === Patch IVT with the address of our interrupt handler.
   ; --- Better safe, than sorry.
   cli
   ; --- Store offset part.
   mov   word [es:4 * 28h], newint28h
   ; --- Store segment part.
   mov   ax, cs
   mov   [es:4 * 28h + 2], ax
   ; --- Don't worry anymore.
   sti

   ; === Output info to the user.
   mov   dx, instmsg
   ; --- Call DOS function 9 "WRITE STRING TO STANDARD OUTPUT"
   mov   ah, 9
   int   21h

   ; === Release EIDL's environment memory block.
   ; --- Put segment to free from PSP to ES.
   mov   ax, word [2Ch]
   mov   es, ax
   ; --- Call DOS function 49h "FREE MEMORY".
   mov   ah, 49h
   int   21h

   ; === Stay in RAM and return control to DOS.
   ; --- Number of memory paragraphs to keep.
   mov   dx, TSR_END / 16
   ; --- Call DOS function 31h "TERMINATE AND STAY RESIDENT".
   mov   ah, 31h
   int   21h

instmsg \
   DB    'EIDL installed.', 13, 10, '$'

; --- End of file.
