;
; DWOL v1.0 - a tiny Wake-On-LAN implementation for DOS
; Release date: 2017-03-18
;
; http://dwol.sourceforge.net
;
; To be assembled with NASM: nasm -f bin -o dwol.com dwol.asm
;
; DWOL is distributed under the terms of the MIT License, as listed below.
;
; Copyright (C) 2017 Mateusz Viste
;
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to
; deal in the Software without restriction, including without limitation the
; rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
; sell copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
;
; The above copyright notice and this permission notice shall be included in
; all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
; IN THE SOFTWARE.
;

CPU 8086           ; restrict instruction set to 8086
org 100h           ; COM file originates at 100h


; === CODE SEGMENT ===========================================================
section .text

; set ES to DS, expected by a few routines later
push ds
pop es

; fill in dst mac (FF FF FF FF FF FF) and magic packet 'sync stream' (also
; filling everything in-between with FFh, but this will be re-filled later)
mov al, 0FFh
mov di, pktbuf     ; destination -> ES:DI (ES == DS already)
mov cx, 22         ; fill 22 bytes to cover both dst mac and sync stream
cld                ; DF = 0 (direction: increase)
rep stosb          ; fill CX bytes at ES:DI with AL

; convert argument into all uppercase (in-place), and store the start of the
; argument in SI (trimming all space prefixes)
xor si, si
;xor cx, cx        ; no need to reset CX, it is already 0 (rep stosb above)
mov cl, [80h]      ; set CX to length of argument
; set DI to the *END* of the string (conversion will happen backward)
mov di, 81h
add di, cx
push di            ; push end-of-arg ptr to stack (will come handy below)
; if no argument was provided -> help screen
mov dx, MSGHELP
jcxz ERRQUIT
; add a string terminator to the MAC argument, so I can print it out later
mov [di], byte '$'
; convert the string
NEXTUPPERCASEMAC:
dec di
mov al, [di]
cmp al, 'a'
jb short NOTLOWER
cmp al, 'z'
ja short NOTLOWER
sub al, 20h        ; convert a-z to A-Z
mov [di], al       ; replace the character with its uppercase equivalent
NOTLOWER:
cmp al, ' '        ; if al != ' ' then si = di
je short SKIPSPACE
mov si, di
SKIPSPACE:
loop NEXTUPPERCASEMAC ; CX-- ; jnz NEXTUPPERCASEMAC

; save argptr for later
mov [argptr], si

; I expect a parameter of exactly 17 bytes (xx:xx:xx:xx:xx:xx)
pop di             ; load end-of-arg ptr from stack
sub di, si         ; end-of-arg - argptr (computes arg's length)
cmp di, byte 17
jne short ERRQUIT

; load the MAC address once into its destination in pktbuf
;mov si, [argptr]  ; SI still contains argptr, no need to reload it
mov di, pktbuf+14h
mov bx, 6          ; how many bytes I want to read
;mov dx, MSGHELP   ; no need to reload dx, it still contains MSGHELP's offset
LOADMACNEXTBYTE:
mov cl, 2          ; nibble counter (set CX=2, CH already 0 so I save 1 byte)
; first nibble
DONEXTNIBBLE:
mov ah, al         ; save AL into AH, just in case it contains the 1st nibble
lodsb              ; mov al, [si] + inc si
cmp al, '9'
jbe short NIBBLEISDIGIT
sub al, 'A'-'0'-10 ; AL was in range 'A'...'F'
cmp al, '9'        ; [validation] AL should be > '9' now
jbe short ERRQUIT  ; [validation] if not, then it's not a valid MAC
NIBBLEISDIGIT:
sub al, '0'        ; adjust AL
test al, 240       ; [validation] make sure AL is in range 0..15 now
jnz short ERRQUIT  ; [validation] if not, then it's not a valid MAC
loop DONEXTNIBBLE  ; CX-- ; jnz DONEXTNIBBLE
; both nibbles done, let's glue them together now
mov cl, 4          ; 8086 doesn't support SHL reg, 4 so I have to use CL
shl ah, cl         ; (still more efficient than 4 times SHL ah, 1)
or al, ah          ; combine AH and AL to form the result in AL

stosb              ; mov [di], al + inc di

dec bx
jz short ENDOFMAC

lodsb              ; load the mac-addr separator and si++
cmp al, ':'        ; [validation] separator is expected to be ':'
jne short ERRQUIT  ; [validation] othewise MAC format is not valid

jmp short LOADMACNEXTBYTE

ENDOFMAC:

; replicate the destination MAC addr 15x, as expected by WOL
mov di, pktbuf+1Ah ; destination -> ES:DI (ES == DS already)
;cld               ; DF = 0 (direction: increase) - already done before
mov cl, 6*15       ; set CX to 6*15 (CH already zero, so I save a byte)
mov si, pktbuf+14h
rep movsb

; scan interrupts 60h to 80h to find a packet driver
mov ax, 3560h      ; DOS call 'get vector' for int 60h
SCANNEXTINT:
int 21h            ; ES:BX contains the int vector address now
; see if the int vector address contains a packet driver
lea di, [bx+3]     ; the pkt drvr sig is at vector +3
mov si, pktdrvsig
mov cl, 8          ; set CX to 8 (CH is already 0, so I can save a byte)
repe cmpsb         ; compares CX bytes at ES:DI and DS:SI, ZF is 0 if matching
jz short PKTDRVFOUND
inc ax             ; test next int (effectively INC AL, but INT AX is smaller)
cmp al, 81h        ; unless we are already at int 81h, in such case abort
jb short SCANNEXTINT
; being here means no pktdrv could be found within 60h..80h
mov dx, MSGNOPKTDRV

ERRQUIT:           ; the seemingly unusual position of this branch in the
                   ; middle of other code is on purpose - makes it possible
                   ; to use (faster, smaller) near jumps
mov ah, 9          ; print out the error message
int 21h
mov ax, 4C01h      ; terminate process with return code 1
int 21h


PKTDRVFOUND:
; packet driver routine is at ES:BX - remember this for later
mov [pktptr], bx
mov [pktptr+2], es

; load es = ds, and keep it that way from now on
push ds
pop es

; fill in ethernet headers: src mac (ask the packet driver for this)
mov ah, 6          ; pkt drv 'get_address'
xor bx, bx         ; handle (optional since v1.10 of the Packet Driver Spec)
mov di, pktbuf+6   ; ES:DI will be filled with the MAC addr (ES == DS already)
mov cl, 6          ; addr lenght -> set CX to 6 (CH is already 0)
pushf
cli
call far [pktptr]
mov dx, MSGFAILGETMAC
jc short ERRQUIT

; fill in ethernet headers: ethertype (0842h, network byte order)
mov word [pktbuf+12], 4208h

; frame ready, send it now!
mov ah, 4          ; send_pkt()
mov si, pktbuf
mov cx, 116        ; frame lenght (dstmac + srcmac + ethertype + sig + 16xMAC)
                   ; I do not assume CH == 0 here, since regs might have been
                   ; destroyed during the pkt drv call just before
pushf
cli
call far [pktptr]
mov dx, MSGFAILSEND
jc short ERRQUIT

; all good: print success message and quit
mov ah, 9
mov dx, MSGSENTTO
int 21h            ; 'packet sent to...'
mov ah, 9          ; it would be tempting to assume AH stayed 9 and avoid
mov dx, [argptr]   ; reloading it, but DOS might have modified it
int 21h            ; print out the destination mac address

mov ax, 4C00h      ; terminate process with return code 0
int 21h


; === DATA SEGMENT ===========================================================
;section .data
; no need to specify a new section - tiny mode implies DS == CS anyway (and I
; can save a byte or two on segment alignment)

MSGHELP db 'DWOL v1.0 Copyright (C) 2017 Mateusz Viste', 13, 10, 'A tiny Wake-On-Lan implementation for DOS.', 13, 10, 10, 'Usage: DWOL macaddr', 13, 10, 10, 'Read DWOL.TXT for details.$'

MSGNOPKTDRV db 'ERROR: No packet driver found!$'

MSGFAILGETMAC db 'ERROR: Failed to detect local MAC address!$'

MSGFAILSEND db 'ERROR: Packet driver send() failure!$'

MSGSENTTO db 'WOL packet sent to $'

pktdrvsig db 'PKT DRVR'

; === BSS SEGMENT ============================================================
section .bss

pktbuf resb 128    ; buffer that will hold the magic packet
pktptr resd 1      ; far pointer to the packet driver routine
argptr resw 1      ; ptr to the command-line argument
