/*  
 * Copyright (c) 2008, Sun Microsystems, Inc.
 * 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * Neither the name of Sun Microsystems nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 *  Note:  In order to comply with the binary form redistribution 
 *         requirement in the above license, the licensee may include 
 *         a URL reference to a copy of the required copyright notice, 
 *         the list of conditions and the disclaimer in a human readable 
 *         file with the binary form of the code that is subject to the
 *         above license.  For example, such file could be put on a 
 *         Blu-ray disc containing the binary form of the code or could 
 *         be put in a JAR file that is broadcast via a digital television 
 *         broadcast medium.  In any event, you must include in any end 
 *         user licenses governing any code that includes the code subject 
 *         to the above license (in source and/or binary form) a disclaimer 
 *         that is at least as protective of Sun as the disclaimers in the 
 *         above license.
 * 
 *         A copy of the required copyright notice, the list of conditions and
 *         the disclaimer will be maintained at 
 *         https://hdcookbook.dev.java.net/misc/license.html .
 *         Thus, licensees may comply with the binary form redistribution
 *         requirement with a text file that contains the following text:
 * 
 *             A copy of the license(s) governing this code is located
 *             at https://hdcookbook.dev.java.net/misc/license.html
 */

package net.java.bd.tools.playlist;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.util.ArrayList;
import net.java.bd.tools.StringIOHelper;

/**
 * BD-ROM Part 3-1 5.3.4 PlayItem
 */
public class PlayItem {
    private ClipInfo[] angles;
    private boolean isMultiAngle;
    private int connectionCondition;
    private int inTime;
    private int outTime;
    private UOMaskTable uoMaskTable = new UOMaskTable();
    private boolean playItemRandomAccessFlag;
    private int stillMode;
    private int stillTime;
    private boolean isDifferentAudios;
    private boolean isSeamlessAngleChange;
    private STNTable stnTable = new STNTable();
    private int id;
    
    public PlayItem() {}
    public PlayItem(int id) {
        this.id = id;
    }
    
    public void readObject(DataInputStream din) throws IOException {
        int length = din.readUnsignedShort();
        byte[] data = new byte[length];
        din.readFully(data);
        readObject0(new DataInputStream(new ByteArrayInputStream(data)));
    }
    
    private void readObject0(DataInputStream din) throws IOException {
        // 16 bit length
        // 8*5 ClipInfo[0] clipinfo filename
        // 8*4 ClipInfo[0] clip codec id
        // 11 bit reserved
        // 1 bit isMultiAngle
        // 4 bit connectionCondition
        // 8 bit ClipInfo[0] STC id ref
        // 32 bit inTime
        // 32 bit outTime
        // UOMaskTable
        // 1 bit playItemRandomAccessFlag
        // 7 bit reserve
        // 8 bit stillMode
        // 16 bit stillTime
        // if isMultiAngle=true, 1-N Angles
        // STNTAble
        
        ArrayList angleList = new ArrayList();
        String clipName;
        String codecId;
        int stcId;
        byte b;
       
        //din.skipBytes(2); // length
        clipName = StringIOHelper.readISO646String(din, 5);
        codecId = StringIOHelper.readISO646String(din, 4);
        din.skipBytes(1);
        b = din.readByte();
        setIsMultiAngle((b & 0x10) != 0);
        setConnectionCondition(b & 0x0f);
        stcId = din.readByte();
        setInTime(din.readInt());
        setOutTime(din.readInt());
        uoMaskTable.readObject(din);
        b = din.readByte();
        setPlayItemRandomAccessFlag((b & 0x80) != 0);
        setStillMode(din.readByte());
        if (getStillTime() == 1) {
            setStillTime(din.readUnsignedShort());
        } else {
            din.skipBytes(2);
        }
        
        ClipInfo entry = new ClipInfo(0, clipName, codecId, stcId);
        angleList.add(entry);
        if (getIsMultiAngle()) { // more ClipInfo data here
            int entries = din.readByte();
            b = din.readByte();
            setIsDifferentAudios((b & 0x02) != 0);
            setIsSeamlessAngleChange((b & 0x01) != 0);
            for (int i = 1; i < entries; i++) {
               clipName = StringIOHelper.readISO646String(din, 5);
               codecId = StringIOHelper.readISO646String(din, 4);
               stcId = din.readByte();
               entry = new ClipInfo(i, clipName, codecId, stcId);
               angleList.add(entry);
            }
        } 
        ClipInfo[] entries = 
             (ClipInfo[]) angleList.toArray(new ClipInfo[angleList.size()]);
        setAngles(entries);
        stnTable.readObject(din);
    }
   
    public ClipInfo[] getAngles() {
        return angles;
    }

    public void setAngles(ClipInfo[] angles) {
        if ( (getIsMultiAngle() && (angles.length < 2 || angles.length > 9)) ||
             (!getIsMultiAngle() && angles.length != 1) ) {
            throw new IllegalArgumentException("Unexpected number of Clip Angle Information");
        }
        for (int i = 0; i < angles.length; i++) {
            if (i != angles[i].getId()) {
                throw new IllegalArgumentException("angle_id does not match the array order");
            }
        }
        this.angles = angles;
    }

    public boolean getIsMultiAngle() {
        return isMultiAngle;
    }

    public void setIsMultiAngle(boolean isMultiAngle) {
        this.isMultiAngle = isMultiAngle;
    }

    public int getConnectionCondition() {
        return connectionCondition;
    }

    public void setConnectionCondition(int connectionCondition) {
        
        if (connectionCondition != 1 &&
            connectionCondition != 5 &&
            connectionCondition != 6) {
            throw new IllegalArgumentException("Unexpected connectionCondition value "
                    + connectionCondition);
        }
                                
        this.connectionCondition = connectionCondition;
    }

    public int getInTime() {
        return inTime;
    }

    public void setInTime(int inTime) {
        this.inTime = inTime;
    }

    public int getOutTime() {
        return outTime;
    }

    public void setOutTime(int outTime) {
        this.outTime = outTime;
    }

    public UOMaskTable getUoMaskTable() {
        return uoMaskTable;
    }

    public void setUoMaskTable(UOMaskTable uoMaskTable) {
        this.uoMaskTable = uoMaskTable;
    }

    public boolean getPlayItemRandomAccessFlag() {
        return playItemRandomAccessFlag;
    }

    public void setPlayItemRandomAccessFlag(boolean playItemRandomAccessFlag) {
        this.playItemRandomAccessFlag = playItemRandomAccessFlag;
    }

    public int getStillMode() {
        return stillMode;
    }

    public void setStillMode(int stillMode) {
        this.stillMode = stillMode;
    }

    public int getStillTime() {
        return stillTime;
    }

    public void setStillTime(int stillTime) {
        this.stillTime = stillTime;
    }

    public STNTable getStnTable() {
        return stnTable;
    }

    public void setStnTable(STNTable stnTable) {
        this.stnTable = stnTable;
    }

    public boolean getIsDifferentAudios() {
        return isDifferentAudios;
    }

    public void setIsDifferentAudios(boolean isDifferentAudios) {
        this.isDifferentAudios = isDifferentAudios;
    }

    public boolean getIsSeamlessAngleChange() {
        return isSeamlessAngleChange;
    }

    public void setIsSeamlessAngleChange(boolean isSeamlessAngleChange) {
        this.isSeamlessAngleChange = isSeamlessAngleChange;
    }

    public int getId() {
        return id;
    }

    public void setId(int id) {
        this.id = id;
    }
    
}
