/*
 * Copyright (c) 1995, 2006, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package jail.java.util.zip;

import java.io.InputStream;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.io.IOException;
import jail.java.io.File;
import jail.java.util.jar.JarEntry;

import java.util.zip.ZipException;

import impl.java.io.BluesFileSystem;
import impl.java.util.jar.JarEntryPeer;
import impl.sun.misc.IOUtils;

import java.util.Enumeration;

/**
 * This class is used to read entries from a zip file.
 *
 * <p>
 * Unless otherwise noted, passing a <tt>null</tt> argument to a constructor or
 * method in this class will cause a {@link NullPointerException} to be thrown.
 *
 * @author David Connelly
 */
public class ZipFile implements ZipConstants {

    private java.util.zip.ZipFile peer = null;
    private boolean locsig; // if zip file starts with LOCSIG (usually true)
    private boolean closeRequested;

    /**
     * Mode flag to open a zip file for reading.
     */
    public static final int OPEN_READ = 0x1;

    /**
     * Mode flag to open a zip file and mark it for deletion. The file will be
     * deleted some time between the moment that it is opened and the moment
     * that it is closed, but its contents will remain accessible via the
     * <tt>ZipFile</tt> object until either the close method is invoked or the
     * virtual machine exits.
     */
    public static final int OPEN_DELETE = 0x4;

    /**
     * Opens a zip file for reading.
     *
     * <p>
     * First, if there is a security manager, its <code>checkRead</code> method
     * is called with the <code>name</code> argument as its argument to ensure
     * the read is allowed.
     *
     * @param name
     *            the name of the zip file
     * @throws ZipException
     *             if a ZIP format error has occurred
     * @throws IOException
     *             if an I/O error has occurred
     * @throws SecurityException
     *             if a security manager exists and its <code>checkRead</code>
     *             method doesn't allow read access to the file.
     * @see SecurityManager#checkRead(java.lang.String)
     */
    public ZipFile(String name) throws IOException {
        this(new File(name), OPEN_READ);
    }

    /**
     * Opens a new <code>ZipFile</code> to read from the specified
     * <code>File</code> object in the specified mode. The mode argument must be
     * either <tt>OPEN_READ</tt> or <tt>OPEN_READ | OPEN_DELETE</tt>.
     *
     * <p>
     * First, if there is a security manager, its <code>checkRead</code> method
     * is called with the <code>name</code> argument as its argument to ensure
     * the read is allowed.
     *
     * @param file
     *            the ZIP file to be opened for reading
     * @param mode
     *            the mode in which the file is to be opened
     * @throws ZipException
     *             if a ZIP format error has occurred
     * @throws IOException
     *             if an I/O error has occurred
     * @throws SecurityException
     *             if a security manager exists and its <code>checkRead</code>
     *             method doesn't allow read access to the file, or its
     *             <code>checkDelete</code> method doesn't allow deleting the
     *             file when the <tt>OPEN_DELETE</tt> flag is set.
     * @throws IllegalArgumentException
     *             if the <tt>mode</tt> argument is invalid
     * @see SecurityManager#checkRead(java.lang.String)
     * @since 1.3
     */
    public ZipFile(File file, int mode) throws IOException {
        this(file, mode, null);
    }

    protected ZipFile(final File file, final int mode, final Boolean jarverify) throws IOException {
        if (((mode & OPEN_READ) == 0) || ((mode & ~(OPEN_READ | OPEN_DELETE)) != 0)) {
            throw new IllegalArgumentException("Illegal mode: 0x" + Integer.toHexString(mode));
        }
        String name = file.getPath();
        SecurityManager sm = glue.java.lang.System.getSecurityManager();
        if (sm != null) {
            glue.java.lang.SecurityManager.checkRead(sm,name);
            if ((mode & OPEN_DELETE) != 0) {
                glue.java.lang.SecurityManager.checkDelete(sm,name);
            }
        }
        IOException x = (IOException)AccessController.doPrivileged(new PrivilegedAction() {
            @Override
            public Object run() {
                return initPeer(file,mode,jarverify);
            }
        });
        if (null!=x) throw x;
    }
    
    private IOException initPeer(final File file, final int mode, final Boolean jarverify) {
        try {
            String npath = BluesFileSystem.getLocalPath(file.getCanonicalPath());
            java.io.File nf = new java.io.File(npath);
            locsig = readLOCSIG(nf);
            if (jarverify != null) {
                peer = new java.util.jar.JarFile(nf, jarverify.booleanValue(), mode);
            } else {
                peer = new java.util.zip.ZipFile(nf, mode);
            }
            return null;
        } catch (IOException x) {
            peer = null;
            locsig = false;
            return x;
        }
    }

    private boolean readLOCSIG(java.io.File nf) {
        byte[] tmpbuf;
        try {
            java.io.FileInputStream fis = new java.io.FileInputStream(nf);
            tmpbuf = IOUtils.readFully(fis, LOCHDR, true);
            fis.close();
        } catch (IOException e) {
            return false;
        }
        if (get32(tmpbuf, 0) != LOCSIG) {
            return false;
        }
        return true;
    }

    /*
     * Fetches unsigned 16-bit value from byte array at specified offset. The
     * bytes are assumed to be in Intel (little-endian) byte order.
     */
    private static final int get16(byte b[], int off) {
        return (b[off] & 0xff) | ((b[off + 1] & 0xff) << 8);
    }

    /*
     * Fetches unsigned 32-bit value from byte array at specified offset. The
     * bytes are assumed to be in Intel (little-endian) byte order.
     */
    private static final long get32(byte b[], int off) {
        return get16(b, off) | ((long) get16(b, off + 2) << 16);
    }

    static {
        impl.sun.misc.SharedSecrets.setJavaUtilZipFileAccess(new impl.sun.misc.JavaUtilZipFileAccess() {
            @Override
            public boolean startsWithLocHeader(ZipFile zip) {
                return zip.startsWithLocHeader();
            }

            @Override
            public java.util.zip.ZipFile getPeer(ZipFile zip) {
                return zip.peer;
            }
        });
    }

    /**
     * Returns {@code true} if, and only if, the zip file begins with {@code
     * LOCSIG}.
     */
    private boolean startsWithLocHeader() {
        return locsig;
    }

    /**
     * Opens a ZIP file for reading given the specified File object.
     * 
     * @param file
     *            the ZIP file to be opened for reading
     * @throws ZipException
     *             if a ZIP error has occurred
     * @throws IOException
     *             if an I/O error has occurred
     */
    public ZipFile(File file) throws ZipException, IOException {
        this(file, OPEN_READ);
    }

    /**
     * Returns the zip file entry for the specified name, or null if not found.
     *
     * @param name
     *            the name of the entry
     * @return the zip file entry, or null if not found
     * @throws IllegalStateException
     *             if the zip file has been closed
     */
    public ZipEntry getEntry(String name) {
        if (name == null) {
            throw new NullPointerException("name");
        }
        ensureOpen();
        java.util.zip.ZipEntry ne = peer.getEntry(name);
        if (ne == null)
            return null;
        if (ne instanceof java.util.jar.JarEntry) {
            JarEntry je = new JarEntry(new ZipEntry(ne));
            JarEntryPeer.access.setPeer(je, ((java.util.jar.JarEntry) ne));
            return je;
        }
        return new ZipEntry(ne);
    }

    /**
     * Returns an input stream for reading the contents of the specified zip
     * file entry.
     *
     * <p>
     * Closing this ZIP file will, in turn, close all input streams that have
     * been returned by invocations of this method.
     *
     * @param entry
     *            the zip file entry
     * @return the input stream for reading the contents of the specified zip
     *         file entry.
     * @throws ZipException
     *             if a ZIP format error has occurred
     * @throws IOException
     *             if an I/O error has occurred
     * @throws IllegalStateException
     *             if the zip file has been closed
     */
    public InputStream getInputStream(ZipEntry entry) throws IOException {
        return getInputStream(entry.getName());
    }

    /**
     * Returns an input stream for reading the contents of the specified entry,
     * or null if the entry was not found.
     */
    private InputStream getInputStream(String name) throws IOException {
        if (name == null) {
            throw new NullPointerException("name");
        }
        ensureOpen();
        return peer.getInputStream(new java.util.zip.ZipEntry(name));
    }

    /**
     * Returns the path name of the ZIP file.
     * 
     * @return the path name of the ZIP file
     */
    public String getName() {
        return peer.getName();
    }

    /**
     * Returns an enumeration of the ZIP file entries.
     * 
     * @return an enumeration of the ZIP file entries
     * @throws IllegalStateException
     *             if the zip file has been closed
     */
    public Enumeration<? extends ZipEntry> entries() {
        ensureOpen();
        final Enumeration<? extends java.util.zip.ZipEntry> ne = peer.entries();
        if (ne == null)
            return null;
        return new Enumeration() {
            @Override
            public boolean hasMoreElements() {
                return ne.hasMoreElements();
            }

            @Override
            public Object nextElement() {
                Object n = ne.nextElement();
                Object j = null;
                if (n instanceof java.util.jar.JarEntry) {
                    java.util.jar.JarEntry ne = (java.util.jar.JarEntry) n;
                    JarEntry je = new JarEntry(new ZipEntry(ne));
                    JarEntryPeer.access.setPeer(je, ((java.util.jar.JarEntry) ne));
                    j = je;
                }
                if (n instanceof java.util.zip.ZipEntry) {
                    j = new ZipEntry((java.util.zip.ZipEntry) n);
                }
                return j;
            }
        };
    }

    /**
     * Returns the number of entries in the ZIP file.
     * 
     * @return the number of entries in the ZIP file
     * @throws IllegalStateException
     *             if the zip file has been closed
     */
    public int size() {
        ensureOpen();
        return peer.size();
    }

    /**
     * Closes the ZIP file.
     * <p>
     * Closing this ZIP file will close all of the input streams previously
     * returned by invocations of the {@link #getInputStream getInputStream}
     * method.
     *
     * @throws IOException
     *             if an I/O error has occurred
     */
    public void close() throws IOException {
        closeRequested = true;
        peer.close();
    }

    /**
     * Ensures that the <code>close</code> method of this ZIP file is called
     * when there are no more references to it.
     *
     * <p>
     * Since the time when GC would invoke this method is undetermined, it is
     * strongly recommended that applications invoke the <code>close</code>
     * method as soon they have finished accessing this <code>ZipFile</code>.
     * This will prevent holding up system resources for an undetermined length
     * of time.
     *
     * @throws IOException
     *             if an I/O error has occurred
     * @see java.util.zip.ZipFile#close()
     */
    protected void finalize() throws IOException {
        close();
    }

    private void ensureOpen() {
        if (closeRequested) {
            throw new IllegalStateException("zip file closed");
        }

        if (peer == null) {
            throw new IllegalStateException("The object is not initialized.");
        }
    }
}
