/*
 * Copyright (c) 1997, 2013, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package jail.java.security;

import java.security.Permission;
import java.security.PermissionCollection;

import impl.java.lang.PeerAccess;
import impl.java.security.CodeSourcePeer;
import impl.java.security.ProtectionDomainPeer;

/**
 *
 * <p>
 * This ProtectionDomain class encapsulates the characteristics of a domain,
 * which encloses a set of classes whose instances are granted a set of
 * permissions when being executed on behalf of a given set of Principals.
 * <p>
 * A static set of permissions can be bound to a ProtectionDomain when it is
 * constructed; such permissions are granted to the domain regardless of the
 * Policy in force. However, to support dynamic security policies, a
 * ProtectionDomain can also be constructed such that it is dynamically mapped
 * to a set of permissions by the current Policy whenever a permission is
 * checked.
 * <p>
 *
 * @author Li Gong
 * @author Roland Schemers
 * @author Gary Ellison
 */

public class ProtectionDomain {

    private java.security.ProtectionDomain peer;

    /**
     * Creates a new ProtectionDomain with the given CodeSource and Permissions.
     * If the permissions object is not null, then <code>setReadOnly())</code>
     * will be called on the passed in Permissions object. The only permissions
     * granted to this domain are the ones specified; the current Policy will
     * not be consulted.
     *
     * @param codesource
     *            the codesource associated with this domain
     * @param permissions
     *            the permissions granted to this domain
     */
    public ProtectionDomain(CodeSource codesource, PermissionCollection permissions) {
        peer = new java.security.ProtectionDomain(CodeSourcePeer.instance.fromJail(codesource), permissions);
    }

    private ProtectionDomain(java.security.ProtectionDomain peer) {
        this.peer = peer;
    }

    /**
     * Returns the CodeSource of this domain.
     * 
     * @return the CodeSource of this domain which may be null.
     * @since 1.2
     */
    public final CodeSource getCodeSource() {
        return CodeSourcePeer.instance.toJail(peer.getCodeSource());
    }

    /**
     * Returns the static permissions granted to this domain.
     *
     * @return the static set of permissions for this domain which may be null.
     * @see Policy#refresh
     * @see Policy#getPermissions(ProtectionDomain)
     */
    public final PermissionCollection getPermissions() {
        return peer.getPermissions();
    }

    /**
     * Check and see if this ProtectionDomain implies the permissions expressed
     * in the Permission object.
     * <p>
     * The set of permissions evaluated is a function of whether the
     * ProtectionDomain was constructed with a static set of permissions or it
     * was bound to a dynamically mapped set of permissions.
     * <p>
     * If the ProtectionDomain was constructed to a
     * {@link #ProtectionDomain(CodeSource, PermissionCollection) statically
     * bound} PermissionCollection then the permission will only be checked
     * against the PermissionCollection supplied at construction.
     * <p>
     * However, if the ProtectionDomain was constructed with the constructor
     * variant which supports
     * {@link #ProtectionDomain(CodeSource, PermissionCollection, ClassLoader, java.security.Principal[])
     * dynamically binding} permissions, then the permission will be checked
     * against the combination of the PermissionCollection supplied at
     * construction and the current Policy binding.
     * <p>
     *
     * @param permission
     *            the Permission object to check.
     *
     * @return true if "permission" is implicit to this ProtectionDomain.
     */
    public boolean implies(Permission permission) {
        return peer.implies(permission);
    }

    /**
     * Convert a ProtectionDomain to a String.
     */
    @Override
    public String toString() {

        java.security.CodeSource codesource = peer.getCodeSource();
        PermissionCollection pc = peer.getPermissions();

        return "ProtectionDomain " + " " + codesource + "\n" + " " + pc + "\n";
    }

    static {
        PeerAccess a = new PeerAccess<java.security.ProtectionDomain, jail.java.security.ProtectionDomain>() {
            public jail.java.security.ProtectionDomain construct(java.security.ProtectionDomain peer) {
                return new ProtectionDomain(peer);
            }

            public java.security.ProtectionDomain getPeer(jail.java.security.ProtectionDomain obj) {
                return (obj).peer;
            }
        };
        ProtectionDomainPeer.instance.setPeerAccess(a);
    }
}
