/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2023 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package impl.java.io;

import java.util.HashMap;

public abstract class FsDir extends FsObject {
    protected HashMap<String, FsObject> entries = new HashMap();
    private boolean entriesPopulated = false;

    protected abstract void populateEntries();

    protected abstract boolean deleteObject(FsObject obj);

    protected abstract FsDir createDirectoryObject(String name);

    protected abstract FsFile createFileObject(String name);

    protected abstract void closeThis();

    public FsObject findEntry(String Name) {
        if (!entriesPopulated) {
            populateEntries();
            entriesPopulated = true;
        }
        return entries.get(Name);
    }

    public static FsObject findEntry(FsDir root, String relativePath) {
        int index = relativePath.indexOf('/');
        if (index < 0) {
            return root.findEntry(relativePath);
        }

        String dirName = relativePath.substring(0, index);
        FsObject dir = root.findEntry(dirName);
        if (dir == null)
            return null;
        if (!(dir instanceof FsDir))
            return null;

        String tailPath = relativePath.substring(index + 1);

        return findEntry(((FsDir) dir), tailPath);
    }

    public String[] list() {
        if (!entriesPopulated) {
            populateEntries();
            entriesPopulated = true;
        }
        String[] names = new String[entries.size()];
        System.arraycopy(entries.keySet().toArray(), 0, names, 0, names.length);

        return names;
    }

    public int getBooleanAttributes() {
        return BA_EXISTS | BA_DIRECTORY;
    }

    public boolean delete(String name) {
        FsObject obj = findEntry(name);
        if (obj == null)
            return false;
        if (deleteObject(obj) == false) {
            return false;
        }
        entries.remove(name);
        return true;
    }

    public boolean createDirectory(String name) {
        if ((name.equals(".")) || (name.equals("..")))
            return false;
        FsDir dir = createDirectoryObject(name);
        if (dir == null)
            return false;
        entries.put(name, dir);
        return true;
    }

    public boolean createFileExclusively(String name) {
        if ((name.equals(".")) || (name.equals("..")))
            return false;
        FsFile file = createFileObject(name);
        if (file == null)
            return false;
        entries.put(name, file);
        return true;
    }

    public void close() {
        for (FsObject entry : entries.values()) {
            entry.close();
        }
        entries.clear();
        closeThis();
    }

}
