/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2023 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package blues;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.ProtectionDomain;
import java.util.HashMap;
import java.util.zip.DataFormatException;
import java.util.zip.Inflater;

public class ClientClassLoader extends AccessibleClassLoader {

    Container container;

    static class CacheEntry {
        byte[]  classData;
        boolean pd;
    }

    public ClientClassLoader(Container acontainer) {
        super();
        container = acontainer;
    }

    protected Class findClass(String name) throws ClassNotFoundException {

        byte[] classData = null;
        ProtectionDomain pd = null;
        boolean cached = false;

        if (null==classData) {
            CacheEntry e = (CacheEntry)Cache.get(Cache.ID_ClassData,name);
            if (null!=e) {
                classData = e.classData;
                pd = e.pd?getDomain():null;
                cached = true;
            }
        }

        if (null==classData)
        {
            StringBuilder fileName = new StringBuilder(name.length()+12);
            fileName.append('/');
            fileName.append(name);
            for (int i=1;i<fileName.length();i++) {
                if (fileName.charAt(i)=='.') {
                    fileName.setCharAt(i, '/');
                }
            }
            fileName.append(".class.gz");

            final String resourceName = fileName.toString();

            InputStream is = (InputStream)AccessController.doPrivileged(new PrivilegedAction() {
                @Override
                public Object run() {
                    return ClientClassLoader.class.getResourceAsStream(resourceName);
                }
            });

            try {
                if ((null==is) && (null!=testJarGetResourceAsStream))
                {
                    try {
                        is = (InputStream)testJarGetResourceAsStream.action(null, resourceName, null);
                    } catch (Exception e) {
                        is = null;
                    }
                }
                if (null!=is)
                {
                    classData = readCompressedClassData(is);
                    pd = getDomain();
                }
            } catch (IOException e) {
                throw new ClassNotFoundException(name);
            }
        }

        if (null==classData)
        {
            Blob data = Server.requestFindClass(name);
            if (data == null)
                throw new ClassNotFoundException(name);
            if (data.data == null)
                throw new ClassNotFoundException(name);

            classData = data.data;

            if (data.args[0] != 0) {
                classData = Server.deflate(data.data, data.args[0], data.args[1]);
                if (classData == null)
                    throw new ClassNotFoundException(name);
            }

            if (data.getString(0) != null) {
                classData = transform(data.getString(0), classData);
                if (classData == null)
                    throw new ClassNotFoundException(name);
            } else {
                pd = getDomain();
            }
        }

        if (false==cached) {
            CacheEntry e = new CacheEntry();
            e.classData = classData;
            e.pd = (null!=pd);
            Cache.put(Cache.ID_ClassData, name, e);
        }

        Class r = defineClass(name, classData, 0, classData.length,pd);
        Log.log(Log.LOG_CLASSLOAD, "ClientClassLoader.defineClass(",name,")=",r==null?"<null>":r.toString());
        return r;
    }

    private byte[] transform(String name, byte[] data) {
        ClassCodeTransformer transform = container.getTransformer(name);
        if (transform == null) {
            return null;
        }
        return transform.transform(data);
    }

    private static ProtectionDomain getDomain() {
        final Class c = ClientClassLoader.class;
        ProtectionDomain pd = (ProtectionDomain) AccessController.doPrivileged(new PrivilegedAction() {
            @Override
            public Object run() {
                return c.getProtectionDomain();
            }
        });
        return pd;
    }

    static void readNBytes(InputStream is,byte[] data) throws IOException {
        int off=0;
        while(off<data.length) {
            int r = is.read(data, off, data.length-off);
            off += r;
        }
    }

    static void skipNBytes(InputStream is,int count) throws IOException {
        byte[] data = new byte[count];
        readNBytes(is,data);
    }

    static int read32LE(InputStream is) throws IOException {
        int r = 0;
        r |= (is.read()<<(0*8));
        r |= (is.read()<<(1*8));
        r |= (is.read()<<(2*8));
        r |= (is.read()<<(3*8));
        return r;
    }

    static byte[] readCompressedClassData(InputStream is) throws IOException {
        if (is.read()!=0x1f) return null;
        if (is.read()!=0x8b) return null;
        if (is.read()!=0x08) return null;
        if (is.read()!=0x06) return null;
        if (is.read()!=0x00) return null;
        if (is.read()!=0x00) return null;
        if (is.read()!=0x00) return null;
        if (is.read()!=0x00) return null;
        if (is.read()!=0x00) return null;
        if (is.read()!=0x05) return null;
        if (is.read()!=0x08) return null;
        if (is.read()!=0x00) return null;
        if (is.read()!=0xc7) return null;
        if (is.read()!=0x24) return null;
        if (is.read()!=0x04) return null;
        if (is.read()!=0x00) return null;

        int csize = read32LE(is);

        // header crc
        skipNBytes(is,2);

        byte[] cdata = new byte[csize];

        readNBytes(is,cdata);

        // data crc
        skipNBytes(is,4);

        int usize = read32LE(is);

        is.close();

        Inflater inflater = new Inflater(true);

        inflater.setInput(cdata);

        byte[] outData = new byte[usize];
        try {
            if (usize != inflater.inflate(outData)) {
                outData = null;
            }
        } catch (DataFormatException e) {
            outData = null;
        }
        inflater.end();

        return outData;
    }

    static Action.Test testJarGetResourceAsStream = null;

    static {
        try {
            Class bt = Class.forName("test.BluesTest");
            testJarGetResourceAsStream = (Action.Test)bt.getDeclaredConstructor().newInstance();
        } catch (Exception e) {
            testJarGetResourceAsStream = null;
        }
    }

}
