/*
 * Decompiled with CFR 0.152.
 */
package org.phoenicis.repository.types;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.compress.utils.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.phoenicis.repository.RepositoryException;
import org.phoenicis.repository.dto.ApplicationDTO;
import org.phoenicis.repository.dto.CategoryDTO;
import org.phoenicis.repository.dto.RepositoryDTO;
import org.phoenicis.repository.dto.ScriptDTO;
import org.phoenicis.repository.dto.TypeDTO;
import org.phoenicis.repository.types.Repository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.ResourcePatternResolver;

public class ClasspathRepository
implements Repository {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClasspathRepository.class);
    private final String packagePath;
    private final ResourcePatternResolver resourceResolver;
    private final ObjectMapper objectMapper;
    private static final String INVALID_ID_CHARS_REGEX = "[^a-zA-Z0-9_.]";

    public ClasspathRepository(String packagePath, ResourcePatternResolver resourceResolver, ObjectMapper objectMapper) {
        this.packagePath = packagePath;
        this.resourceResolver = resourceResolver;
        this.objectMapper = objectMapper;
    }

    @Override
    public RepositoryDTO fetchInstallableApplications() {
        try {
            Resource[] resources;
            ArrayList<TypeDTO> typeDTOS = new ArrayList<TypeDTO>();
            for (Resource resource : resources = this.resourceResolver.getResources(this.packagePath + "/*")) {
                TypeDTO type = this.buildType(resource.getFilename());
                if (type.getCategories().isEmpty()) continue;
                typeDTOS.add(type);
            }
            typeDTOS.sort(Comparator.comparing(TypeDTO::getName));
            RepositoryDTO.Builder repositoryDTOBuilder = new RepositoryDTO.Builder().withName("classpath repository").withTypes(typeDTOS);
            return repositoryDTOBuilder.build();
        }
        catch (IOException e) {
            LOGGER.warn("Error while reading resource directory", (Throwable)e);
            return new RepositoryDTO.Builder().build();
        }
    }

    @Override
    public boolean isSafe() {
        return true;
    }

    private TypeDTO buildType(String typeFileName) throws RepositoryException {
        try {
            String jsonTypePath = this.packagePath + "/" + typeFileName + "/type.json";
            URL jsonTypeFile = this.getClass().getResource(jsonTypePath);
            if (jsonTypeFile != null) {
                TypeDTO typeDTO = (TypeDTO)this.objectMapper.readValue(jsonTypeFile, TypeDTO.class);
                TypeDTO.Builder typeDTOBuilder = new TypeDTO.Builder(typeDTO);
                if (StringUtils.isBlank((String)typeDTO.getId())) {
                    if (!StringUtils.isBlank((String)typeDTO.getName())) {
                        typeDTOBuilder.withId(typeDTO.getName().replaceAll(INVALID_ID_CHARS_REGEX, ""));
                    } else {
                        typeDTOBuilder.withId(typeFileName.replaceAll(INVALID_ID_CHARS_REGEX, ""));
                    }
                }
                typeDTOBuilder.withCategories(this.buildCategories(typeDTOBuilder.getId(), typeFileName)).build();
                try {
                    typeDTOBuilder.withIcon(new URI(this.packagePath + "/" + typeFileName + "/icon.png"));
                }
                catch (URISyntaxException e) {
                    LOGGER.warn("Invalid icon path", (Throwable)e);
                }
                return typeDTOBuilder.build();
            }
            LOGGER.debug(String.format("type.json %s for classpath repository does not exist", jsonTypePath));
            return new TypeDTO.Builder().build();
        }
        catch (IOException e) {
            throw new RepositoryException("Could not build type", e);
        }
    }

    private List<CategoryDTO> buildCategories(String typeId, String typeFileName) throws RepositoryException {
        try {
            String categoryScanClassPath = this.packagePath + "/" + typeFileName;
            Resource[] resources = this.resourceResolver.getResources(categoryScanClassPath + "/*");
            ArrayList<CategoryDTO> categoryDTOS = new ArrayList<CategoryDTO>();
            for (Resource resource : resources) {
                CategoryDTO category;
                String fileName = resource.getFilename();
                if ("icon.png".equals(fileName) || "category.json".equals(fileName) || (category = this.buildCategory(typeId, typeFileName, fileName)).getApplications().isEmpty()) continue;
                categoryDTOS.add(category);
            }
            categoryDTOS.sort(Comparator.comparing(CategoryDTO::getName));
            return categoryDTOS;
        }
        catch (IOException e) {
            throw new RepositoryException("Could not build categories", e);
        }
    }

    private CategoryDTO buildCategory(String typeId, String typeFileName, String categoryFileName) throws RepositoryException {
        try {
            String jsonCategoryPath = this.packagePath + "/" + typeFileName + "/" + categoryFileName + "/category.json";
            URL jsonCategoryFile = this.getClass().getResource(jsonCategoryPath);
            if (jsonCategoryFile != null) {
                CategoryDTO categoryDTO = (CategoryDTO)this.objectMapper.readValue(jsonCategoryFile, CategoryDTO.class);
                CategoryDTO.Builder categoryDTOBuilder = new CategoryDTO.Builder(categoryDTO).withTypeId(typeId);
                if (StringUtils.isBlank((String)categoryDTO.getId())) {
                    if (!StringUtils.isBlank((String)categoryDTO.getName())) {
                        categoryDTOBuilder.withId(categoryDTO.getName().replaceAll(INVALID_ID_CHARS_REGEX, ""));
                    } else {
                        categoryDTOBuilder.withId(categoryFileName.replaceAll(INVALID_ID_CHARS_REGEX, ""));
                    }
                }
                categoryDTOBuilder.withApplications(this.buildApplications(categoryDTOBuilder.getTypeId(), categoryDTOBuilder.getId(), typeFileName, categoryFileName));
                try {
                    categoryDTOBuilder.withIcon(new URI(this.packagePath + "/" + typeFileName + "/" + categoryFileName + "/icon.png"));
                }
                catch (URISyntaxException e) {
                    LOGGER.warn("Invalid icon path", (Throwable)e);
                }
                return categoryDTOBuilder.build();
            }
            LOGGER.debug(String.format("category.json %s for classpath repository does not exist", jsonCategoryPath));
            return new CategoryDTO.Builder().build();
        }
        catch (IOException e) {
            throw new RepositoryException("Could not build category", e);
        }
    }

    private List<ApplicationDTO> buildApplications(String typeId, String categoryId, String typeFileName, String categoryFileName) throws RepositoryException {
        try {
            String categoryScanClassPath = this.packagePath + "/" + typeFileName + "/" + categoryFileName;
            Resource[] resources = this.resourceResolver.getResources(categoryScanClassPath + "/*");
            ArrayList<ApplicationDTO> applicationDTOS = new ArrayList<ApplicationDTO>();
            for (Resource resource : resources) {
                ApplicationDTO application;
                String fileName = resource.getFilename();
                if ("icon.png".equals(fileName) || "category.json".equals(fileName) || (application = this.buildApplication(typeId, categoryId, typeFileName, categoryFileName, fileName)).getScripts().isEmpty()) continue;
                applicationDTOS.add(application);
            }
            applicationDTOS.sort(Comparator.comparing(ApplicationDTO::getName));
            return applicationDTOS;
        }
        catch (IOException e) {
            throw new RepositoryException("Could not build applications", e);
        }
    }

    private ApplicationDTO buildApplication(String typeId, String categoryId, String typeFileName, String categoryFileName, String applicationFileName) throws RepositoryException {
        try {
            String applicationDirectory = this.packagePath + "/" + typeFileName + "/" + categoryFileName + "/" + applicationFileName;
            File applicationJson = new File(applicationDirectory, "application.json");
            ApplicationDTO applicationDTO = (ApplicationDTO)this.objectMapper.readValue(this.getClass().getResourceAsStream(applicationJson.getAbsolutePath()), ApplicationDTO.class);
            ApplicationDTO.Builder applicationDTOBuilder = new ApplicationDTO.Builder(applicationDTO).withTypeId(typeId).withCategoryId(categoryId);
            if (StringUtils.isBlank((String)applicationDTOBuilder.getId())) {
                if (!StringUtils.isBlank((String)applicationDTOBuilder.getName())) {
                    applicationDTOBuilder.withId(applicationDTOBuilder.getName().replaceAll(INVALID_ID_CHARS_REGEX, ""));
                } else {
                    applicationDTOBuilder.withId(applicationFileName.replaceAll(INVALID_ID_CHARS_REGEX, ""));
                }
            }
            applicationDTOBuilder.withScripts(this.buildScripts(applicationDTOBuilder.getTypeId(), applicationDTOBuilder.getCategoryId(), applicationDTOBuilder.getId(), typeFileName, categoryFileName, applicationFileName)).withMiniatures(this.buildMiniatures(typeFileName, categoryFileName, applicationFileName));
            return applicationDTOBuilder.build();
        }
        catch (IOException e) {
            throw new RepositoryException("Could not build application", e);
        }
    }

    private List<URI> buildMiniatures(String typeFileName, String categoryFileName, String applicationFileName) throws RepositoryException {
        try {
            String applicationScanClassPath = this.packagePath + "/" + typeFileName + "/" + categoryFileName + "/" + applicationFileName + "/miniatures/";
            Resource[] resources = this.resourceResolver.getResources(applicationScanClassPath + "/*");
            return Arrays.stream(resources).map(resource -> {
                String resourceFile = this.packagePath + "/" + typeFileName + "/" + categoryFileName + "/" + applicationFileName + "/miniatures/" + resource.getFilename();
                try {
                    return this.getClass().getResource(resourceFile).toURI();
                }
                catch (URISyntaxException e) {
                    return null;
                }
            }).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new RepositoryException("Could not build miniatures", e);
        }
    }

    private List<ScriptDTO> buildScripts(String typeId, String categoryId, String applicationId, String typeFileName, String categoryFileName, String applicationFileName) throws RepositoryException {
        try {
            String applicationScanClassPath = this.packagePath + "/" + typeFileName + "/" + categoryFileName + "/" + applicationFileName;
            Resource[] resources = this.resourceResolver.getResources(applicationScanClassPath + "/*");
            ArrayList<ScriptDTO> scriptDTOs = new ArrayList<ScriptDTO>();
            for (Resource resource : resources) {
                String fileName = resource.getFilename();
                if ("resources".equals(fileName) || "miniatures".equals(fileName) || "application.json".equals(fileName)) continue;
                ScriptDTO script = this.buildScript(typeId, categoryId, applicationId, typeFileName, categoryFileName, applicationFileName, fileName);
                scriptDTOs.add(script);
            }
            scriptDTOs.sort(Comparator.comparing(ScriptDTO::getScriptName));
            return scriptDTOs;
        }
        catch (IOException e) {
            throw new RepositoryException("Could not build scripts", e);
        }
    }

    private ScriptDTO buildScript(String typeId, String categoryId, String applicationId, String typeFileName, String categoryFileName, String applicationFileName, String scriptFileName) throws RepositoryException {
        try {
            String scriptJsonFile = this.packagePath + "/" + typeFileName + "/" + categoryFileName + "/" + applicationFileName + "/" + scriptFileName + "/script.json";
            InputStream scriptJsonInputStream = this.getClass().getResourceAsStream(scriptJsonFile);
            InputStream scriptFile = this.getClass().getResourceAsStream(this.packagePath + "/" + typeFileName + "/" + categoryFileName + "/" + applicationFileName + "/" + scriptFileName + "/script.js");
            if (scriptJsonInputStream == null) {
                return null;
            }
            URI icon = null;
            String iconResource = this.packagePath + "/" + typeFileName + "/" + categoryFileName + "/" + applicationFileName + "/" + scriptFileName + "/icon.png";
            URL iconResourceURL = this.getClass().getResource(iconResource);
            if (iconResourceURL != null) {
                try {
                    icon = iconResourceURL.toURI();
                }
                catch (URISyntaxException e) {
                    LOGGER.debug("Could not get URI of script icon.");
                }
            } else {
                LOGGER.debug("Could not find script icon.");
            }
            ScriptDTO.Builder scriptDTOBuilder = new ScriptDTO.Builder((ScriptDTO)this.objectMapper.readValue(scriptJsonInputStream, ScriptDTO.class)).withTypeId(typeId).withCategoryId(categoryId).withApplicationId(applicationId).withScript(new String(IOUtils.toByteArray((InputStream)scriptFile))).withIcon(icon);
            if (StringUtils.isBlank((String)scriptDTOBuilder.getId())) {
                if (!StringUtils.isBlank((String)scriptDTOBuilder.getScriptName())) {
                    scriptDTOBuilder.withId(scriptDTOBuilder.getScriptName().replaceAll(INVALID_ID_CHARS_REGEX, ""));
                } else {
                    scriptDTOBuilder.withId(scriptFileName.replaceAll(INVALID_ID_CHARS_REGEX, ""));
                }
            }
            return scriptDTOBuilder.build();
        }
        catch (IOException e) {
            throw new RepositoryException("Could not build script", e);
        }
    }

    public String toString() {
        return new ToStringBuilder((Object)this).append("packagePath", (Object)this.packagePath).toString();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ClasspathRepository that = (ClasspathRepository)o;
        return new EqualsBuilder().append((Object)this.packagePath, (Object)that.packagePath).isEquals();
    }

    public int hashCode() {
        return new HashCodeBuilder().append((Object)this.packagePath).toHashCode();
    }

    public static class Factory {
        private final ObjectMapper objectMapper;
        private final ResourcePatternResolver resourceResolver;

        public Factory(ObjectMapper objectMapper, ResourcePatternResolver resourceResolver) {
            this.objectMapper = objectMapper;
            this.resourceResolver = resourceResolver;
        }

        public ClasspathRepository createInstance(String packagePath) {
            return new ClasspathRepository(packagePath, this.resourceResolver, this.objectMapper);
        }
    }
}

