/*
 * Decompiled with CFR 0.152.
 */
package org.phoenicis.tools.files;

import com.google.common.collect.Sets;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import org.apache.commons.io.FileUtils;
import org.phoenicis.configuration.security.Safe;
import org.phoenicis.tools.files.FilesManipulator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

@Safe
public class FileUtilities
extends FilesManipulator {
    private final Logger LOGGER = LoggerFactory.getLogger(FileUtilities.class);
    @Value(value="${application.user.tmp}")
    private String tmpDirectory;

    public String[] ls(String path) {
        File directory = new File(path);
        if (!directory.exists()) {
            throw new IllegalArgumentException(String.format("Path \"%s\" does not exist", path));
        }
        if (!directory.isDirectory()) {
            throw new IllegalArgumentException(String.format("Path \"%s\" is not a directory", path));
        }
        this.assertInDirectory(directory);
        File[] files = directory.listFiles();
        if (files == null) {
            throw new IllegalArgumentException(String.format("Path \"%s\" does not denote a directory", path));
        }
        return (String[])Arrays.stream(files).map(File::getName).toArray(String[]::new);
    }

    public boolean exists(String path) {
        File file = new File(path);
        this.assertInDirectory(file);
        return file.exists();
    }

    public void mkdir(String path) {
        File directoryToCreate = new File(path);
        this.assertInDirectory(directoryToCreate);
        directoryToCreate.mkdirs();
    }

    public void createSymbolicLink(String linkPath, String targetPath) throws IOException {
        File link = new File(linkPath);
        File target = new File(targetPath);
        if (!target.exists()) {
            throw new IllegalArgumentException(String.format("Path \"%s\" does not exist", targetPath));
        }
        if (link.exists()) {
            throw new IllegalArgumentException(String.format("Link \"%s\" already exists", linkPath));
        }
        this.assertInDirectory(link);
        this.assertInDirectory(target);
        Files.createSymbolicLink(link.toPath(), target.toPath(), new FileAttribute[0]);
    }

    public void copy(String sourcePath, String targetPath) throws IOException {
        File source = new File(sourcePath);
        File target = new File(targetPath);
        if (!source.exists()) {
            throw new IllegalArgumentException(String.format("Path \"%s\" does not exist", sourcePath));
        }
        this.assertInDirectory(source);
        this.assertInDirectory(target);
        if (source.isDirectory()) {
            FileUtils.copyDirectory((File)source, (File)target);
        } else if (target.isDirectory()) {
            FileUtils.copyFile((File)source, (File)new File(target, source.getName()));
        } else {
            FileUtils.copyFile((File)source, (File)target);
        }
    }

    public void remove(String path) throws IOException {
        File fileToDelete = new File(path);
        this.assertInDirectory(fileToDelete);
        if (!fileToDelete.exists()) {
            this.LOGGER.debug(String.format("Cannot remove file or directory: path \"%s\" does not exist", path));
            return;
        }
        if (fileToDelete.isDirectory()) {
            FileUtils.deleteDirectory((File)fileToDelete);
        } else {
            Files.deleteIfExists(fileToDelete.toPath());
        }
    }

    public String getFileName(String path) {
        File file = new File(path);
        if (!file.exists()) {
            throw new IllegalArgumentException(String.format("Path \"%s\" does not exist", path));
        }
        this.assertInDirectory(file);
        return file.getName();
    }

    public String getFileContent(String path) throws IOException {
        File file = new File(path);
        if (!file.exists()) {
            throw new IllegalArgumentException(String.format("Path \"%s\" does not exist", path));
        }
        this.assertInDirectory(file);
        return FileUtils.readFileToString((File)file, (String)"UTF-8");
    }

    public long getSize(String path) throws IOException {
        File file = new File(path);
        if (!file.exists()) {
            throw new IllegalArgumentException(String.format("Path \"%s\" does not exist", path));
        }
        this.assertInDirectory(file);
        Path folder = Paths.get(file.getAbsolutePath(), new String[0]);
        return Files.walk(folder, new FileVisitOption[0]).filter(p -> p.toFile().isFile()).mapToLong(p -> p.toFile().length()).sum();
    }

    public void writeToFile(String path, String content) throws IOException {
        File file = new File(path);
        this.assertInDirectory(file);
        FileUtils.writeStringToFile((File)file, (String)content, (String)"UTF-8");
    }

    public String createTmpFile(String extension) throws IOException {
        File tmpDirectoryFile = new File(this.tmpDirectory);
        tmpDirectoryFile.mkdirs();
        File file = File.createTempFile("phoenicis", "." + extension, tmpDirectoryFile);
        file.deleteOnExit();
        return file.getAbsolutePath();
    }

    public String createTmpDir() throws IOException {
        File tmpDirectoryFile = new File(this.tmpDirectory);
        tmpDirectoryFile.mkdirs();
        File file = Files.createTempDirectory(tmpDirectoryFile.toPath(), "phoenicis", new FileAttribute[0]).toFile();
        file.deleteOnExit();
        return file.getAbsolutePath();
    }

    public void chmod(String filePath, String permissions) throws IOException {
        Path path = Paths.get(filePath, new String[0]);
        if (!path.toFile().exists()) {
            throw new IllegalArgumentException(String.format("Path \"%s\" does not exist", path));
        }
        Set<PosixFilePermission> permissionsObj = PosixFilePermissions.fromString(permissions);
        Files.setPosixFilePermissions(path, permissionsObj);
    }

    private Set<PosixFilePermission> singleIntToFilePermission(Integer mode, String groupType) {
        HashSet<PosixFilePermission> permissions = new HashSet<PosixFilePermission>(9);
        if (Arrays.asList(1, 3, 5, 7).contains(mode)) {
            permissions.add(PosixFilePermission.valueOf(groupType + "_EXECUTE"));
        }
        if (Arrays.asList(2, 3, 6, 7).contains(mode)) {
            permissions.add(PosixFilePermission.valueOf(groupType + "_WRITE"));
        }
        if (Arrays.asList(4, 5, 6, 7).contains(mode)) {
            permissions.add(PosixFilePermission.valueOf(groupType + "_READ"));
        }
        return permissions;
    }

    public Set<PosixFilePermission> intToPosixFilePermission(int mode) {
        if (mode >= 1000 || mode < 0) {
            throw new IllegalArgumentException("Invalid mode " + mode);
        }
        int owner = mode / 100;
        int group = (mode - owner * 100) / 10;
        int others = mode - owner * 100 - group * 10;
        if (owner > 7 || group > 7 || others > 7) {
            throw new IllegalArgumentException("Invalid mode " + mode);
        }
        return Sets.union((Set)Sets.union(this.singleIntToFilePermission(owner, "OWNER"), this.singleIntToFilePermission(group, "GROUP")), this.singleIntToFilePermission(others, "OTHERS"));
    }

    public Set<PosixFilePermission> octToPosixFilePermission(int modeOct) {
        int modeInt = Integer.parseInt(Integer.toString(modeOct, 8));
        return this.intToPosixFilePermission(modeInt);
    }
}

