#ifndef _DRV_TAPI_LL_INTERFACE_H
#define _DRV_TAPI_LL_INTERFACE_H

/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/
/**
   \file drv_tapi_ll_interface.h
   Contains the structures which are shared between the high-level TAPI and
    low-level TAPI drivers.
   These structures mainly contains the function pointers which will be exported
   by the low level driver.

   This file is divided in different sections depending on the firmware modules:
   - INTERRUPT_AND_PROTECTION_MODULE
   - CODER_MODULE
   - PCM_MODULE
   - SIG_MODULE
   - ALM_MODULE
   - CON_MODULE
   - MISC_MODULE

*/
/* ============================= */
/* Includes                      */
/* ============================= */

#include "ifx_types.h"
#include "drv_tapi_io.h"
#include "drv_tapi_kpi_io.h"
#ifdef TAPI_PACKET
#include "lib_fifo.h"
#include "ifxos_select.h"
#endif /* TAPI_PACKET */

#ifndef HAVE_IFX_ULONG_T
   #warning please update your ifx_types.h, using local definition of IFX_ulong_t
   /* unsigned long type - valid for 32bit systems only */
   typedef unsigned long               IFX_ulong_t;
   #define HAVE_IFX_ULONG_T
#endif /* HAVE_IFX_ULONG_T */

/* ============================= */
/* Local Macros  Definitions    */
/* ============================= */
#define TAPI_MAX_DEVFS_HANDLES 20


#ifdef LINUX_2_6
#undef CONFIG_DEVFS_FS
#endif /* LINUX_2_6 */

/* Channel 255 indicates the control device */
#define IFX_TAPI_DEVICE_CH_NUMBER    255

/* channel flags */
enum CH_FLAGS
{
   /* indicates a non-blocking read driver's function */
   CF_NONBLOCK             = 0x00000008,
   /* Indicates that a task is pending via select on this device */
   /* Only used in vxWorks */
   CF_NEED_WAKEUP          = 0x00100000,
   /* indication the wakeup source of this channel */
   /* only used in vxworks */
   CF_WAKEUPSRC_TAPI       = 0x00200000,
   CF_WAKEUPSRC_STREAM     = 0x00400000,
   CF_WAKEUPSRC_GR909      = 0x00800000
};

/** MFTD Answering tone detector phase reversal threshold
    Defines on which occurence of a phase reversal the signal is sent when
    enabled. Allowed range: 1-3. */
#define IFX_TAPI_SIG_MFTD_PR_THRESHOLD      (2)

/** FXO event status bit numbers */
#define FXO_BATTERY  0
#define FXO_APOH     1
#define FXO_POLARITY 2
#define FXO_RING     3
/** FXO line mode bit */
#define FXO_LINE_MODE 31

/* ================================ */
/* Enumerations                     */
/* ================================ */

/** Hook validation time array offsets for IFX_TAPI_LL_ALM_HookVt */
enum HOOKVT_IDX
{
   HOOKVT_IDX_HOOK_OFF = 0,
   HOOKVT_IDX_HOOK_ON,
   HOOKVT_IDX_HOOK_FLASH,
   HOOKVT_IDX_HOOK_FLASHMAKE,
   HOOKVT_IDX_DIGIT_LOW,
   HOOKVT_IDX_DIGIT_HIGH,
   HOOKVT_IDX_INTERDIGIT
};

/** Definition of different data streams */
typedef enum
{
   IFX_TAPI_STREAM_COD = 0,
   IFX_TAPI_STREAM_DECT,
   IFX_TAPI_STREAM_LIN,
   IFX_TAPI_STREAM_HDLC,
   IFX_TAPI_STREAM_MAX
} IFX_TAPI_STREAM_t;

/** Counters for packet statistics */
typedef enum
{
   /** total number of packets delivered without errors */
   TAPI_STAT_COUNTER_EGRESS_DELIVERED,
   /** total number of packets discarded */
   TAPI_STAT_COUNTER_EGRESS_DISCARDED,
   /** number of packets discarded due to congestion */
   TAPI_STAT_COUNTER_EGRESS_CONGESTED,
#if 0
   /** minimum latency of all packets transported */
   TAPI_STAT_COUNTER_EGRESS_LATENCY_MIN,
   /** maximum latency of all packets transported */
   TAPI_STAT_COUNTER_EGRESS_LATENCY_MAX,
   /** sum of the latency */
   TAPI_STAT_COUNTER_EGRESS_LATENCY_SUM,
#endif
   /** total number of packets delivered without errors */
   TAPI_STAT_COUNTER_INGRESS_DELIVERED,
   /** total number of packets discarded */
   TAPI_STAT_COUNTER_INGRESS_DISCARDED,
   /** number of packets discarded due to congestion */
   TAPI_STAT_COUNTER_INGRESS_CONGESTED,
#if 0
   /** minimum latency of all packets transported */
   TAPI_STAT_COUNTER_INGRESS_LATENCY_MIN,
   /** maximum latency of all packets transported */
   TAPI_STAT_COUNTER_INGRESS_LATENCY_MAX,
   /** sum of the latency */
   TAPI_STAT_COUNTER_INGRESS_LATENCY_SUM,
#endif
   /* maximum value of this enum */
   TAPI_STAT_COUNTER_MAX
} TAPI_STAT_COUNTER_t;

/** Firmware module type definition */
typedef enum
{
   /** Reserved */
   IFX_TAPI_MODULE_TYPE_NONE = 0x0,
   /** ALM-Module */
   IFX_TAPI_MODULE_TYPE_ALM = 0x1,
   /** PCM-Module */
   IFX_TAPI_MODULE_TYPE_PCM = 0x2,
   /** COD-Module (Coder Module) */
   IFX_TAPI_MODULE_TYPE_COD = 0x3,
   /* Conference-Module (currently not supported) */
   IFX_TAPI_MODULE_TYPE_CONF = 0x4,
   /** DECT-Module */
   IFX_TAPI_MODULE_TYPE_DECT = 0x5,
   /** Applies to all available module types. */
   IFX_TAPI_MODULE_TYPE_ALL = 0xFF
} IFX_TAPI_MODULE_TYPE_t;

typedef enum
{
   IFX_TAPI_LL_TONE_DIR_NONE = 0,
   IFX_TAPI_LL_TONE_EXTERNAL = 0x1,
   IFX_TAPI_LL_TONE_INTERNAL = 0x2,
   IFX_TAPI_LL_TONE_BOTH = 0x3
} IFX_TAPI_LL_TONE_DIR_t;

/** Direction a SIG detector receives it's input from */
typedef enum
{
   IFX_TAPI_LL_SIG_IN_DIR_NONE = 0,
   IFX_TAPI_LL_SIG_IN_DIR_LOCAL = 1,
   IFX_TAPI_LL_SIG_IN_DIR_REMOTE = 2,
   IFX_TAPI_LL_SIG_IN_DIR_BOTH = 3
} IFX_TAPI_LL_SIG_IN_DIR;

typedef IFX_void_t      IFX_TAPI_LL_DEV_t;
typedef IFX_void_t      IFX_TAPI_LL_CH_t;
typedef IFX_uint16_t    IFX_TAPI_LL_ERR_t;

#ifndef DRV_TAPI_H
typedef struct _TAPI_DEV      TAPI_DEV;
typedef struct _TAPI_CHANNEL  TAPI_CHANNEL;
#endif /* DRV_TAPI_H */

typedef struct
{
   IFX_TAPI_COD_TYPE_t   dec_type;
   IFX_TAPI_COD_LENGTH_t dec_framelength;
} IFX_TAPI_DEC_DETAILS_t;

/** Internal struct that is put as an element into the packet fifos and keeps
    the actual packet buffer together with packet related extra information. */
typedef struct
{
   /** Pointer to a buffer from lib-bufferpool with the payload data */
   IFX_void_t          *pBuf;
   /** Offset from buffer header to the data counted in bytes */
   IFX_uint32_t         nDataOffset;
   /* add additional attributes here such as length, type, ... */
} IFX_TAPI_FIFO_ELEM_t;


/** \defgroup TAPI_LL_INTERFACE TAPI Low-Level driver interface
   Lists all the functions which are registered by the low level TAPI driver */
/*@{*/

/** \defgroup INTERRUPT_AND_PROTECTION_MODULE Protection service
   This service is used to lock and unlock the interrupts for protecting the
   access to shared data structures.   */

/** \defgroup CODER_MODULE  Coder service
   This service is used to access the coder module functionalities like encoding
   and decoding, RTP, RTCP etc., */

/** \defgroup PCM_MODULE PCM module services
   The PCM - */

/** \defgroup SIG_MODULE Signaling module service
   This service includes the functionalities like DTMF receiver, Tone detection/
   Generation, CID receiver and sender */

/** \defgroup ALM_MODULE Analong Line interface Module service
   Contains the functionalities of ALM module */

/** \defgroup DECT_MODULE DECT module service
   This services provides access to the special coders/decoders that are used
   in conjunction with DECT packet streams. Also a special tone generator is
   provided for DECT channels. */

/** \defgroup CON_MODULE Connection Module
  This module provides functions to connect different DSP modules. It is used
  for conferencing, but also for basic dynamic connections */

/*@}*/

/* ============================= */
/* Structure for LEC data        */
/*                               */
/* ============================= */
typedef struct
{
   /* LEC operating mode */
   IFX_TAPI_WLEC_TYPE_t nOpMode;
   /* Non Linear Processing on or off */
   IFX_int8_t bNlp;
   /* Gain for input or LEC off */
   IFX_int8_t nGainIn;
   /* Gain for ouput or LEC off */
   IFX_int8_t nGainOut;
   /** LEC tail length - unused only a storage needed for get function */
   IFX_int8_t nLen;
   /** Size of the near-end window in narrowband sampling mode. */
   IFX_TAPI_WLEC_WIN_SIZE_t   nNBNEwindow;
   /** Size of the far-end window in narrowband sampling mode.
       Note: this is used only if nOpMode is set to IFX_TAPI_LEC_TYPE_NFE */
   IFX_TAPI_WLEC_WIN_SIZE_t   nNBFEwindow;
   /** Size of the near-end window in wideband sampling mode. */
   IFX_TAPI_WLEC_WIN_SIZE_t   nWBNEwindow;
}TAPI_LEC_DATA_t;

/** tone play destinations */
typedef enum
{
   TAPI_TONE_DST_DEFAULT  = 0x0,
   TAPI_TONE_DST_LOCAL    = 0x1,
   TAPI_TONE_DST_NET      = 0x2,
   TAPI_TONE_DST_NETLOCAL = 0x3
}TAPI_TONE_DST;


/** Specifies the capability of the tone generator regarding tone
    sequence support. */
typedef enum
{
   /** Plays out a frequency or silence. No tone sequence with
      cadences are supported */
   IFX_TAPI_TONE_RESSEQ_FREQ = 0x0,
   /** Plays out a full simple tone including cadences and loops */
   IFX_TAPI_TONE_RESSEQ_SIMPLE = 0x1
}IFX_TAPI_TONE_RESSEQ_t;

/** Tone resource information. */
typedef struct
{
   /** Resource ID or number of the generator. Used as index in the tone status array and
   must be a number between 0 and TAPI_TONE_MAXRES */
   IFX_uint8_t nResID;
   /** Number of maximum supported frequencies at one time */
   IFX_uint8_t nFreq;
   /** Specifies the capability of the tone generator regarding tone
       sequence support. See \ref IFX_TAPI_TONE_RESSEQ_t for details. */
   IFX_TAPI_TONE_RESSEQ_t sequenceCap;
} IFX_TAPI_TONE_RES_t;

/* =============================== */
/* Defines for complex tone states */
/* =============================== */
typedef enum
{
   /** Initialization state */
   TAPI_CT_IDLE = 0,
   /** UTG tone sequence is not active, but the tone is in pause state */
   TAPI_CT_ACTIVE_PAUSE,
   /** Tone is currently playing out on the tone generator */
   TAPI_CT_ACTIVE,
   /* UTG tone sequence is deactived by the LL driver automatically. Afterwards a
   next step can be programmed or a new tone can be started. */
   TAPI_CT_DEACTIVATED

}TAPI_CMPLX_TONE_STATE_t;

typedef enum
{
   /** RTP protocol */
   IFX_TAPI_PRT_TYPE_RTP,
   /** AAL protocol */
   IFX_TAPI_PRT_TYPE_AAL
}IFX_TAPI_PRT_TYPE_t;

/* ============================= */
/* Structure for CID data        */
/* ============================= */

/** Caller ID transmission data types. */
typedef enum
{
   IFX_TAPI_CID_DATA_TYPE_FSK_BEL202,
   IFX_TAPI_CID_DATA_TYPE_FSK_V23,
   IFX_TAPI_CID_DATA_TYPE_DTMF,
} IFX_TAPI_CID_DATA_TYPE_t;

/** Structure used for starting the CID transmitter in the LL-driver. */
typedef struct
{
   /* Select CID type 1 (ON-) or type 2 (OFF-HOOK) */
   IFX_TAPI_CID_HOOK_MODE_t   txHookMode;
   /* The type of CID data to be sent */
   IFX_TAPI_CID_DATA_TYPE_t   cidDataType;
   /* CID data to be sent */
   IFX_uint8_t                *pCidParam;
   /* number of CID data octets */
   IFX_uint16_t               nCidParamLen;
   /* FSK transmitter/receiver specific settings */
   IFX_TAPI_CID_FSK_CFG_t     *pFskConf;
   /* DTMF transmitter specific settings */
   IFX_TAPI_CID_DTMF_CFG_t    *pDtmfConf;
} IFX_TAPI_CID_TX_t;

/** Struct used for starting the CID receiver in the LL-driver. */
typedef struct
{
   /* The configured CID standard */
   IFX_TAPI_CID_STD_t         nStandard;
   /* Select CID type 1 (ON-) or type 2 (OFF-HOOK) */
   IFX_TAPI_CID_HOOK_MODE_t   txHookMode;
   /* FSK transmitter/receiver specific settings */
   IFX_TAPI_CID_FSK_CFG_t     *pFskConf;
} IFX_TAPI_CID_RX_t;


/* =============================== */
/* Defines for error reporting     */
/* =============================== */

/** Classify error when reported via event dispatcher. For internal use */
typedef enum
{
   /** Report TAPI channel specific error */
   IFX_TAPI_ERRSRC_TAPI_CH    = 0,
   /** Report TAPI device or global error */
   IFX_TAPI_ERRSRC_TAPI_DEV   = 0x1000,
   /** Report low level global error */
   IFX_TAPI_ERRSRC_LL_DEV     = 0x2000,
   /** Report low level channel specific error */
   IFX_TAPI_ERRSRC_LL_CH      = 0x4000,
   /** Bit mask for modification of low level driver error codes. This bit
       is set for low level driver error codes */
   IFX_TAPI_ERRSRC_LL         = 0x8000,
   /** Maks of error sources used for clearing */
   IFX_TAPI_ERRSRC_MASK       = (IFX_TAPI_ERRSRC_LL |
                                 IFX_TAPI_ERRSRC_LL_CH |
                                 IFX_TAPI_ERRSRC_LL_DEV |
                                 IFX_TAPI_ERRSRC_TAPI_DEV |
                                 IFX_TAPI_ERRSRC_TAPI_CH)
}IFX_TAPI_ERRSRC;


/* ============================= */
/* Defines for resource counts   */
/* ============================= */

/** Struct used for reporting resources from LL to HL. */
typedef struct
{
   /** Number of ALM modules or analog lines */
   IFX_uint16_t            AlmCount;
   /** Number of SIG modules */
   IFX_uint16_t            SigCount;
   /** Number of COD modules */
   IFX_uint16_t            CodCount;
   /** Number of PCM modules */
   IFX_uint16_t            PcmCount;
   /** Number of DECT coder modules */
   IFX_uint16_t            DectCount;
   /** Number of AUDIO modules */
   IFX_uint16_t            AudioCount;
   /** Number of DTMF generators */
   IFX_uint16_t            DTMFGCount;
   /** Number of DTMF receivers */
   IFX_uint16_t            DTMFRCount;
   /** Number of FSK generators */
   IFX_uint16_t            FSKGCount;
   /** Number of FSK receivers */
   IFX_uint16_t            FSKRCount;
   /** Number of UGT/TG capable channels */
   IFX_uint16_t            ToneCount;
   /** Number of HDLC channel resources */
   IFX_uint16_t            HdlcCount;
   /* additional resources can be added here
      initialise them in ifx_tapi_Prepare_Dev() */
} IFX_TAPI_RESOURCE;

/* ============================= */
/* Structure for CERR reporting  */
/* ============================= */

/** Report the reason and details of a cmd error to drv_tapi. */
typedef struct _IFX_TAPI_DBG_CERR {
   /** reason code */
   IFX_uint32_t      cause;
   /** cmd header */
   IFX_uint32_t      cmd;
} IFX_TAPI_DBG_CERR_t;



/* ============================= */
/* Defines for Driver Context    */
/* ============================= */

/** Interrupt and Protection Module */
/** \addtogroup INTERRUPT_AND_PROTECTION_MODULE */
/** Used for data protection by higher layer */
/*@{*/
typedef struct
{
   /** This function disables the irq line if the driver is in interrupt mode
   \param
   pLLDev   - Low-level device pointer */
   IFX_void_t (*LockDevice)                 (IFX_TAPI_LL_DEV_t *);
   /** This function enables the irq line if the driver is in interrupt mode
   \param
   pLLDev   - Low-level device pointer */
   IFX_void_t (*UnlockDevice)               (IFX_TAPI_LL_DEV_t *);
   IFX_void_t (*IrqEnable)                  (IFX_TAPI_LL_DEV_t *);
   IFX_void_t (*IrqDisable)                 (IFX_TAPI_LL_DEV_t *);
} IFX_TAPI_DRV_CTX_IRQ_t;

/*@}*/ /* INTERRUPT_AND_PROTECTION_MODULE */


/** CODer Module */  /* ***************************************************** */
/** \addtogroup CODER_MODULE */
/** Used for Coder services higher layer */
/*@{*/
typedef struct
{
   /** Starts the coder in Upstream / record data
   \param pLLCh       Pointer to low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured
   */
   IFX_int32_t (*ENC_Start)                (IFX_TAPI_LL_CH_t *pLLCh);

   /** Stops the coder in Upstream / record data
   \param pLLCh       Pointer to low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured
   */
   IFX_int32_t (*ENC_Stop)                 (IFX_TAPI_LL_CH_t *pLLCh);

   /**
      Put encoder into hold or unhold state.

      \param pLLCh       Pointer to Low-level channel structure
      \param nOnHold     Hold state (IFX_ENABLE - hold, IFX_DISABLE - unhold)

      \return  IFX_SUCCESS if successful
               IFX_ERROR if an error occured

      \remark
   */
   IFX_int32_t (*ENC_Hold)                 (IFX_TAPI_LL_CH_t *pLLChannel,
                                             IFX_operation_t nOnHold);

   /** Sets the Recording codec and frame length
   \param pLLCh      Pointer to low-level channel structure
   \param nCoder     Selected coder type
   \param nFrameLength    Length of frames to be generated by the coder
   \param nBitPack   normal or AAL bit alignment (not supported by all devices)
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured
   */
   IFX_int32_t (*ENC_Cfg)                  (IFX_TAPI_LL_CH_t *pLLChannel,
                                             IFX_TAPI_COD_TYPE_t nCoder,
                                             IFX_TAPI_COD_LENGTH_t nFrameLength,
                                             IFX_TAPI_COD_AAL2_BITPACK_t nBitPack);

   /** Sets Decoder specific parameters
   \param pLLCh      Pointer to low-level channel structure
   \param nBitPack   normal or AAL bit alignment (not supported by all devices)
   \param nPLC       Select PLC type: defined in codec or special pattern.
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured
   */
   IFX_int32_t (*DEC_Cfg)                  (IFX_TAPI_LL_CH_t *pLLChannel,
                                             IFX_TAPI_COD_AAL2_BITPACK_t nBitPack,
                                             IFX_TAPI_DEC_PLC_t nPLC);

   /** Turns the room noise detection mode on or off
   \param pLLCh           Pointer to Low-level channel structure
   \param bEnable         IFX_TRUE to enable or IFX_FALSE to disable
   \param nThreshold      detection level in minus dB
   \param nVoicePktCnt    count of consecutive voice packets required for event
   \param nSilencePktCnt  count of consecutive silence packets required for event */
   IFX_int32_t (*ENC_RoomNoise)            (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_boolean_t bEnable,
                                             IFX_uint32_t nThreshold,
                                             IFX_uint8_t nVoicePktCnt,
                                             IFX_uint8_t nSilencePktCnt);

   /** Starts the playing.
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*DEC_Start)                (IFX_TAPI_LL_CH_t *pLLCh);

   /** Stops the playing.
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*DEC_Stop)                 (IFX_TAPI_LL_CH_t *pLLCh);

   /** Sets the Voice Activity Detection mode
   \param pLLCh           Pointer to Low-level channel structure
   \param nVAD         Switch On or Off
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*VAD_Cfg)                  (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_int32_t nVAD);

   /**
      Enable/Disable the AGC resource inside the device

      \param pLLCh       Pointer to Low-level channel structure
      \param agcMode     AGC mode to be configured (enable/disable)

      \return  IFX_SUCCESS if successful
               IFX_ERROR if an error occured

      \remark
   */
   IFX_int32_t (*AGC_Enable)               (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_ENC_AGC_MODE_t agcMode);

   /**
      Enable/Disable the AGC resource inside the device

      \param pLLCh           Pointer to Low-level channel structure
      \param pAGC_Cfg        New AGC parameters
      \return  IFX_SUCCESS if successful
               IFX_ERROR if an error occured


      \remark
   */
   IFX_int32_t (*AGC_Cfg)                  (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_ENC_AGC_CFG_t *pAGC_Cfg);


   /** Configures the jitter buffer
   \param pLLCh           Pointer to Low-level channel structure
   \param pJbConf      Pointer to the Jitter buffer configuration
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*JB_Cfg)                   (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_JB_CFG_t const *pJbConf);

   /** Query the Jitter buffer statistics
   \param pLLCh           Pointer to Low-level channel structure
   \param pJbData      Pointer to the Jitter buffer data
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*JB_Stat_Get)              (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_JB_STATISTICS_t *pJbData);

   /** Reset the Jitter buffer statistics
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*JB_Stat_Reset)            (IFX_TAPI_LL_CH_t *pLLCh);

   /** Configure RTP and RTCP for a connection
   \param pLLCh           Pointer to Low-level channel structure
   \param pRtpConf        Pointer to IFX_TAPI_PKT_RTP_CFG_t, RTP configuraton
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*RTP_Cfg)                  (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_PKT_RTP_CFG_t const *pRtpConf);

   /** Configure a new payload type
   \param pLLCh           Pointer to Low-level channel structure
   \param pRtpPTConf      Pointer to RTP payload configuraton
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*RTP_PayloadTable_Cfg)     (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_PKT_RTP_PT_CFG_t const *pRtpPTConf);

   /** Start or stop generation of RTP event packets
   \param pLLCh           Pointer to Low-level channel structure
   \param nEvent          Event code as defined in RFC2833
   \param nStart          Start (true) or stop (false)
   \param nDuration       Duration of event in units of 10 ms (0 = forever)
   \param nVolume         Volume of event. Value range [0-63] corresponding
                          to 0 dB to -63 dB. Values outside of the allowed
                          range will activate firmware defaults.
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*RTP_EV_Generate)          (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint8_t nEvent,
                                             IFX_boolean_t bStart,
                                             IFX_uint8_t nDuration,
                                             IFX_int8_t nVolume);

   /** Gets the RTCP statistic information for the addressed channel
   \param pLLCh           Pointer to Low-level channel structure
   \param pRTCP           Pointer to RTCP Statistics structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*RTCP_Get)                 (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_PKT_RTCP_STATISTICS_t *pRTCP);

   /**  Resets  RTCP statistics
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*RTCP_Reset)               (IFX_TAPI_LL_CH_t *pLLCh);

   /* CURRENTLY NOT SUPPORTED
      the following pair of function pointers can be used to retrieve ("gather") the RTCP statistics non blocking,
      therefore RTCP_Prepare must be called within a noninterruptible context
    */
   IFX_int32_t (*RTCP_Prepare_Unprot)      (IFX_TAPI_LL_CH_t *pLLCh);
   IFX_int32_t (*RTCP_Prepared_Get)        (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_PKT_RTCP_STATISTICS_t *pRTCP);

   IFX_int32_t (*AAL_Cfg)                  (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_PCK_AAL_CFG_t const *pAalConf);
   IFX_int32_t (*AAL_Profile_Cfg)          (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_PCK_AAL_PROFILE_t const *pProfile);

   /*  Configures the Datapump for Modulation
   \param pLLCh           Pointer to Low-level channel structure
   \param pFaxMod         Pointer
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*T38_Mod_Enable)           (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_T38_MOD_DATA_t const *pFaxMod);

   /*  Configures the Datapump for Demodulation
   \param pLLCh           Pointer to Low-level channel structure
   \param pFaxMod         Pointer
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*T38_DeMod_Enable)         (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_T38_DEMOD_DATA_t const *pFaxDemod);

   /** Disables the Fax datapump
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*T38_Datapump_Disable)     (IFX_TAPI_LL_CH_t *pLLCh);

   /** Query the Fax Status
   \param pLLCh           Pointer to Low-level channel structure
   \param pFaxStatus      Pointer to T38 status structure where the status will
   be copied
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*T38_Status_Get)           (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_T38_STATUS_t *pFaxStatus);

   /** Set the Fax Status
   \param pLLCh           Pointer to Low-level channel structure
   \param status          The status which to be set
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*T38_Status_Set)           (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint8_t status);

   /** Set the Fax Error Status
   \param pLLCh           Pointer to Low-level channel structure
   \param error           The error status which to be set
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*T38_Error_Set)            (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint8_t error);

   /** Decoder Change event reporting request new decoder details
   \param pLLCh           pointer to low-level channel structure
   \param pDecDetails     pointer to a decoder details structure
   \return
      IFX_SUCCESS if successful
      IFX_ERROR if an error occured */
   IFX_int32_t (*DEC_Chg_Evt_Detail_Req)   (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_DEC_DETAILS_t *pDec);

   /** Switches on/off HP filter of decoder path
   \param pLLCh           Pointer to Low-level channel structure
   \param bHp             IFX_FALSE to switch HP off, IFX_TRUE to switch HP on
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*DEC_HP_Set)                   (IFX_TAPI_LL_CH_t *pLLCh, IFX_boolean_t bHp);

   /** Sets the COD interface volume
   \param pLLCh           Pointer to Low-level channel structure
   \param pVol            Pointer to the IFX_TAPI_PKT_VOLUME_t structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*Volume_Set)               (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_PKT_VOLUME_t const *pVol);

   /** Gets the supported T.38 fax channel implementation capabilities
   \param pLLDev          Pointer to Low-level device structure
   \param pCap            Pointer to the IFX_TAPI_T38_CAP_t structure
   \return
      VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*FAX_Cap_Get)               (IFX_TAPI_LL_DEV_t *pLLDev,
                                             IFX_TAPI_T38_CAP_t *pCap);

   /** Gets the parameters of T.38 fax channel
   \param pLLCh           Pointer to Low-level channel structure
   \param pCfg            Pointer to the IFX_TAPI_T38_FAX_CFG_t structure
   \return
   VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*FAX_Cfg_Get)               (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_T38_FAX_CFG_t *pCfg);

   /** Sets the parameters of T.38 fax channel
   \param pLLCh           Pointer to Low-level channel structure
   \param pCfg            Pointer to the IFX_TAPI_T38_FAX_CFG_t structure
   \return
   VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*FAX_Cfg_Set)               (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_T38_FAX_CFG_t const *pCfg);

   /** Gets the current configuration of the FAX Data Pump parameters
   \param pLLCh           Pointer to Low-level channel structure
   \param pFDPCfg         Pointer to the IFX_TAPI_T38_FDP_CFG_t structure
   \return
   VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*FAX_FDP_Cfg_Get)           (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_T38_FDP_CFG_t *pFDPCfg);

   /** Sets the FAX Data Pump parameters
   \param pLLCh           Pointer to Low-level channel structure
   \param pFDPCfg         Pointer to the IFX_TAPI_T38_FDP_CFG_t structure
   \return
   VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*FAX_FDP_Cfg_Set)           (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_T38_FDP_CFG_t const *pFDPCfg);

   /** starts a T.38 fax session on the given channel
   \param pLLCh           Pointer to Low-level channel structure
   \param pT38Cfg         Pointer to the IFX_TAPI_T38_SESS_CFG_t structure
   \return
   VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*FAX_Start)                 (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_T38_SESS_CFG_t const *pT38Cfg);

   /** Gets a T.38 fax session statistics
   \param pLLCh           Pointer to Low-level channel structure
   \param pStat           Pointer to IFX_TAPI_T38_SESS_STATISTICS_t structure
   \return
   VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*FAX_Stat_Get)              (IFX_TAPI_LL_CH_t *pLLCh,
                                        IFX_TAPI_T38_SESS_STATISTICS_t *pStat);

   /** stops currently running T.38 fax session on the given channel
   \param pLLCh           Pointer to Low-level channel structure
   \return
   VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*FAX_Stop)                  (IFX_TAPI_LL_CH_t *pLLCh);

   /** used to enable/disable a trace of FAX events
   \param pLLCh           Pointer to Low-level channel structure
   \param pTrace          Pointer to the IFX_TAPI_T38_TRACE_CFG_t structure
   \return
   VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*FAX_TraceSet)              (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_T38_TRACE_CFG_t *pTrace);

   /** Starts playing announcement
   \param pLLCh           Pointer to Low-level channel structure
   \param pStart          Pointer to IFX_TAPI_COD_ANNOUNCE_START_t structure
   \param pAnn            Pointer to the announcement data. TAPI makes sure,
                          that the data are stored in external DRAM (invalidate
                          caches, use uncached access) and provides the pointer
                          to the physical memory location (in kseg1) to the
                          firmware.
   \param nAnnSize        Size of payload of announcement.
   \return
   VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*Ann_Start)                 (IFX_TAPI_LL_CH_t *pLLCh,
                                        IFX_TAPI_COD_ANNOUNCE_START_t *pStart,
                                        IFX_char_t* pAnn, IFX_uint32_t nAnnSize);

   /** Stops playing announcment
   \param pLLCh           Pointer to Low-level channel structure
   \return
   VMMC_statusOk if successful else device specific return code.
    */
   IFX_int32_t (*Ann_Stop)                  (IFX_TAPI_LL_CH_t *pLLCh);

} IFX_TAPI_DRV_CTX_COD_t;

/*@}*/ /* CODER_MODULE*/

/** PCM Module */  /* ***************************************************** */
/** \addtogroup PCM_MODULE */
/** Used for PCM services higher layer */
/*@{*/
typedef struct
{
   /** Configure and enable the PCM interface
   \param pLLCh           Pointer to Low-level device structure
   \param pCfg            Pointer to the configuration structure
   */
   IFX_int32_t (*ifCfg)                    (IFX_TAPI_LL_DEV_t *pLLDev, const IFX_TAPI_PCM_IF_CFG_t *pCfg);

   /** Prepare parameters and call the target function to activate PCM module
   \param pLLCh           Pointer to Low-level channel structure
   \param nMode           Activation mode
   \param pPcmCfg         Pointer to the PCM configuration structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*Enable)                   (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint32_t nMode, IFX_TAPI_PCM_CFG_t *);

   /** Prepare parameters and call the target function to Configure the PCM module
   \param pLLCh           Pointer to Low-level channel structure
   \param pPcmCfg         Pointer to the PCM configuration structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*Cfg)                      (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_PCM_CFG_t const *pPCMConfig);

   /** Sets the LEC configuration on the PCM
   \param pLLCh           Pointer to Low-level channel structure
   \param pLecConf        Pointer to the LEC configuration structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Lec_Cfg)                  (IFX_TAPI_LL_CH_t *pLLCh,
                                             TAPI_LEC_DATA_t *pLecConf);

   /** Sets the PCM interface volume
   \param pLLCh           Pointer to Low-level channel structure
   \param pVol            Pointer to the IFX_TAPI_LINE_VOLUME_t structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Volume_Set)               (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_LINE_VOLUME_t const *pVol);

   /** Switches on/off HP filter of decoder path
   \param pLLCh           Pointer to Low-level channel structure
   \param bHp             IFX_FALSE to switch HP off, IFX_TRUE to switch HP on
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*DEC_HP_Set)                (IFX_TAPI_LL_CH_t *pLLCh, IFX_boolean_t bHp);

   /** Configure and activate the PCM channel with HDLC support
   \param pLLCh           Pointer to Low-level channel structure
   \param pHdlcCfg        Pointer to the HDLC channel configuration structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*HDLC_Cfg)                   (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_PCM_HDLC_CFG_t const *pHdlcCfg);

   /** Configure and activate loop for PCM channels
   \param pLLCh           Pointer to Low-level channel structure
   \param pLoopCfg        Pointer to the PCM Loop configuration structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*Loop)                       (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_PCM_LOOP_CFG_t const *pLoopCfg);

} IFX_TAPI_DRV_CTX_PCM_t;
/*@}*/ /* PCM_MODULE*/


/** SIG Module */ /* ********************************************************/
/** \addtogroup SIG_MODULE */
/** Signalling module services*/
/*@{*/
typedef struct
{
   /** Do low level UTG configuration and activation
   \param pLLCh         Pointer to Low-level channel structure
   \param pSimpleTone   Internal simple tone table entry
   \param dst           Destination
   \param nResID          Low level resource ID queried before by
                          ToneGen_ResIdGet. Otherwise TAPI defaults
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*UTG_Start)                (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_TONE_SIMPLE_t const *pSimpleTone,
                                             TAPI_TONE_DST dst,
                                             IFX_uint8_t nResID);

   /** Stop playing the tone with the given tone definition
   \param pLLCh         Pointer to Low-level channel structure
   \param nResID          Low level resource ID queried before by
                          ToneGen_ResIdGet. Otherwise TAPI defaults
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*UTG_Stop)                 (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint8_t nResID);

   IFX_return_t (*UTG_Level_Set)            (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_PREDEF_TONE_LEVEL_t const *pToneLevel);

   /** Returns the total number of UTGs per channel
   \param pChannel      Handle to TAPI_CHANNEL structure
   \return Returns the total number of UTGs per channel */
   IFX_uint8_t  (*UTG_Count_Get)            (IFX_TAPI_LL_CH_t *pLLCh);


   IFX_void_t   (*UTG_Event_Deactivated)    (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint8_t utgNum );

   /** Do low level UTG configuration and activation
   \param pLLCh           Pointer to Low-level channel structure
   Allocate a tone generation resource on a specific module "modType".

   \param pLLChannel  Handle to TAPI low level channel structure
   \param nResID The resouce identifier returned
   \param genType Describes the module type where to play the tone.
   \param genDir Tone generation direction on the module.

   \return
      IFX_SUCCESS if successful
      IFX_ERROR if an error occured
   \remarks
   nResID is returned and stored inside LL driver if necessary. It is used
   during tone event report to TAPI HL. TAPI HL uses nResID as an index for
   event dispatch functionality.
   This index is used for IFX_TAPI_LL_ToneStart, IFX_TAPI_LL_ToneStop
   and IFX_TAPI_LL_ToneGenResRelease.
   This API only supports to allocate a resource with genDir set to
   IFX_TAPI_LL_TONE_EXTERNAL or IFX_TAPI_LL_TONE_INTERNAL. It does not support
   to allocate a resource for both directions IFX_TAPI_LL_TONE_BOTH.
   TAPI LL disables all tone generator events in case the return value is
   not success.
   */
   IFX_int32_t (*ToneGen_ResIdGet)          (IFX_TAPI_LL_CH_t * pLLChannel,
                                             IFX_TAPI_MODULE_TYPE_t genType,
                                             IFX_TAPI_LL_TONE_DIR_t genDir,
                                             IFX_TAPI_TONE_RES_t* pRes);

   /** Request Details about the last RFC2833 event packet received for
       playout.
   \param pLLCh           Pointer to Low-level channel structure
   \param nEvent          Pointer where to store the event code
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*EPOU_Trig_Evt_Detail_Req)  (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint16_t *nEvent);

   /** Configure the DTMF tone generator
   \param nInterDigitTime Inter-digit-time in ms
   \param nDigitPlayTime  Active digit-play-time in ms
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*DTMFG_Cfg  )              (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint16_t nInterDigitTime,
                                             IFX_uint16_t nDigitPlayTime);

   /** Start the DTMF tone generator
   \param pLLCh           Pointer to Low-level channel structure
   \param nDigits         Number of digits in the data string to be sent
   \param *data           String with the digits (ascii 0-9 A-D) to be sent
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*DTMFG_Start)              (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint8_t nDigits,
                                             IFX_char_t  *data);

   /** Stop the DTMF tone generator
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*DTMFG_Stop)               (IFX_TAPI_LL_CH_t *pLLCh);

   /** Start the DTMF tone detector
   \param pLLCh           Pointer to Low-level channel structure
   \param direction       Direction from which to detect the DTMF signal
   \param enable_end_event Generate event when DTMF ends?
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*DTMFD_Start)              (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_LL_SIG_IN_DIR direction,
                                             IFX_boolean_t enableEndEvent);

   /** Stop the DTMF tone detector
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*DTMFD_Stop)               (IFX_TAPI_LL_CH_t *pLLCh);

   /** Controls the DTMF sending mode.
   \param pLLCh           Pointer to Low-level channel structure
   \param nOobMode        Mode of DTMFD (Inband or Out of band transmission
                          of RFC2833 event packets)
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*DTMFD_OOB)                 (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_PKT_EV_OOB_t nOobMode);

   /** Override control of the DTMF tone detector
   \param  pLLCh           Pointer to Low-level channel structure.
   \param  bOverride       IFX_TRUE: override, IFX_FALSE: normal operation.
   \param  bOperation      Enable or disable the DTMF detector.
   \param  direction       Direction from which to detect the DTMF signal.
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*DTMFD_Override)           (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_boolean_t bOverride,
                                             IFX_enDis_t bOperation,
                                             IFX_TAPI_LL_SIG_IN_DIR direction);

   /** Sets/Gets DTMF receiver coefficients.
   \param pLLCh           Pointer to Low-level channel structure
   \param bRW             IFX_FALSE to write, IFX_TRUE to read settings
   \param pDtmfRxCoeff    Pointer to DTMF Rx coefficients settings

   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*DTMF_RxCoeff)             (IFX_TAPI_LL_CH_t *pLLCh, IFX_boolean_t bRW, IFX_TAPI_DTMF_RX_CFG_t *pDtmfRxCoeff);

   /** Starts Call Progress Tone Detection
   \param pLLCh           Pointer to Low-level channel structure
   \param pTone           Pointer to simple tone structure
   \param signal          The type of signal and direction to detect
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*CPTD_Start)               (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_TONE_SIMPLE_t const *pTone, IFX_int32_t signal);

   /** Stops the Call Progress Tone Detection
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*CPTD_Stop)                (IFX_TAPI_LL_CH_t *pLLCh);

   /** Enables signal detection
   \param pLLCh           Pointer to Low-level channel structure
   \param pSig            Pointer to IFX_TAPI_SIG_DETECTION_t structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*DetectEnable)             (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_SIG_DETECTION_t const *pSig);

   /** Disables signal detection
   \param pLLCh           Pointer to Low-level channel structure
   \param pSig            Pointer to IFX_TAPI_SIG_DETECTION_t structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*DetectDisable)            (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_SIG_DETECTION_t const *pSig);

   /** Configures the transmission of tone signals detected by MFTD. Tones can
       be transmitted inband or out-of-band with RFC2833 RTP event packets.
   \param pLLCh           Pointer to Low-level channel structure
   \param nOobMode        Mode of signal transmission. (Inband or Out-of-band
                          transmission with RFC2833 event packets)
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*MFTD_OOB)                  (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_PKT_EV_OOB_t nOobMode);

   /** Start CID data transmission
   \param pLLChannel      Pointer to Low-level channel structure
   \param pCid            Contains the CID configuration and data to be sent
   \return
   Device specific return code.
   In case of success the return code must correspond to IFX_SUCCESS */
   IFX_int32_t (*CID_TX_Start)              (IFX_TAPI_LL_CH_t *pLLChannel,
                                             IFX_TAPI_CID_TX_t const *pCidData);

   /** Stop CID data transmission
   \param pLLChannel      Pointer to Low-level channel structure
   \return
   Device specific return code.
   In case of success the return code must correspond to IFX_SUCCESS */
   IFX_int32_t (*CID_TX_Stop)               (IFX_TAPI_LL_CH_t *pLLChannel);

   /** Start the CID receiver
   \param pLLChannel      Pointer to Low-level channel structure
   \param pCid            Contains the CID configuration and data to be sent
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*CID_RX_Start)              (IFX_TAPI_LL_CH_t *pLLChannel,
                                             IFX_TAPI_CID_RX_t const *pCidData);

   /** Stop the CID receiver
   \param pLLChannel      Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*CID_RX_Stop)               (IFX_TAPI_LL_CH_t *pLLChannel);
} IFX_TAPI_DRV_CTX_SIG_t;


/** AUDIO Module */ /* ******************************************************** */
typedef struct
{
   IFX_return_t  (*Volume_Set)            (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint32_t level);
   IFX_return_t  (*Room_Set)              (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint32_t type);
   IFX_return_t  (*Mode_Set)              (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint32_t mode);
   IFX_return_t  (*Mute_Set)              (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint32_t action);
   IFX_return_t  (*Ring_Start)            (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint32_t index);
   IFX_return_t  (*Ring_Stop)             (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint32_t index);
   IFX_return_t  (*Ring_Volume_Set)       (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint32_t level);
   IFX_return_t  (*Incall_Anouncement)    (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint32_t action);
   IFX_return_t  (*AFE_Cfg_Set)           (IFX_TAPI_LL_CH_t *pLLCh, IFX_TAPI_AUDIO_AFE_CFG_SET_t *pAFECfg);
   IFX_return_t  (*Test_Mode_Set)         (IFX_TAPI_LL_CH_t *pLLCh, IFX_uint32_t action);
} IFX_TAPI_DRV_CTX_AUDIO_t;


/** ALM Module */ /* ********************************************************/
/** \addtogroup ALM_MODULE */
/** Analog line module services*/
/*@{*/
typedef struct
{
   /** Set line type and sampling operation mode of the analog line.
   \param pLLCh         Pointer to Low-level channel structure
   \param nType         Line type and sampling mode to be set
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*Line_Type_Set)            (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_LINE_TYPE_t nType);

   /** Set the line mode of the analog line
   \param pLLCh         Pointer to Low-level channel structure
   \param nMode
   \param nTapiLineMode
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t   (*Line_Mode_Set)           (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_int32_t nMode,
                                             IFX_uint8_t nTapiLineMode);

   /** Switch Line Polarity
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t   (*Line_Polarity_Set)       (IFX_TAPI_LL_CH_t *pLLCh);


   IFX_int32_t  (*Volume_Set)              (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_LINE_VOLUME_t const *pVol);

   /** This service enables or disables a high level path of a phone channel.
   \param pLLCh           Pointer to Low-level channel structure
   \param bEnable         Enable or disable
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t   (*Volume_High_Level)       (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_int32_t bEnable);

   /** Ring configuration set
   \param pLLCh           Pointer to Low-level channel structure.
   \param pRingConfig     Pointer to ring config structure.
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*Ring_Cfg_Set)             (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_RING_CFG_t const *pRingConfig);

   /** Ring configuration get
   \param pLLCh           Pointer to Low-level channel structure.
   \param pRingConfig     Pointer to ring config structure which is to be
                          filled with data.
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*Ring_Cfg_Get)             (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_RING_CFG_t *pRingConfig);

   /** Enable/Disalbe Auto battery switch
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t   (*AutoBatterySwitch)       (IFX_TAPI_LL_CH_t *pLLCh);

   /** Send metering burst
   \param pLLCh           Pointer to Low-level channel structure
   \param nPulseNum       Number of pulses
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t   (*Metering_Start)          (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint32_t);


   /** Restores the line state back after fault
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t   (*FaultLine_Restore)       (IFX_TAPI_LL_CH_t *pLLCh);

   /** Sets the LEC configuration on the ALM
   \param pLLCh           Pointer to Low-level channel structure
   \param pLecConf        Pointer to LEC configuration structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*Lec_Cfg)                  (IFX_TAPI_LL_CH_t *pLLCh,
                                             TAPI_LEC_DATA_t *pLecConf);

   /** Starts playing out a tone on the ALM tone generator.
   \param pLLCh         Pointer to Low-level channel structure
   \param res           Resource number used for playing the tone.
   \param pToneSimple   Pointer to the tone definition to play
   \param dst           Destination where to play the tone
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*TG_Play)         (IFX_TAPI_LL_CH_t *pLLCh,
                                    IFX_uint8_t res,
                                    IFX_TAPI_TONE_SIMPLE_t const *pToneSimple,
                                    TAPI_TONE_DST dst);

   /** Stop playing the tone with the given tone definition
   \param pLLCh      Pointer to Low-level channel structure
   \param res        Resource number used for playing the tone
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*TG_Stop)                  (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint8_t res);

   /** Starts playing out the next tone of the simple tone definition.
   \param pLLCh         Pointer to Low-level channel structure
   \param pTone         Pointer to the current simple tone definition
   \param res           Resource number used for playing the tone
   \param nToneStep     Identifies the next tone step of the simple tone
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*TG_ToneStep)              (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_TONE_SIMPLE_t const *pTone,
                                             IFX_uint8_t res, IFX_uint8_t *nToneStep);

   /** Gets the activation status of the message waiting lamp.
   \param pLLChannel    Handle to TAPI low level channel structure
   \param pActivation   Handle to \ref IFX_TAPI_MWL_ACTIVATION_t structure
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*WML_Activation_Get)   (IFX_TAPI_LL_CH_t *pLLChannel,
                                        IFX_TAPI_MWL_ACTIVATION_t *pActivation);

   /** Activate/deactivates the message waiting lamp.
   \param pLLChannel    Handle to TAPI low level channel structure
   \param pActivation   Handle to \ref IFX_TAPI_MWL_ACTIVATION_t structure
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*WML_Activation_Set)   (IFX_TAPI_LL_CH_t *pLLChannel,
                                        IFX_TAPI_MWL_ACTIVATION_t const *pActivation);

   /** Simulate Hook generation (for debug use only)
   */
   IFX_int32_t  (*TestHookGen)              (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_boolean_t arg);

   /** ALM 8kHz test loop switch (for debug use only)
   */
   IFX_int32_t  (*TestLoop)                 (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_TEST_LOOP_t* pLoop);

   /** Start calibration process for analog channel
   */
   IFX_int32_t  (*Calibration_Start) (IFX_TAPI_LL_CH_t *pLLCh);

   /** Stop calibration process for analog channel
   */
   IFX_int32_t  (*Calibration_Stop)  (IFX_TAPI_LL_CH_t *pLLCh);

   /** Finish the calibration process on an analog channel
   */
   IFX_int32_t  (*Calibration_Finish) (IFX_TAPI_LL_CH_t *pLLCh);

   /** Read out the current calibration values of the analog channel
   */
   IFX_int32_t  (*Calibration_Get)   (IFX_TAPI_LL_CH_t *pLLChannel,
                                      IFX_TAPI_CALIBRATION_CFG_t *pClbConfig);

   /** Writes the calibration values of the analog channel
   */
   IFX_int32_t  (*Calibration_Set)   (IFX_TAPI_LL_CH_t *pLLChannel,
                                      IFX_TAPI_CALIBRATION_CFG_t *pClbConfig);

   /** Read out the calibration results of the analog channel
   */
   IFX_int32_t  (*Calibration_Results_Get) (IFX_TAPI_LL_CH_t *pLLChannel,
                                        IFX_TAPI_CALIBRATION_CFG_t *pClbConfig);
   /** Start selected subset (or all) GR909 tests
   */
   IFX_int32_t  (*GR909_Start) (IFX_TAPI_LL_CH_t *pLLChannel,
                                        IFX_TAPI_GR909_START_t *pGR909Start);
   /** Stop GR909 tests
   */
   IFX_int32_t  (*GR909_Stop) (IFX_TAPI_LL_CH_t *pLLChannel);
   /** Read GR909 results
   */
   IFX_int32_t  (*GR909_Result) (IFX_TAPI_LL_CH_t *pLLChannel,
                                        IFX_TAPI_GR909_RESULT_t *pGR909Result);
   /** Request continuous measurement results
   */
   IFX_int32_t  (*ContMeasReq)  (IFX_TAPI_LL_CH_t *pLLChannel);
   /** Reset continous measurement results
   */
   IFX_int32_t  (*ContMeasReset)(IFX_TAPI_LL_CH_t *pLLChannel);
   /** Return the stored continuous measurement results
   */
   IFX_int32_t  (*ContMeasGet)  (IFX_TAPI_LL_CH_t *pLLChannel,
                                    IFX_TAPI_CONTMEASUREMENT_GET_t *pContMeas);
   /** Set the new hook state on FXO line
   */
   IFX_int32_t  (*FXO_HookSet)  (IFX_TAPI_LL_CH_t *pLLChannel,
                                             IFX_TAPI_FXO_HOOK_t hook);
   /** Get current hook state for FXO line
   */
   IFX_int32_t  (*FXO_HookGet)  (IFX_TAPI_LL_CH_t *pLLChannel,
                                          IFX_TAPI_FXO_HOOK_t *p_hook);
   /** Issue a hook flash on FXO line
   */
   IFX_int32_t  (*FXO_FlashSet) (IFX_TAPI_LL_CH_t *pLLChannel);
   /** Configure a hook flash time
   */
   IFX_int32_t  (*FXO_FlashCfg) (IFX_TAPI_LL_CH_t *pLLChannel,
                                    IFX_TAPI_FXO_FLASH_CFG_t *p_fhCfg);
   /** Configure maximum detection time for OSI
   */
   IFX_int32_t  (*FXO_OsiCfg)   (IFX_TAPI_LL_CH_t *pLLChannel,
                                     IFX_TAPI_FXO_OSI_CFG_t *p_osiCfg);
   /** Get FXO event status from LL driver
   */
   IFX_int32_t  (*FXO_StatGet)  (IFX_TAPI_LL_CH_t *pLLChannel,
                                     IFX_uint32_t bit_pos, IFX_enDis_t *enable);
   /** Set FXO line mode
   */
   IFX_int32_t  (*FXO_LineModeSet)  (IFX_TAPI_LL_CH_t *pLLChannel,
                                     IFX_TAPI_FXO_LINE_MODE_t *pMode);
   /** Start an analog line capacitance measurement session
   */
   IFX_int32_t  (*CapMeasStart)     (IFX_TAPI_LL_CH_t *pLLChannel);
   /** Start an analog line capacitance measurement session
       (internal API, not available on application side)
   */
   IFX_int32_t  (*CapMeasStartInt)  (IFX_TAPI_LL_CH_t *pLLChannel);
   /** Stop any running analog line capacitance measurement session
   */
   IFX_int32_t  (*CapMeasStop)      (IFX_TAPI_LL_CH_t *pLLChannel);
   /** Get capacitance measurement result
   */
   IFX_int32_t  (*CapMeasResult)    (IFX_TAPI_LL_CH_t *pLLChannel);
} IFX_TAPI_DRV_CTX_ALM_t;
/*@}*/

/** DECT Module */  /* **************************************************** */
/** \addtogroup DECT_MODULE */
/** Used for DECT services higher layer */
/*@{*/
typedef struct
{
   /** Sets the encoder and decoder start delay
   \param pLLCh      Pointer to low-level channel structure
   \param nEncDelay  Delay from the start of the decoder to the start of
                     the encoder in steps of 2.5ms. Range 0ms - 10ms.
   \param nDecDelay  Delay from the arrival of the first packet to the start
                     of the decoder in steps of 2.5ms. Range 0ms - 10ms.
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Ch_Cfg)                   (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint8_t nEncDelay,
                                             IFX_uint8_t nDecDelay);

   /** Sets the coder type and frame length for the DECT encoding path
   \param pLLCh      Pointer to low-level channel structure
   \param nCoder     Selected coder type
   \param nFrameLength  length of packets to be generated by the coder
   \return
      LL return code
   */
   IFX_int32_t (*ENC_Cfg)                   (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_DECT_ENC_TYPE_t nCoder,
                                             IFX_TAPI_DECT_ENC_LENGTH_t nFrameLength);

   /** Prepare parameters and call the target function to activate DECT module
   \param pLLCh      Pointer to low-level channel structure
   \param nEnable    Enable or disable the module
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*Enable)                    (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint8_t nEnable);

   /** Sets the gains for the DECT en-/decoding path.
   \param pLLCh      Pointer to low-level channel structure
   \param pVol       Pointer to IFX_TAPI_LINE_VOLUME_t structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Gain_Set)                 (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_PKT_VOLUME_t const *pVol);

   /** Get the statistic data from the DECT coder channel
   \param pLLCh      Pointer to low-level channel structure
   \param pStatistic pointer to struct where to store the statistic data
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Statistic)                (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_DECT_STATISTICS_t *pStatistic);

   /** Do low level UTG configuration and activation
   \param pLLCh         Pointer to Low-level channel structure
   \param pSimpleTone   Internal simple tone table entry
   \param dst           Destination (unused)
   \param utgNum        UTG number  (always 0)
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*UTG_Start)                (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_TONE_SIMPLE_t const *pSimpleTone,
                                             TAPI_TONE_DST dst,
                                             IFX_uint8_t res);

   /** Stop playing the tone with the given tone definition
   \param pLLCh         Pointer to Low-level channel structure
   \param res           Resource number used for playing the tone (always 0)
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t  (*UTG_Stop)                 (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_uint8_t res);

   /** Do low level DECT echo canceller configuration
   \param pLLCh      Pointer to low-level channel structure
   \param pEC_cfg    Pointer to \ref IFX_TAPI_DECT_EC_CFG_t structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_int32_t (*EC_Cfg)                    (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_DECT_EC_CFG_t *pEC_cfg);
} IFX_TAPI_DRV_CTX_DECT_t;
/*@}*/ /* DECT_MODULE*/


/** CONnection Module */
/** \addtogroup CON_MODULE */
/** Connection Module services*/
/*@{*/
typedef struct
{
   /** Adds a connection between a data channel and any of these modules:
       PCM, ALM, DECT, AUDIO.
   \param pLLCh           Pointer to Low-level channel structure
   \param pMap            Pointer to IFX_TAPI_MAP_DATA_t structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Data_Channel_Add)         (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_MAP_DATA_t const *pMap);

   /** Removes a connection between a data channel and any of these modules:
       PCM, ALM, DECT, AUDIO.
   \param pLLCh           Pointer to Low-level channel structure
   \param pMap            Pointer to IFX_TAPI_MAP_DATA_t structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Data_Channel_Remove)      (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_MAP_DATA_t const *pMap);

   /** Adds a connection between any two of the following modules:
       PCM, ALM, DECT, AUDIO.
   \param pLLCh           Pointer to Low-level channel structure
   \param nType1          Module type in the first channel
   \param nCh2            Channel number of second channel
   \param nType2          Module type in the second channel
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Module_Connect)           (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_MAP_TYPE_t nType1,
                                             IFX_uint8_t nCh2,
                                             IFX_TAPI_MAP_TYPE_t nType2);

   /** Removes a connection between any two of the following modules:
       PCM, ALM, DECT, AUDIO.
   \param pLLCh           Pointer to Low-level channel structure
   \param nType1          Module type in the first channel
   \param nCh2            Channel number of second channel
   \param nType2          Module type in the second channel
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Module_Disconnect)        (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_MAP_TYPE_t nType1,
                                             IFX_uint8_t nCh2,
                                             IFX_TAPI_MAP_TYPE_t nType2);

   /** Mute/Unmute all connections to modules which are attached to the given
      data channel except the first module connected to the local side.
   \param pLLCh           Pointer to Low-level channel structure
   \param nMute           IFX_TRUE: mute / IFX_FALSE: unmute
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Data_Channel_Mute)        (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_boolean_t nMute);

   /** Find the module which gets the input from the data channel and provides
       it's input to the data channel.
   \param pLLCh           Pointer to Low-level channel structure
   \param pTapiCh         Returns pointer to the TAPI channel the found module
                          belongs to or IFX_NULL if no module is connected.
   \param pModType        Returns of which type the found module is.
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Data_Channel_Find_Connected_Module)
                                            (IFX_TAPI_LL_CH_t *pLLCh,
                                             TAPI_CHANNEL **pTapiCh,
                                             IFX_TAPI_MAP_TYPE_t *pModType);

   /** Find the data channel which gets it's main input from a given module
       and which provides it's input to the given module.

       There can be more than one data channel connected to a module.

   \param pLLCh           Pointer to Low-level channel structure
   \param pModType        Returns of which type the found module is.
   \param pTapiCh         On input specifies which was the last channel found
                          by this function. For the first call use IFX_NULL.
                          For iteration calls use the last output.
                          Returns pointer to the TAPI channel the found module
                          belongs to or IFX_NULL if no module is connected.
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*Module_Find_Connected_Data_Channel)
                                            (IFX_TAPI_LL_CH_t *pLLCh,
                                             IFX_TAPI_MAP_TYPE_t pModType,
                                             TAPI_CHANNEL **pTapiCh);

} IFX_TAPI_DRV_CTX_CON_t;
/*@}*/

/** Polling Interface */ /* ****************************************************/
/** \addtogroup POLL_INTERFACE */
/** List of required low-level polling services */
/*@{*/
typedef struct
{
   /** Read voice/fax/event packets from the device
   \param pLLDev           Pointer to low-level device structure
   \param ppPkts           Array of free buffer pointers
   \param pPktsNum         On entry identifies the number of packets to be read,
                           on return it contains the number of packets read
   \return
   IFX_SUCCESS on success
   IFX_ERROR on error */
   IFX_return_t (*rdPkts) (IFX_TAPI_LL_DEV_t *pLLDev,
                           IFX_void_t **ppPkts,
                           IFX_int32_t *pPktsNum,
                           IFX_int32_t nDevID);

   /** Write voice/fax/event packets available
   \param pLLDev           Pointer to low-level device structure
   \param ppPkts           Array of packet buffer pointers to be written
   \param pPktsNum         On entry identifies the number of packets to be written,
                           on return it contains the number of packets successfully
                           written. On success (IFX_SUCCESS) all packets have been
                           successfully written.
   \return
   IFX_SUCCESS on success
   IFX_ERROR on error */
   IFX_return_t (*wrPkts) (IFX_TAPI_LL_DEV_t *pLLDev, IFX_int32_t *pPktsNum);

   /** Updates the low-level TAPI device status by reading the hardware status
       registers and taking the appropriate actions upon status change.
       Typically this function executes the device's ISR.

   \param pLLDev           Pointer to low-level device structure
   \return
   IFX_SUCCESS on success
   IFX_ERROR on error */
   IFX_return_t (*pollEvents) (IFX_TAPI_LL_DEV_t *pLLDev);

   /** Used to control the packet-generation related interrupts. In case a
       device is registerred for packet polling it is necessary to disable
       the related interrupts so as to prohibit any unwanted overhead of
       switching to interrupt context.

   \param pLLDev           Pointer to low-level device structure
   \param bEnable          IFX_TRUE to enable, IFX_FALSE to disable the
                           related interrupts
   \return
   IFX_SUCCESS on success
   IFX_ERROR on error */
   IFX_return_t (*pktsIRQCtrl) (IFX_TAPI_LL_DEV_t *pLLDev,
                                IFX_boolean_t bEnable);

   /** Used to control the TAPI event-generation related interrupts. In case a
       device is registerred for events polling it is necessary to disable
       the related interrupts so as to prohibit any unwanted overhead of
       switching to interrupt context.

   \param pLLDev           Pointer to low-level device structure
   \param bEnable          IFX_TRUE to enable, IFX_FALSE to disable the
                           related interrupts
   \return
   IFX_SUCCESS on success
   IFX_ERROR on error */
   IFX_return_t (*evtsIRQCtrl) (IFX_TAPI_LL_DEV_t *pLLDev,
                                IFX_boolean_t bEnable);

} IFX_TAPI_DRV_CTX_POLL_t;
/*@}*/

typedef struct
{
   /** Starts LINear channel upstream packet generation.
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*ENC_Start)                (IFX_TAPI_LL_CH_t *pLLCh);

   /** Stops LINear channel upstream packet generation.
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*ENC_Stop)                 (IFX_TAPI_LL_CH_t *pLLCh);

   /** Starts LINear channel downstream playing.
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*DEC_Start)                (IFX_TAPI_LL_CH_t *pLLCh);

   /** Stops LINear channel downstream playing.
   \param pLLCh           Pointer to Low-level channel structure
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured */
   IFX_return_t (*DEC_Stop)                 (IFX_TAPI_LL_CH_t *pLLCh);

} IFX_TAPI_DRV_CTX_LIN_t;

/* driver context data structure */
/** \addtogroup TAPI_LL_INTERFACE */
/** Interface between High-Level TAPI and Low-Level TAPI */
/*@{*/
typedef struct
{
   /** high-level and low-level interface API version, keep as first element */
   IFX_char_t                              *hlLLInterfaceVersion;
   /** device nodes prefix (if DEVFS is used) /dev/\<devNodeName\>\<number\> */
   IFX_char_t                              *devNodeName;
   /** driverName */
   IFX_char_t                              *drvName;
   /** driverVersion */
   IFX_char_t                              *drvVersion;

   IFX_uint16_t                             majorNumber;
   IFX_uint16_t                             minorBase;
   IFX_uint16_t                             maxDevs;
   IFX_uint16_t                             maxChannels;

   /* The following two prepare functions receive a pointer to the
      HL device and channel structures and return pointer to the
      LL device and channel structures(!) */
   IFX_TAPI_LL_DEV_t* (*Prepare_Dev)        (TAPI_DEV *pTapiDev,
                                             IFX_uint32_t devNum);
   IFX_int32_t   (*Init_Dev)                (IFX_TAPI_LL_DEV_t *pDev);
   IFX_void_t    (*Exit_Dev)                (IFX_TAPI_LL_DEV_t *pDev,
                                             IFX_boolean_t bChipAccess);

   IFX_TAPI_LL_CH_t*  (*Prepare_Ch)         (TAPI_CHANNEL *pTapiCh,
                                             IFX_TAPI_LL_DEV_t *pLLDev,
                                             IFX_uint32_t chNum);
   IFX_int32_t   (*Init_Ch)                 (IFX_TAPI_LL_CH_t *pCh);

   IFX_int32_t   (*FW_Start)                (IFX_TAPI_LL_DEV_t *pLLDev,
                                             IFX_void_t const *pProc);
   IFX_int32_t   (*FW_Init)                 (IFX_TAPI_LL_DEV_t *pLLDev,
                                             IFX_uint8_t nMode);
   IFX_int32_t   (*BBD_Dnld)                (IFX_TAPI_LL_DEV_t *pLLDev,
                                             IFX_void_t const *pProc);

   /* */
   IFX_int32_t   (*Pwr_Save_Dev)            (IFX_TAPI_LL_DEV_t *pDev);

   /* capabilities */
   IFX_uint32_t  (*CAP_Number_Get)          (IFX_TAPI_LL_DEV_t *pDev);
   IFX_return_t  (*CAP_List_Get)            (IFX_TAPI_LL_DEV_t *pDev,
                                             IFX_TAPI_CAP_t *pCapList);
   IFX_int32_t   (*CAP_Check)               (IFX_TAPI_LL_DEV_t *pDev,
                                             IFX_TAPI_CAP_t *pCapList);

   /* */
   IFX_return_t  (*GetCmdMbxSize)           (IFX_TAPI_LL_DEV_t *pDev,
                                             IFX_uint8_t *cmdmbx_size);

   /* called when TAPI channel is opened or closed */
   IFX_int32_t   (*Open)                    (IFX_TAPI_LL_CH_t *pLLChDev);
   IFX_int32_t   (*Release)                 (IFX_TAPI_LL_CH_t *pLLChDev);

   /* forward standard ioctl to low level driver */
   IFX_int32_t   (*Ioctl)                   (IFX_TAPI_LL_CH_t *pCh,
                                             IFX_uint32_t nCmd,
                                             IFX_ulong_t ioarg);

   /* write a packet downstream */
   IFX_int32_t   (*Write)                   (IFX_TAPI_LL_CH_t *pCh,
                                             const IFX_char_t *buf,
                                             IFX_int32_t, IFX_int32_t*,
                                             IFX_TAPI_STREAM_t);
   /* amount of bytes HL must reserve before the data in downstream packets */
   IFX_uint32_t                             pktBufPrependSpace;
   /* shows that LL has a packet read routine and needs an upstream fifo */
   IFX_boolean_t                            bProvidePktRead;
   /* for each stream the number of channels that provide packet read */
   IFX_uint16_t                             readChannels[IFX_TAPI_STREAM_MAX];

   /* Coder related functions for the HL TAPI */
   IFX_TAPI_DRV_CTX_COD_t                   COD;

   /* PCM related functions for the HL TAPI */
   IFX_TAPI_DRV_CTX_PCM_t                   PCM;

   /* Signalling Module related functions for HL TAPI */
   IFX_TAPI_DRV_CTX_SIG_t                   SIG;

   /* Analog Line Module related functions for HL TAPI */
   IFX_TAPI_DRV_CTX_ALM_t                   ALM;

   /* DECT Module related functions for HL TAPI */
   IFX_TAPI_DRV_CTX_DECT_t                  DECT;

   /* Connection related functions for the HL TAPI */
   IFX_TAPI_DRV_CTX_CON_t                   CON;

   /* Protection andn Interrupt module functions for the HL TAPI */
   IFX_TAPI_DRV_CTX_IRQ_t                   IRQ;

   /* Audio Module related functions for HL TAPI */
   IFX_TAPI_DRV_CTX_AUDIO_t                 AUDIO;

   /* Polling interface related LL routines to be used by HL TAPI */
   IFX_TAPI_DRV_CTX_POLL_t                  POLL;

   /* Linear channel interface related LL routines to be used by HL TAPI */
   IFX_TAPI_DRV_CTX_LIN_t                   LIN;

   /** array of pTapiDev pointers associated with this driver context */
   TAPI_DEV                                *pTapiDev;

   IFX_int32_t (*Dbg_CErr_Handler)          (IFX_TAPI_LL_DEV_t *pLLDev,
                                             IFX_TAPI_DBG_CERR_t *pData);

} IFX_TAPI_DRV_CTX_t;
/*@}*/

#ifndef DRV_TAPI_H
struct _TAPI_DEV
{
   /* channel number IFX_TAPI_DEVICE_CH_NUMBER indicates the control device */
   /* ATTENTION, nChannel must be the first element */
   IFX_uint8_t               nChannel;
   /* TODO: required modularization, access  to low level device
            should be a pointer to void -> see list of AI */
   IFX_TAPI_LL_DEV_t        *pLLDev;
};

struct _TAPI_CHANNEL
{
   /* channel number */
   /* ATTENTION, nChannel must be the first element */
   IFX_uint8_t                   nChannel;
   /* pointer to the Low level driver channel */
   IFX_TAPI_LL_CH_t             *pLLChannel;
   /* pointer to the tapi device structure */
   TAPI_DEV                     *pTapiDevice;
   /**\todo To be removed */
#ifdef TAPI_PACKET
   /* semaphore used only in blocking read access,
      in this case given from interrupt context */
   TAPI_OS_event_t               semReadBlock;

   /* wakeup queue for select on read */
   TAPI_OS_drvSelectQueue_t      wqRead;
   /* wakeup queue for select on write */
   TAPI_OS_drvSelectQueue_t      wqWrite;
   /* stores the current fax status */
   volatile IFX_boolean_t        bFaxDataRequest;
   /* stores the current linear channel status */
   volatile IFX_boolean_t        bLinChannelActive;
   /* stores the current linear channel data request status */
   volatile IFX_boolean_t        bLinDataRequest;
   /* flags for different purposes, see CH_FLAGS */
   IFX_uint32_t                  nFlags;
#endif /* TAPI_PACKET */
};
#endif /* DRV_TAPI_H */

typedef struct IFX_TAPI_DRV_CTX_t TAPI_LOW_LEVEL_DRV_CTX_t;

/* Registration function for the Low Level TAPI driver */
extern IFX_int32_t  IFX_TAPI_Register_LL_Drv   (IFX_TAPI_DRV_CTX_t*);
extern IFX_int32_t  IFX_TAPI_Unregister_LL_Drv (IFX_int32_t majorNumber);
extern IFX_void_t   IFX_TAPI_DeviceReset       (TAPI_DEV *pTapiDev);
extern IFX_void_t   IFX_TAPI_ReportResources   (TAPI_DEV *pTapiDev,
                                                IFX_TAPI_RESOURCE *pResources);

#ifdef TAPI_PACKET
extern IFX_return_t  IFX_TAPI_UpStreamFifo_Reset(TAPI_CHANNEL* pTapiCh,
                                                 IFX_TAPI_STREAM_t nStream);
extern IFX_return_t  IFX_TAPI_UpStreamFifo_Put(TAPI_CHANNEL* pTapiCh,
                                               IFX_TAPI_STREAM_t nStream,
                                               const IFX_void_t * const pData,
                                               const IFX_uint32_t nLength,
                                               const IFX_uint32_t nOffset);
extern FIFO_ID*      IFX_TAPI_DownStreamFifo_Handle_Get(TAPI_DEV* pTapiDev);
#endif /* TAPI_PACKET */

extern IFX_return_t IFX_TAPI_Event_Dispatch      (TAPI_CHANNEL *,
                                                  IFX_TAPI_EVENT_t *);
extern IFX_void_t   TAPI_Tone_Set_Source         (TAPI_CHANNEL *,
                                                  IFX_uint8_t res,
                                                  IFX_int32_t src);
extern IFX_void_t TAPI_Cid_Abort                 (TAPI_CHANNEL *pChannel);
extern IFX_boolean_t TAPI_Cid_IsActive           (TAPI_CHANNEL *pChannel);

extern TAPI_CMPLX_TONE_STATE_t TAPI_ToneState    (TAPI_CHANNEL *pChannel,
                                                  IFX_uint8_t res);

/** Put packet received in the irq handler into the egress fifo */
extern IFX_return_t irq_IFX_TAPI_KPI_PutToEgress(TAPI_CHANNEL *pChannel,
                                                 IFX_TAPI_KPI_STREAM_t stream,
                                                 IFX_void_t *pPacket,
                                                 IFX_uint32_t nPacketLength);

/** Retrieve the KPI Channel number of a given stream on a given TAPI Channel */
extern IFX_TAPI_KPI_CH_t IFX_TAPI_KPI_ChGet     (TAPI_CHANNEL *pChannel,
                                                 IFX_TAPI_KPI_STREAM_t stream);

/** Raise handler for KPI ingress direction */
extern IFX_void_t IFX_TAPI_KPI_ScheduleIngressHandling (IFX_void_t);

/* Packet path statistic */
extern IFX_void_t  IFX_TAPI_Stat_Add            (TAPI_CHANNEL *pChannel,
                                                 IFX_TAPI_STREAM_t stream,
                                                 TAPI_STAT_COUNTER_t counter,
                                                 IFX_uint32_t value);

/* Update the SLIC FXO flag in TAPI HL device context;
   the flag is required for special handling of FXO ioctls on systems
   with 3-channel SLIC and Clare LITELINK PLI */
extern IFX_void_t IFX_TAPI_Update_SlicFxo       (TAPI_DEV *pTapiDev,
                                                 IFX_boolean_t bVal);

#endif /* _DRV_TAPI_LL_INTERFACE_H */
