#!/sbin/sh
#
###########################################
#
# Copyright (C) 2020 FlameGApps Project
#
# This file is part of the FlameGApps Project by ayandebnath <ayandebnath000@gmail.com>
#
# The FlameGApps scripts are free software, you can redistribute it and/or modify it
# 
# These scripts are distributed in the hope that they will be useful, but WITHOUT ANY WARRANTY
#
###########################################
# File Name    : installer.sh
###########################################
##

# List of the addon files
addon_apps_list="
Gmail"

# Pre-installed unnecessary app list
remove_list="
app/Email
app/Email2
app/Gmail
app/GmailPrebuilt
app/PrebuiltGmail"

##

ui_print() {
  echo "ui_print $1
    ui_print" >> $OUTFD
}

set_progress() { echo "set_progress $1" >> $OUTFD; }

is_mounted() { mount | grep -q " $1 "; }

setup_mountpoint() {
  test -L $1 && mv -f $1 ${1}_link
  if [ ! -d $1 ]; then
    rm -f $1
    mkdir $1
  fi
}

recovery_actions() {
  OLD_LD_LIB=$LD_LIBRARY_PATH
  OLD_LD_PRE=$LD_PRELOAD
  OLD_LD_CFG=$LD_CONFIG_FILE
  unset LD_LIBRARY_PATH
  unset LD_PRELOAD
  unset LD_CONFIG_FILE
}

recovery_cleanup() {
  [ -z $OLD_LD_LIB ] || export LD_LIBRARY_PATH=$OLD_LD_LIB
  [ -z $OLD_LD_PRE ] || export LD_PRELOAD=$OLD_LD_PRE
  [ -z $OLD_LD_CFG ] || export LD_CONFIG_FILE=$OLD_LD_CFG
}

clean_up() {
  rm -rf /tmp/flamegapps
  rm -rf /tmp/config.prop
  rm -rf /tmp/flame.prop
  rm -rf /tmp/addon.d.sh
  rm -rf /tmp/tar_gapps
  rm -rf /tmp/unzip_dir
}

path_info() {
  ls / > $root_info
  ls -R $SYSTEM > $system_info
  ls -R $SYSTEM/product > $product_info 2>/dev/null
  ls -R $SYSTEM/system_ext > $system_ext_info 2>/dev/null
}

space_before() {
  df -h > $log_dir/space_before.txt
}

space_after() {
  df -h > $log_dir/space_after.txt
}

take_logs() {
  ui_print " "
  ui_print "- Copying logs to /sdcard & $zip_dir"
  cp -f $TMP/recovery.log $log_dir/recovery.log
  cd $log_dir
  tar -cz -f "$TMP/flamegapps_addon_debug_logs.tar.gz" *
  cp -f $TMP/flamegapps_addon_debug_logs.tar.gz $zip_dir/flamegapps_addon_debug_${flame_addon}_logs.tar.gz
  cp -f $TMP/flamegapps_addon_debug_logs.tar.gz /sdcard/flamegapps_addon_debug_${flame_addon}_logs.tar.gz
  cd /
  rm -rf $TMP/flamegapps_addon_debug_logs.tar.gz
}

get_file_prop() {
  grep -m1 "^$2=" "$1" | cut -d= -f2
}

get_prop() {
  #check known .prop files using get_file_prop
  for f in $PROPFILES; do
    if [ -e "$f" ]; then
      prop="$(get_file_prop "$f" "$1")"
      if [ -n "$prop" ]; then
        break #if an entry has been found, break out of the loop
      fi
    fi
  done
  #if prop is still empty; try to use recovery's built-in getprop method; otherwise output current result
  if [ -z "$prop" ]; then
    getprop "$1" | cut -c1-
  else
    printf "$prop"
  fi
}

abort() {
  sleep 1
  ui_print "- Aborting..."
  sleep 3
  path_info
  unmount_all
  take_logs
  clean_up
  recovery_cleanup
  exit 1;
}

exit_all() {
  sleep 0.5
  path_info
  space_after
  unmount_all
  sleep 0.5
  set_progress 0.90
  take_logs
  clean_up
  recovery_cleanup
  sleep 0.5
  ui_print " "
  ui_print "- Installation Successful..!"
  ui_print " "
  set_progress 1.00
  exit 0;
}

mount_apex() {
  # For reference, check: https://github.com/osm0sis/AnyKernel3/blob/master/META-INF/com/google/android/update-binary
  if [ -d $SYSTEM/apex ]; then
    local apex dest loop minorx num
    setup_mountpoint /apex
    test -e /dev/block/loop1 && minorx=$(ls -l /dev/block/loop1 | awk '{ print $6 }') || minorx=1
    num=0
    for apex in $SYSTEM/apex/*; do
      dest=/apex/$(basename $apex .apex)
      test "$dest" = /apex/com.android.runtime.release && dest=/apex/com.android.runtime
      mkdir -p $dest
      case $apex in
        *.apex)
          unzip -qo $apex apex_payload.img -d /apex
          mv -f /apex/apex_payload.img $dest.img
          mount -t ext4 -o ro,noatime $dest.img $dest 2>/dev/null
          if [ $? != 0 ]; then
            while [ $num -lt 64 ]; do
              loop=/dev/block/loop$num
              (mknod $loop b 7 $((num * minorx))
              losetup $loop $dest.img) 2>/dev/null
              num=$((num + 1))
              losetup $loop | grep -q $dest.img && break
            done
            mount -t ext4 -o ro,loop,noatime $loop $dest
            if [ $? != 0 ]; then
              losetup -d $loop 2>/dev/null
            fi
          fi
        ;;
        *) mount -o bind $apex $dest;;
      esac
    done
    export ANDROID_RUNTIME_ROOT=/apex/com.android.runtime
    export ANDROID_TZDATA_ROOT=/apex/com.android.tzdata
    export BOOTCLASSPATH=/apex/com.android.runtime/javalib/core-oj.jar:/apex/com.android.runtime/javalib/core-libart.jar:/apex/com.android.runtime/javalib/okhttp.jar:/apex/com.android.runtime/javalib/bouncycastle.jar:/apex/com.android.runtime/javalib/apache-xml.jar:/system/framework/framework.jar:/system/framework/ext.jar:/system/framework/telephony-common.jar:/system/framework/voip-common.jar:/system/framework/ims-common.jar:/system/framework/android.test.base.jar:/apex/com.android.conscrypt/javalib/conscrypt.jar:/apex/com.android.media/javalib/updatable-media.jar
  fi
}

unmount_apex() {
  if [ -d $SYSTEM/apex ]; then
    local dest loop
    for dest in $(find /apex -type d -mindepth 1 -maxdepth 1); do
      if [ -f $dest.img ]; then
        loop=$(mount | grep $dest | cut -d" " -f1)
      fi
      (umount -l $dest
      losetup -d $loop) 2>/dev/null
    done
    rm -rf /apex 2>/dev/null
    unset ANDROID_RUNTIME_ROOT ANDROID_TZDATA_ROOT BOOTCLASSPATH
  fi
}

mount_all() {
  set_progress 0.10
  ui_print "- Mounting partitions"
  sleep 1
  dynamic_partitions=`getprop ro.boot.dynamic_partitions`
  SLOT=`getprop ro.boot.slot_suffix`
  [ ! -z "$SLOT" ] && ui_print "- Current boot slot: $SLOT"

  if [ -n "$(cat /etc/fstab | grep /system_root)" ]; then
    MOUNT_POINT=/system_root
  else
    MOUNT_POINT=/system
  fi

  for p in "/cache" "/data" "$MOUNT_POINT" "/product" "/system_ext" "/vendor"; do
    if [ -d "$p" ] && grep -q "$p" "/etc/fstab" && ! is_mounted "$p"; then
      mount "$p"
    fi
  done

  if [ "$dynamic_partitions" = "true" ]; then
    ui_print "- Dynamic partition detected"
    for m in "/system" "/system_root" "/product" "/system_ext" "/vendor"; do
      (umount "$m"
      umount -l "$m") 2>/dev/null
    done
    mount -o ro -t auto /dev/block/mapper/system$SLOT /system_root
    mount -o ro -t auto /dev/block/mapper/vendor$SLOT /vendor 2>/dev/null
    mount -o ro -t auto /dev/block/mapper/product$SLOT /product 2>/dev/null
    mount -o ro -t auto /dev/block/mapper/system_ext$SLOT /system_ext 2>/dev/null
  else
    mount -o ro -t auto /dev/block/bootdevice/by-name/system$SLOT $MOUNT_POINT 2>/dev/null
  fi

  if [ "$dynamic_partitions" = "true" ]; then
    for block in system vendor product system_ext; do
      for slot in "" _a _b; do
        blockdev --setrw /dev/block/mapper/$block$slot 2>/dev/null
      done
    done
    mount -o rw,remount -t auto /dev/block/mapper/system$SLOT /system_root
    mount -o rw,remount -t auto /dev/block/mapper/vendor$SLOT /vendor 2>/dev/null
    mount -o rw,remount -t auto /dev/block/mapper/product$SLOT /product 2>/dev/null
    mount -o rw,remount -t auto /dev/block/mapper/system_ext$SLOT /system_ext 2>/dev/null
  else
    mount -o rw,remount -t auto $MOUNT_POINT
    mount -o rw,remount -t auto /vendor 2>/dev/null
    mount -o rw,remount -t auto /product 2>/dev/null
    mount -o rw,remount -t auto /system_ext 2>/dev/null
  fi

  sleep 0.3

  if is_mounted /system_root; then
    ui_print "- Device is system-as-root"
    if [ -f /system_root/build.prop ]; then
      mount -o bind /system_root /system
      SYSTEM=/system_root
      ui_print "- System is $SYSTEM"
    else
      mount -o bind /system_root/system /system
      SYSTEM=/system_root/system
      ui_print "- System is $SYSTEM"
    fi
  elif is_mounted /system; then
    if [ -f /system/build.prop ]; then
      SYSTEM=/system
      ui_print "- System is $SYSTEM"
    elif [ -f /system/system/build.prop ]; then
      ui_print "- Device is system-as-root"
      mkdir -p /system_root
      mount --move /system /system_root
      mount -o bind /system_root/system /system
      SYSTEM=/system_root/system
      ui_print "- System is /system/system"
    fi
  else
    ui_print "- Failed to mount/detect system"
    abort
  fi
  mount_apex
}

unmount_all() {
  unmount_apex
  ui_print " "
  ui_print "- Unmounting partitions"
  for m in "/system" "/system_root" "/product" "/system_ext" "/vendor"; do
    if [ -e $m ]; then
      (umount $m
      umount -l $m) 2>/dev/null
    fi
  done
}

mount -o bind /dev/urandom /dev/random
unmount_all
mount_all

recovery_actions

PROPFILES="$SYSTEM/build.prop $TMP/flame.prop"
GAPPS_DIR="$TMP/tar_gapps"
UNZIP_FOLDER="$TMP/unzip_dir"
EX_SYSTEM="$UNZIP_FOLDER/system"
zip_dir="$(dirname "$ZIPFILE")"
log_dir="$TMP/flamegapps/logs"
flame_log="$log_dir/installation_log.txt"
build_info="$log_dir/build_info.prop"
root_info="$log_dir/rootpathinfo.txt"
system_info="$log_dir/systempathinfo.txt"
product_info="$log_dir/productpathinfo.txt"
system_ext_info="$log_dir/system_extpathinfo.txt"
mkdir -p $UNZIP_FOLDER
mkdir -p $log_dir
space_before

# Get ROM, device & package information
flame_android=`get_prop ro.flame.android`
flame_sdk=`get_prop ro.flame.sdk`
flame_arch=`get_prop ro.flame.arch`
flame_addon=`get_prop ro.flame.addon`
rom_version=`get_prop ro.build.version.release`
rom_sdk=`get_prop ro.build.version.sdk`
device_architecture=`get_prop ro.product.cpu.abilist`
device_code=`get_prop ro.product.device`

if [ -z "$device_architecture" ]; then
  device_architecture=`get_prop ro.product.cpu.abi`
fi

case "$device_architecture" in
  *x86_64*) arch="x86_64"
    ;;
  *x86*) arch="x86"
    ;;
  *arm64*) arch="arm64"
    ;;
  *armeabi*) arch="arm"
    ;;
  *) arch="unknown"
    ;;
esac

echo ------------------------------------------------------------------- >> $flame_log
(echo "  --------------- FlameGApps Installation Logs ---------------"
echo "- Mount Point: $MOUNT_POINT"
echo "- Current slot: $SLOT"
echo "- Dynamic partition: $dynamic_partitions"
echo "- Flame version: $flame_android"
echo "- Flame SDK: $flame_sdk"
echo "- Flame ARCH: $flame_arch"
echo "- ROM version: $rom_version"
echo "- ROM SDK: $rom_sdk"
echo "- Device ARCH: $device_architecture ($arch)"
echo "- Device code: $device_code") >> $flame_log

# Get prop details before compatibility checks
cat $SYSTEM/build.prop > $build_info
cat $TMP/flame.prop >> $build_info

set_progress 0.20
sleep 1
ui_print " "
ui_print "- Android: $rom_version, SDK: $rom_sdk, ARCH: $arch"
sleep 1

if [ "$flame_sdk" = "all" ]; then
  if [ $rom_sdk -lt 29 ]; then
    ui_print " "
    ui_print "****************** WARNING *******************"
    ui_print " "
    ui_print "! Wrong android version detected"
    sleep 0.5
    ui_print "This package is for android: $flame_android only"
    sleep 0.5
    ui_print "Your ROM is Android: $rom_version"
    sleep 0.5
    ui_print " "
    ui_print "******* FlameGApps Installation Failed *******"
    ui_print " "
    abort
  fi
else
  if [ ! "$rom_sdk" = "$flame_sdk" ]; then
    ui_print " "
    ui_print "****************** WARNING *******************"
    ui_print " "
    ui_print "! Wrong android version detected"
    sleep 0.5
    ui_print "This package is for android: $flame_android only"
    sleep 0.5
    ui_print "Your ROM is Android: $rom_version"
    sleep 0.5
    ui_print " "
    ui_print "******* FlameGApps Installation Failed *******"
    ui_print " "
    abort
  fi
fi

if [ ! "$arch" = "$flame_arch" ]; then
  ui_print " "
  ui_print "****************** WARNING *******************"
  ui_print " "
  ui_print "! Wrong device architecture detected"
  sleep 0.5
  ui_print "This package is for device: $flame_arch only"
  sleep 0.5
  ui_print "Your device is: $arch"
  sleep 0.5
  ui_print " "
  ui_print "******* FlameGApps Installation Failed *******"
  ui_print " "
  abort
fi

if [ ! -f $SYSTEM/etc/flame.prop ]; then
  ui_print " "
  ui_print "****************** WARNING *******************"
  ui_print " "
  sleep 0.5
  ui_print "! Please install latest FlameGApps Package before this addon package !"
  sleep 3
  ui_print " "
  ui_print "******* FlameGApps Installation Failed *******"
  ui_print " "
  abort
fi

# Remove pre-installed unnecessary system apps
ui_print " "
ui_print "- Removing unnecessary system apps"
set_progress 0.30
sleep 0.5
for f in $remove_list; do
  rm -rf $SYSTEM/$f
  rm -rf $SYSTEM/product/$f
  if [ $rom_sdk -gt 29 ]; then
    rm -rf $SYSTEM/system_ext/$f
  fi
done

# remove_line <file> <line match string> by osm0sis @xda-developers
remove_line() {
  if grep -q "$2" $1; then
    local line=$(grep -n "$2" $1 | head -n1 | cut -d: -f1);
    sed -i "${line}d" $1;
  fi
}

install_gapps() {
  ui_print " "
  set_progress 0.70
  gapps_list="$addon_apps_list"
  for g in $gapps_list; do
    ui_print "- Installing $g"
    unzip -o "$ZIPFILE" "tar_gapps/$g.tar.xz" -d $TMP
    tar -xf "$GAPPS_DIR/$g.tar.xz" -C $UNZIP_FOLDER
    rm -rf $GAPPS_DIR/$g.tar.xz
    file_list="$(find "$EX_SYSTEM/" -mindepth 1 -type f | cut -d/ -f5-)"
    dir_list="$(find "$EX_SYSTEM/" -mindepth 1 -type d | cut -d/ -f5-)"
    for file in $file_list; do
      install -D "$EX_SYSTEM/${file}" "$SYSTEM/${file}"
      chcon -h u:object_r:system_file:s0 "$SYSTEM/${file}"
      chmod 0644 "$SYSTEM/${file}"
    done
    for dir in $dir_list; do
      chcon -h u:object_r:system_file:s0 "$SYSTEM/${dir}"
      chmod 0755 "$SYSTEM/${dir}"
    done
    rm -rf $UNZIP_FOLDER/*
  done
}

# Install gapps files
echo -e "\n- Installing gapps files" >> $flame_log
install_gapps >> $flame_log

echo -e "\n                 Installation Finished            " >> $flame_log
echo ----------------------------------------------------------------- >> $flame_log

sleep 0.5
set_progress 0.80
ui_print " "
ui_print "- Performing other tasks"

# Install addon.d script
if [ -d $SYSTEM/addon.d ]; then
  if [ "$flame_addon" = "Maps" ]; then
    echo -e "\n- Installing backup script for Maps" >> $flame_log
    rm -rf $SYSTEM/addon.d/*Maps.sh
    cp -f $TMP/addon.d.sh $SYSTEM/addon.d/79-flame-Maps.sh
    chcon -h u:object_r:system_file:s0 "$SYSTEM/addon.d/79-flame-Maps.sh"
    chmod 0755 $SYSTEM/addon.d/79-flame-Maps.sh
  elif [ "$flame_addon" = "Gboard" ]; then
    local gapps_edition=`get_file_prop $SYSTEM/etc/flame.prop "ro.flame.edition"`
    if [ ! "$gapps_edition" = "full" ]; then
      echo -e "\n- Installing backup script for Gboard" >> $flame_log
      rm -rf $SYSTEM/addon.d/*Gboard.sh
      cp -f $TMP/addon.d.sh $SYSTEM/addon.d/78-flame-Gboard.sh
      chcon -h u:object_r:system_file:s0 "$SYSTEM/addon.d/78-flame-Gboard.sh"
      chmod 0755 $SYSTEM/addon.d/78-flame-Gboard.sh
    else
      echo -e "\n- Full edition gapps detected. Skipping backup script for Gboard" >> $flame_log
    fi
  elif [ "$flame_addon" = "Gmail" ]; then
    echo -e "\n- Installing backup script for Gmail" >> $flame_log
    rm -rf $SYSTEM/addon.d/*Gmail.sh
    cp -f $TMP/addon.d.sh $SYSTEM/addon.d/77-flame-Gmail.sh
    chcon -h u:object_r:system_file:s0 "$SYSTEM/addon.d/77-flame-Gmail.sh"
    chmod 0755 $SYSTEM/addon.d/77-flame-Gmail.sh
  elif [ "$flame_addon" = "Drive" ]; then
    echo -e "\n- Installing backup script for GDrive" >> $flame_log
    rm -rf $SYSTEM/addon.d/*Drive.sh
    cp -f $TMP/addon.d.sh $SYSTEM/addon.d/76-flame-Drive.sh
    chcon -h u:object_r:system_file:s0 "$SYSTEM/addon.d/76-flame-Drive.sh"
    chmod 0755 $SYSTEM/addon.d/76-flame-Drive.sh
  elif [ "$flame_addon" = "GoogleWebView" ]; then
    echo -e "\n- Installing backup script for GoogleWebView" >> $flame_log
    rm -rf $SYSTEM/addon.d/*GoogleWebView.sh
    if [ -f $SYSTEM/addon.d/*flame-Chrome.sh ]; then
      remove_line $TMP/addon.d.sh "app/TrichromeLibrary/TrichromeLibrary.apk"
    fi
    cp -f $TMP/addon.d.sh $SYSTEM/addon.d/75-flame-GoogleWebView.sh
    chcon -h u:object_r:system_file:s0 "$SYSTEM/addon.d/75-flame-GoogleWebView.sh"
    chmod 0755 $SYSTEM/addon.d/75-flame-GoogleWebView.sh
  elif [ "$flame_addon" = "Chrome" ]; then
    echo -e "\n- Installing backup script for Chrome" >> $flame_log
    rm -rf $SYSTEM/addon.d/*Chrome.sh
    if [ -f $SYSTEM/addon.d/*flame-GoogleWebView.sh ]; then
      remove_line $TMP/addon.d.sh "app/TrichromeLibrary/TrichromeLibrary.apk"
    fi
    cp -f $TMP/addon.d.sh $SYSTEM/addon.d/74-flame-Chrome.sh
    chcon -h u:object_r:system_file:s0 "$SYSTEM/addon.d/74-flame-Chrome.sh"
    chmod 0755 $SYSTEM/addon.d/74-flame-Chrome.sh
  elif [ "$flame_addon" = "PixelLauncher-DPS" ]; then
    echo -e "\n- Installing backup script for PixelLauncher-DPS" >> $flame_log
    rm -rf $SYSTEM/addon.d/*flame-DPS.sh
    rm -rf $SYSTEM/addon.d/*PixelLauncher*
    cp -f $TMP/addon.d.sh $SYSTEM/addon.d/73-flame-PixelLauncher-DPS.sh
    chcon -h u:object_r:system_file:s0 "$SYSTEM/addon.d/73-flame-PixelLauncher-DPS.sh"
    chmod 0755 $SYSTEM/addon.d/73-flame-PixelLauncher-DPS.sh
  elif [ "$flame_addon" = "DPS" ]; then
    if [ ! -e $SYSTEM/addon.d/*flame-PixelLauncher-DPS.sh ]; then
      echo -e "\n- Installing backup script for DPS" >> $flame_log
      rm -rf $SYSTEM/addon.d/*flame-DPS.sh
      cp -f $TMP/addon.d.sh $SYSTEM/addon.d/72-flame-DPS.sh
      chcon -h u:object_r:system_file:s0 "$SYSTEM/addon.d/72-flame-DPS.sh"
      chmod 0755 $SYSTEM/addon.d/72-flame-DPS.sh
    else
      echo -e "\n- Installed PixelLauncher-DPS addon detected. Skipping backup script for DPS" >> $flame_log
    fi
  elif [ "$flame_addon" = "GoogleSearch-Assistant" ]; then
    echo -e "\n- Installing backup script for GoogleSearch-Assistant" >> $flame_log
    rm -rf $SYSTEM/addon.d/*GoogleSearch*
    cp -f $TMP/addon.d.sh $SYSTEM/addon.d/71-flame-GoogleSearch-Assistant.sh
    chcon -h u:object_r:system_file:s0 "$SYSTEM/addon.d/71-flame-GoogleSearch-Assistant.sh"
    chmod 0755 $SYSTEM/addon.d/71-flame-GoogleSearch-Assistant.sh
  fi
else
  echo -e "\n- $SYSTEM/addon.d dir not present \n" >> $flame_log
fi

exit_all
