#ifndef LISPFUNCTIONBASE_H
#define LISPFUNCTIONBASE_H

#include "dsString.h"
#include "dsInterfacePtr.h"
#include "dsDocument_c.h"
#include "dsApplication_c.h"
#include "dsLispFunction_c.h"

// Derive lisp functions you want to add to your add-in from this class,
// defining the abstract methods groupName(), functionName() and the lisp
// function body ExecuteFunctionNotify().
// 
// You should also consider redefining description().
//
// This abstract base class contains only dsAPI code and is thus
// independent from this project. Hence you can use it, as is, in other
// add-in projects as well.
class LispFunctionBase
{
public:
    LispFunctionBase(dsApplication_c *application) : application(application) {}
    virtual ~LispFunctionBase() { unregisterFunction(); }

    // The groupName is used to group a set of lisp functions within an
    // add-in. Usually there is only one group name and that is the
    // add-in's UUID.
    virtual const dsString groupName() const = 0;

    // The functionName is, as it says, the name of the function. It has to
    // be unique and we'd recommend the name does contain diacritics or
    // exotic characters.
    virtual const dsString functionName() const = 0;

    // This method is the function body. It will be
    // called when the function was invoked.
    virtual bool ExecuteFunctionNotify() = 0;

    // Give an explanation of what the function
    // does and also an example of how to use it.
    virtual const dsString description() const { return L""; }

    // Creates the function in a given document.
    dsCreateCommandError_e createFunction(dsDocument_c *document);

    // Registers the function, so that its body ExecuteFunctionNotify()
    // will be invoked upon calling this function.
    // NOTE: createFunction() has to be invoked beforehand.
    void registerFunction();

    // Unregisters the function, i.e. its body will not be invoked anymore.
    void unregisterFunction();

    // Removes the function from the given document.
    DSRESULT removeFunction(dsDocument_c *document);

    DeclareLispFunctionExecuteFunctionNotifyHook

protected:
    dsApplication_c *application;
    dsLispFunction_ptr m_function;
};

#endif // LISPFUNCTIONBASE_H
