const EventEmitter = require('events');
const {
  clearListeners,
  broadcastError,
  CONSTANTS
} = require('../communication');
const invoke = require('./invoke');
const { startParentWatchdog } = require('../watchdog');

const EXECUTION_PROCESS_NAME = 'PostmanExecution';

const callbacks = {
  onStartup: new Set([]),
  onExit: new Set([]),
  onIPCConnection: new Set([]),
  onIPCDisconnection: new Set([])
};

// This is a global object that is used to store all the lifecycle callbacks. This will be registered by system.js
// inside the global scope.
const globals = {
  lifecycle: callbacks
};

/**
 * Executes all onStartup lifecycle callbacks
 */
const startup = async () => {
  for (const callback of callbacks.onStartup) {
    await callback();
    callbacks.onStartup.delete(callback);
  }
};

/**
 * Executes all onExit lifecycle callbacks
 */
const exit = async () => {
  for (const callback of callbacks.onExit) {
    await callback();
    callbacks.onExit.delete(callback);
  }

  process.exit(0);
};

/**
 * Adds a onStartup lifecycle callback
 * @param callback
 */
function onStartup (callback) {
  callbacks.onStartup.add(callback);
}

/**
 * Adds a onExit lifecycle callback
 * @param callback
 */
function onExit (callback) {
  callbacks.onExit.add(callback);
}

/**
 * To be called when the IPC connection is established
 * @private
 * @returns {Promise<void>}
 */
const _handleIPCConnection = async () => {
  for (const callback of callbacks.onIPCConnection) {
    pm.logger.info('Invoking onIPCConnection callback', callback.name);

    await callback();

    callbacks.onIPCConnection.delete(callback);
  }
};

/**
 * To be called when the IPC connection is lost
 * @returns {Promise<void>}
 * @private
 */
const _handleIPCDisconnection = async () => {
  for (const callback of callbacks.onIPCDisconnection) {
    await callback();
    callbacks.onIPCDisconnection.delete(callback);
  }
};

/**
 * Attaches all lifecycle hooks on an object
 * @param object - The object that is a sub-system that has lifecycle hooks
 */
function registerHooks (object) {
  for (const stageHook of Object.keys(globals.lifecycle)) {
    if (typeof object?.[stageHook] === 'function') {
      globals.lifecycle[stageHook].add(object[stageHook]);
    }
  }
}

/**
 * Registers all process listeners
 */
function registerProcessListeners () {
  process.on('unhandledRejection', (reason, p) => {
    pm.logger.error('Unhandled Rejection at: Promise', p, 'reason:', reason);
  });

  process.on('SIGABRT', () => {
    pm.logger.error('PerformanceTestExecutionProcess~SIGABRT');
  });

  process.on('SIGINT', () => {
    pm.logger.error('PerformanceTestExecutionProcess~SIGINT');
  });

  process.on('SIGTERM', () => {
    pm.logger.error('PerformanceTestExecutionProcess~SIGTERM');
  });

  process.on('SIGKILL', () => {
    pm.logger.error('PerformanceTestExecutionProcess~SIGKILL');
  });

  process.on('SIGALRM', () => {
    pm.logger.error('PerformanceTestExecutionProcess~SIGALRM');
  });

  process.on('SIGBREAK', () => {
    pm.logger.error('PerformanceTestExecutionProcess~SIGBREAK');
  });

  process.on('SIGUSR1', () => {
    pm.logger.error('PerformanceTestExecutionProcess~SIGUSR1');
  });

  process.on('SIGUSR2', () => {
    pm.logger.error('PerformanceTestExecutionProcess~SIGUSR2');
  });

  process.on('SIGQUIT', () => {
    pm.logger.error('PerformanceTestExecutionProcess~SIGQUIT');
  });

  process.on('SIGTRAP', () => {
    pm.logger.error('PerformanceTestExecutionProcess~SIGTRAP');
  });

  process.on('uncaughtException', (err) => {
    pm.logger.error('Uncaught Exception thrown', err);
  });
}

const onIPCConnection = (callback) => {
  callbacks.onIPCConnection.add(callback);
};

const onIPCDisconnection = (callback) => {
  callbacks.onIPCDisconnection.add(callback);
};

/**
 * Returns the configuration options passed to the execution process
 * @returns {*}
 */
function getConfigurationOptions () {
  return globals.lifecycle.options;
}

/**
 * Registers all IPC handles
 * @returns {Promise<void>}
 */
function registerIPCHandles () {
  if (global.SUBSYSTEM_REGISTER.ipcHandlesRegistered) {
    pm.logger.info('PerformanceTestExecutionProcess ~ IPC handles already registered');
    return;
  }

  global.SUBSYSTEM_REGISTER.ipcHandlesRegistered = true;

  pm.logger.info('PerformanceTestExecutionProcess ~ Registering IPC handles');

  pm.sdk.ipc.handleOnce('exit', async () => {
    pm.logger.info('PerformanceTestExecutionProcess ~ exit');
    await exit();
  });

  // Called by main once the app boots up and the IPC is ready for communication.
  // This handle is called only once to set up the initial steps as soon as the execution process is spawned by main.
  pm.sdk.ipc.handleOnce('ready', async (_, options) => {
    globals.lifecycle.options = options;

    try {
      await startup();
    } catch (err) {
      broadcastError({
        error: err,
        message: 'Error in starting up the PerformanceTestSubSystemsManager',
        source: 'PerformanceTestSubSystemsManager',
        subsystem: 'system.startup'
      });
    }

    pm.logger.info('PerformanceTestExecutionProcess ~ ready', options);
  });

  // Register handlers for all the subsystems which are defined in singletons and factories.
  for (const subsystemId of [
    ...Object.keys(global.SUBSYSTEM_REGISTER.factories),
    ...Object.keys(global.SUBSYSTEM_REGISTER.singletons)
  ]) {
    pm.sdk.ipc.handle(subsystemId, (_, { action, args, method }) => {
      return invoke(subsystemId, { action, args, method }, false);
    });
  }
}

/**
 * Create an event bus if it doesn't exist.
 *
 * @returns {void}
 */
function init () {
  try {
    if (global.SUBSYSTEM_REGISTER.isInitialized) {
      return;
    }

    // Set the process title
    process.title = EXECUTION_PROCESS_NAME;

    pm.logger.info('PerformanceTestExecutionProcess~system.lifecycle.init~cwd', process.cwd());

    // Start a watchdog to kill all child processes if the parent dies
    startParentWatchdog(async () => {
      pm.logger.info('Parent died. Killing process.');

      try {
        // Initiate a graceful shutdown
        await exit();
      }
      finally {
        // 9 => SIGKILL
        process.exit(9);
      }
    });

    if (!global.eventBus) {
      global.eventBus = new EventEmitter();
    }

    pm.sdk.ipc.onReady(() => {
      registerIPCHandles();
      _handleIPCConnection();
    });

    pm.sdk.ipc.onClose(() => {
      pm.logger.info('PerformanceTestExecutionProcess ~ IPC is closed.');

      _handleIPCDisconnection();
      clearListeners();
    });

    global.SUBSYSTEM_REGISTER.isInitialized = true;

    registerProcessListeners();
  }
  catch (err) {
    pm.logger.error('Something unexpected happened while boot.', err);

    // process.exit(1);
  }
}

module.exports = {
  global: globals,
  exports: {
    // Manual hooks
    exit,

    // Lifecycle callback registration
    onStartup,
    onExit,
    onIPCConnection,
    onIPCDisconnection,
    getConfigurationOptions,
  },
  registerHooks,
  init
};
