/* eslint-disable no-console */
// Console.log and Console.error of this file are piped to pm.logger.info and pm.logger.error of the execution process
// This is because, we were missing logs from this thread as pm object is not available within this scope.

const os = require('os');
const path = require('path');
const _ = require('lodash');

const { parentPort, workerData } = require('worker_threads');
const postmanRuntime = require('postman-runtime');
const postmanCollectionSdk = require('postman-collection');
const PostmanFs = require('../../../../common/utils/postmanFs');

/**
 * Removes references of all functions nested inside an Object to return a plain javascript object.
 *
 * @param {Object} obj - Object to be iterated
 */
const removeFunctionalReferences = (obj) => {
  if (!obj || typeof obj !== 'object') {
    return;
  }

  Object.keys(obj).forEach((k) => {
    if (typeof obj?.[k] === 'function')
      delete obj[k];
    else if (typeof obj?.[k] === 'object')
      removeFunctionalReferences(obj?.[k]);
  });
};

/**
 * Setting up a request timeout of 55 seconds which is large enough to handle the largest possible response time while
 * still being small enough to timeout in case of smallest performance test duration.
 */
const REQUEST_TIMEOUT = 55 * 1000;
const pick = (obj, ...keys) =>
  (obj ? Object.fromEntries(keys.filter((key) => key in obj).map((key) => [key, obj?.[key]])) : {});

const MIN_ITERATION = 20,
  ITERATION_MULTIPLIER = 10,
  MIN_SLEEP_TIMEOUT = 500,
  SLEEP_TIMEOUT_MULTIPLIER = 1000;

/**
 * Send an event to the execution process.
 *
 * @param {Number} executorReferenceId - Process reference number
 * @param {object} executionContext- execution context
 * @param {string} event - Event name
 * @param {object} [data] - Data to send to the execution process
 *
 * @returns {void}
 */
function _sendToParent (executorReferenceId, executionContext, event, data) {
  removeFunctionalReferences(data);

  parentPort.postMessage({
    type: 'data',
    data: {
      executorReferenceId,
      event,
      data,
      executionContext,
      timestamp: new Date().toISOString()
    }
  });
}

/**
 * Sanitizes options to be sent to runtime. Mostly converting objects into SDK instances.
 *
 * @param {Object} rawOptions
 */
function sanitizeRunOptions (options) {
  const rawOptions = _.cloneDeep(options);

  if (!rawOptions) {
    return;
  }

  if (!rawOptions.requester) {
    rawOptions.requester = {};
  }

  if (!rawOptions.requester.authorizer) {
    rawOptions.requester.authorizer = {};
  }

  if (rawOptions.proxies) {
    rawOptions.proxies = new postmanCollectionSdk.ProxyConfigList({}, rawOptions.proxies);
  }

  rawOptions.certificates = new postmanCollectionSdk.CertificateList({}, rawOptions.certificates);

  return rawOptions;
}

/**
 * Spawns up runtime instances and runs the collection. This function is called by the execution process to start or add
 * scenarios for execution. It returns a promise that resolves when all the runners have finished.
 * The promise will reject if any of the runners fail to start.
 * The results are the same as the ones returned by the `postman-runtime` module's summary parameter of the done callback.
 * Each event emitted by the runners is sent to the execution process.
 *
 * @param {string} collection - Collection object to run
 * @param {string} environment - Environment object to run
 * @param {object} globals - Global variables to run
 * @param {string[]} requestSelection - Array of runnable item selection to run
 * @param {Number} count - Number of scenarios to execute the collection with
 * @param {object} executionContext - execution context
 * @param {Number} executorReferenceId - Process reference number
 *
 * @returns {Promise<unknown[]>}
 */
function runRunners ({
                       collection,
                       environment,
                       globals,
                       requestSelection,
                       count,
                       executionContext,
                       executorReferenceId,
                       runOptions: rawRunOptions
                     }) {
  const sdkCollection = new postmanCollectionSdk.Collection(collection),
    collectionVariables = { values: collection.variable || [] },
    environmentObject = environment,
    globalsObject = globals,
    defaultWorkingDir = path.join(os.homedir(), 'Postman', 'files');

  const runOptions = sanitizeRunOptions(rawRunOptions);

  if (runOptions.fileResolver && !!PostmanFs) {
    let { workingDir, insecureFileRead, fileWhitelist } = runOptions.fileResolver;

    _.set(runOptions, 'fileResolver', new PostmanFs(workingDir || defaultWorkingDir, insecureFileRead, fileWhitelist));
  }

  // Create a group of runners and run the collection with them in parallel (each runner will run the collection
  // with a single user)
  return Promise.all(
    Array.from({ length: count }, () => {
      // Create a clone of sendToMainProcess function that is bound to the executionContext
      const sendToMainProcess = _sendToParent.bind(
        null,
        executorReferenceId,
        executionContext
      );

      return new Promise(async (resolve, reject) => {
        const runner = new postmanRuntime.Runner(),

          // Random iteration count value (within range(20,30)) to ensure all scenarios does not die at the same time.
          // This is done to avoid a dip in RPS at a certain interval.
          iterationCount = Math.floor(MIN_ITERATION + (ITERATION_MULTIPLIER * Math.random()));

        await runner.run(
          sdkCollection,
          {
            ...runOptions,
            timeout: {
              request: REQUEST_TIMEOUT
            },
            environment: environmentObject,
            globals: globalsObject,
            entrypoint: requestSelection ?? undefined,
            iterationCount
          },
          function (err, run) {
            if (err) {
              sendToMainProcess('errorStartingRun', err);
              return reject(err);
            }

            run.start({
              // Called when the run begins
              start: function (err) {
                sendToMainProcess('start', { err });
              },

              // Called once with response for each request in a collection
              response: function (err, cursor, response, request, item) {
                sendToMainProcess('response', {
                  err: pick(err || {}, 'code', 'message'),
                  response: pick(response || {}, 'code', 'responseSize', 'status', 'responseTime'),
                  item
                });
              },

              beforeIteration: function () {
                // Reset all the variables to the values they had during the start of the run.
                // Event though we use iterations to improve performance, each scenario is considered independent run.
                // Thus updated variable values should not propagate across iterations.
                // TODO: [APITEST-322] This is not the recommended way to achieve this. Expose an option/interface from postman-runtime
                delete run.state.environment;
                delete run.state.globals;
                delete run.state.collectionVariables;
                delete run.state._variables;

                Object.assign(run.state, {
                  environment: new postmanCollectionSdk.VariableScope(environmentObject),
                  globals: new postmanCollectionSdk.VariableScope(globalsObject),
                  collectionVariables: new postmanCollectionSdk.VariableScope(collectionVariables),
                  _variables: new postmanCollectionSdk.VariableScope({})
                });
              },

              iteration: function (err, cursor) {
                // Using random sleep time using pause and resume between iterations to improve CPU performance
                if (cursor.iteration <= iterationCount - 2) {
                  run.pause();
                  setTimeout(() => {
                    run.resume();
                  }, (MIN_SLEEP_TIMEOUT + (Math.random() * SLEEP_TIMEOUT_MULTIPLIER))); // Random sleep time between 500ms to 1500ms.
                }
              },

              // Called at the end of a run
              done: function (err) {
                sendToMainProcess('done', { err });
                resolve({ executionContext });
              },
            });
          }
        );
      });
    })
  );
}

/**
 * Handle messages from the execution process. The exec. process sends a message to this thread when it wants to start
 * the execution of a scenario instance for a performance test or add more scenarios to the test.
 */
parentPort.on('message', async (message) => {
  if (message.type === 'assign') {
    try {
      console.log(`Received event ${message.type}`);

      await runRunners(message.payload);

      _sendToParent(
        message.payload.executorReferenceId,
        message.payload.executionContext,
        'allRunFinished'
      );
    }
    catch (err) {
      console.error('Error while running collection', err);

      _sendToParent(
        message.payload.executorReferenceId,
        message.payload.executionContext,
        'error',
        { err }
      );
    }
  }
});
