const { Worker } = require('worker_threads');
const EventEmitter = require('events');
const path = require('path');
const scenarioTestingProcessPath = path.join(__dirname, '_thread.js');
const { log } = require('../../lib/logging');

/**
 * Thin wrapper around the worker thread. This will be responsible for spawning the worker thread and handling the
 * events from the worker thread. It will also be responsible for sending the data to the main process.
 * @class WorkerThread
 * @extends EventEmitter
 *
 * @private
 *
 * @param {Number} id - ID of the worker thread
 * @param {Object} workerData - Data to be passed to the worker thread
 * @param {Function} callback - Callback function to be called when the worker thread sends data
 */
class WorkerThread extends EventEmitter {
  constructor (workerData, callback) {
    super();
    this.id = workerData.id;
    this.worker = new Worker(scenarioTestingProcessPath, {
      workerData: {},
      stdout: true,
      stderr: true
    });
    this.handleMessage = this.handleMessage.bind(this);
    this.handleError = this.handleError.bind(this);
    this.handleExit = this.handleExit.bind(this);

    this.worker.on('message', this.handleMessage);
    this.worker.on('error', this.handleError);
    this.worker.on('exit', this.handleExit);

    // pm.logger is intentionally used here instead of log because log is not yet designed to be used in the worker
    this.worker.stdout.on('data', (data) => pm.logger.info([`Worker ${this.id}`, data.toString()]));
    this.worker.stderr.on('data', (data) => pm.logger.error([`Worker ${this.id}`, data.toString()]));
    this.callback = callback;
  }

  handleMessage (message) {
    log.debug('WorkerThread: handleMessage');

    if (message.type === 'data') {
      this.callback(message.data);
    }
  }

  handleError (error) {
    log.error('Worker error:', error);
  }

  handleExit (code) {
    if (code !== 0) {
      log.error(`Worker stopped with exit code ${code}`);
    }
  }

  async assignRun (args) {
    log.debug('WorkerThread: assignRun');
    this.worker.postMessage({ type: 'assign', payload: args });
  }

  async stop () {
    log.debug('WorkerThread: stop');

    this.worker.off('message', this.handleMessage);

    log.info(`Worker ${this.id} stopping`);

    return this.worker.terminate().then(() => {
      log.info(`Worker ${this.id} stopped`);
    });
  }
}

module.exports = WorkerThread;
