const MetricProcessor = require('./metric-processor');

/**
 * This class provides the interface to the custom metrics storage system for performance runs.
 * A unique instance of this is to be initialized for each performance run.
 */
class RunnerMetricStore {
  constructor () {
    this.emit = this.emit.bind(this);
    this.getMetrics = this.getMetrics.bind(this);
    this.updateWindowSize = this.updateWindowSize.bind(this);

    // Default window size is 1 second
    this.metricProcessor = new MetricProcessor(1);
  }

  /**
   * Records the metric measurement in the store, along with the dimensions
   *
   * @param {String} metricName - Name of the metric
   * @param {String} requestId - The ID of request for which this metric is recorded.
   * @param {Number} value - Value of the metric
   * @param {String} timestamp - The ISO timestamp when request was sent
   */
  emit (obj) {
    this.metricProcessor.add(obj);
  }

  /**
   * Returns all the metrics required by the view.
   * This includes the summary metrics, time-series and metrics grouped by request IDs.
   *
   * @returns {Object}
   * Example:
   *  {
   *    summary: {
   *      aggregatedMetrics: {
   *      },
   *      groupedByRequest: {
   *      }
   *    },
   *    timeSeries: {
   *
   *    }
   *  }
   */
  getMetrics () {
    return this.metricProcessor.getMetrics();
  }

  /**
   * Updates the window size used by aggregation logic to return time series data.
   *
   * @param {Number} windowSize
   */
  updateWindowSize (windowSize) {
    this.metricProcessor.updateWindowSize(windowSize);
  }
}

module.exports = RunnerMetricStore;
