const os = require('os'),
    path = require('path'),
    fs = require('fs'),
    _ = require('lodash'),
    sinon = require('sinon'),
    { expect } = require('chai'),
    Originator = require('../AbstractOriginator'),
    Collector = require('../../collectors/AbstractCollector'),
    WinstonConsoleCollector = require('../../../').Collectors.Console,
    WinstonFileCollector = require('../../../').Collectors.File,
    WinstonSentryCollector = require('../../collectors/winston/SentryCollector'),
    LOG_FILE_DIR = path.resolve(__dirname, '..', '..', 'test_logs'),
    LOG_FILE = path.resolve(LOG_FILE_DIR, 'winston.log'),
    dummySentry = {
        withScope: (cb) => {
            scope = {
                setExtra: dummySentry.setExtra
            };
            cb(scope);
        },
        captureException: _.noop,
        setExtra: _.noop
    },
    resetLogFile = (done) => {
        setTimeout(() => {
            try {
                if (!fs.existsSync(LOG_FILE_DIR)) {
                    fs.mkdirSync(LOG_FILE_DIR);
                }
                fs.unlinkSync(LOG_FILE + 1);
                fs.unlinkSync(LOG_FILE + 2);
                fs.unlinkSync(LOG_FILE + 3);
                fs.unlinkSync(LOG_FILE + 4);
                fs.unlinkSync(LOG_FILE + 5);
                fs.unlinkSync(LOG_FILE + 6);
                fs.unlinkSync(LOG_FILE + 7);
            }
            catch (e) {
                // Don't fail test case if unlink fails
            }
            finally {
                done();
            }
        }, 1000);

    },
    stdMocks = require('std-mocks');

describe('Originator', () => {
    let sandbox = null;
    before((done) => {
        global.Date.now = () => {
            return 123456;
        };
        resetLogFile(done);
        sandbox = sinon.sandbox.create();
    });

    afterEach((done) => {
        resetLogFile(done);
        sandbox.restore();
    });


    it('should thrown Invalid param exception on wrong options type', () => {
        // eslint-disable-next-line no-new,max-len
        expect(() => { new Originator('options'); }).to.throw('InvalidParamException: options should be of type Object');
    });

    it('should create originator instance with default level as info', () => {
        let originator = new Originator({ origin: 'foo' });

        expect(originator.error).to.not.empty;
        expect(originator.warn).to.not.empty;
        expect(originator.info).to.not.empty;
        expect(originator.level).to.equal('info');
    });

    it('should create originator instance level as error', () => {
        let originator = new Originator({ level: 'error', origin: 'foo' });

        expect(originator.level).to.equal('error');
    });

    it('should create originator instance level as warn', () => {
        let originator = new Originator({ level: 'warn', origin: 'foo' });

        expect(originator.level).to.equal('warn');
    });

    it('should create originator instance level as info', () => {
        let originator = new Originator({ level: 'info', origin: 'foo' });

        expect(originator.level).to.equal('info');
    });

    it('should create originator instance level as info when invalid level is provided', () => {
        let originator = new Originator({ level: 'invalid', origin: 'foo' });

        expect(originator.level).to.equal('info');
    });

    it('should log on calling error method', () => {
        let originator = new Originator({ origin: 'foo', collectors: [new Collector()] }),
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = ['error { sessionId: 123456,\n  timestamp: 123456,\n  origin: \'foo\',\n  messages: [ \'message\' ] }\n']; // eslint-disable-line max-len

        stdMocks.use();
        originator.error('message');
        stdMocks.restore();

        output = stdMocks.flush();

        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(1);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);
    });

    it('should add collector after initialization', () => {
        let originator = new Originator({ origin: 'foo', collectors: [] }),
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = ['error { sessionId: 123456,\n  timestamp: 123456,\n  origin: \'foo\',\n  messages: [ \'message\' ] }\n']; // eslint-disable-line max-len

        originator.addCollector(new Collector());

        stdMocks.use();
        originator.error('message');
        stdMocks.restore();

        output = stdMocks.flush();

        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(1);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);
    });

    it('should log on calling warn method', () => {
        let originator = new Originator({ origin: 'foo', collectors: [new Collector()] }),
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = ['warn { sessionId: 123456,\n  timestamp: 123456,\n  origin: \'foo\',\n  messages: [ \'message\' ] }\n']; // eslint-disable-line max-len

        stdMocks.use();
        originator.warn('message');
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(1);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);
    });

    it('should log on calling info method', () => {
        let originator = new Originator({ origin: 'foo', collectors: [new Collector()] }),
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = ['info { sessionId: 123456,\n  timestamp: 123456,\n  origin: \'foo\',\n  messages: [ \'message\' ] }\n']; // eslint-disable-line max-len

        stdMocks.use();
        originator.info('message');
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(1);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);
    });

    it('should log on calling error method with sessionId', () => {
        let originator = new Originator({ origin: 'foo', sessionId: 'newSession', collectors: [new Collector()] }),
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = ['error { sessionId: \'newSession\',\n  timestamp: 123456,\n  origin: \'foo\',\n  messages: [ \'message\' ] }\n']; // eslint-disable-line max-len

        stdMocks.use();
        originator.error('message');
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(1);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);
    });

    it('should log on calling info method with empty message', () => {
        let originator = new Originator({ origin: 'foo', collectors: [new Collector()] }),
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = ['info { sessionId: 123456,\n  timestamp: 123456,\n  origin: \'foo\',\n  messages: [] }\n']; // eslint-disable-line max-len

        stdMocks.use();
        originator.info();
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(1);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);
    });

    it('should log on calling info method with 2 messages', () => {
        let originator = new Originator({ origin: 'foo', collectors: [new Collector()] }),
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = ['info { sessionId: 123456,\n  timestamp: 123456,\n  origin: \'foo\',\n  messages: [ \'first\', \'second\' ] }\n']; // eslint-disable-line max-len

        stdMocks.use();
        originator.info('first', 'second');
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(1);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);
    });

    it('should log on calling info method with 3 messages', () => {
        let originator = new Originator({ origin: 'foo', collectors: [new Collector()] }),
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = ['info { sessionId: 123456,\n  timestamp: 123456,\n  origin: \'foo\',\n  messages: [ \'first\', \'second\', \'third\' ] }\n']; // eslint-disable-line max-len

        stdMocks.use();
        originator.info('first', 'second', 'third');
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(1);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);
    });

    it('should create originators with winston console collector', () => {
        let output = {},
            stdout = [],
            stderr = [],
            expectedStdout = [
                `123456 foo info "foo"${os.EOL}`,
                `123456 bar error "Foo Error"${os.EOL}`,
                `123456 foo warn {"foo":"bar"}${os.EOL}`
            ],
            collectors = [new WinstonConsoleCollector()],
            fooLogger = new Originator({ collectors, origin: 'foo' }),
            barLogger = new Originator({ collectors, origin: 'bar' });

        stdMocks.use();
        fooLogger.info('foo');
        barLogger.error('Foo Error');
        fooLogger.warn({ foo: 'bar' });
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(3);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);
    });

    it('should create originators with winston console collector and call with two arguments', () => {
        let output = {},
            stdout = [],
            stderr = [],
            expectedStdout = [
                `123456 foo info "foo","bar"${os.EOL}`
            ],
            collectors = [new WinstonConsoleCollector()],
            fooLogger = new Originator({ collectors, origin: 'foo' });

        stdMocks.use();
        fooLogger.info('foo', 'bar');
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(1);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);
    });

    it('should create originator with winston file collector', (done) => {
        let fileString = '',
            expectedFileOut = `[123456][123456][foo][info]["foo"]${os.EOL}[123456][123456][foo][warn][{"foo":"bar"}]${os.EOL}`, // eslint-disable-line max-len
            collectors = [new WinstonFileCollector({ file: LOG_FILE + 1 })],
            fooLogger = new Originator({ collectors, origin: 'foo' });

        fooLogger.info('foo');
        fooLogger.warn({ foo: 'bar' });

        // Reading file immediately will throw error
        setTimeout(() => {
            fileString = fs.readFileSync(LOG_FILE + 1, { encoding: 'utf-8' });
            try {
                expect(fileString).to.equal(expectedFileOut);
                done();
            }
            catch (e) {
                done(e);
            }
        }, 1000);
    });

    it('should create originator with winston file collector and call with two arguments', (done) => {
        let fileString = '',
            expectedFileOut = `[123456][123456][foo][info]["foo","bar"]${os.EOL}`,
            collectors = [new WinstonFileCollector({ file: LOG_FILE + 2 })],
            fooLogger = new Originator({ collectors, origin: 'foo' });

        fooLogger.info('foo', 'bar');

        // Reading file immediately will throw error
        setTimeout(() => {
            fileString = fs.readFileSync(LOG_FILE + 2, { encoding: 'utf-8' });
            try {
                expect(fileString).to.equal(expectedFileOut);
                done();
            }
            catch (e) {
                done(e);
            }
        }, 1000);
    });

    it('should create originator with winston file and winston console collector', (done) => {
        let fileString = '',
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = [
                `123456 foo info "foo"${os.EOL}`,
                `123456 foo warn {"foo":"bar"}${os.EOL}`
            ],
            expectedFileOut = `[123456][123456][foo][info]["foo"]${os.EOL}[123456][123456][foo][warn][{"foo":"bar"}]${os.EOL}`, // eslint-disable-line max-len
            collectors = [
                new WinstonConsoleCollector(),
                new WinstonFileCollector({ file: LOG_FILE + 3 })
            ],
            fooLogger = new Originator({ collectors: collectors, origin: 'foo' });

        stdMocks.use();
        fooLogger.info('foo');
        fooLogger.warn({ foo: 'bar' });
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(2);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);


        // Reading file immediately will throw error
        setTimeout(() => {
            fileString = fs.readFileSync(LOG_FILE + 3, { encoding: 'utf-8' });
            try {
                expect(fileString).to.equal(expectedFileOut);
                done();
            }
            catch (e) {
                done(e);
            }
        }, 1000);
    });

    it('should log info and higher messages after changing log level to info', (done) => {
        let fileString = '',
            expectedFileOut = `[123456][123456][foo][info]["foo"]${os.EOL}[123456][123456][foo][warn]["bar"]${os.EOL}[123456][123456][foo][error]["foobar",{"crash":true}]${os.EOL}`, // eslint-disable-line max-len
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = [
                `123456 foo info "foo"${os.EOL}`,
                `123456 foo warn "bar"${os.EOL}`,
                `123456 foo error "foobar",{"crash":true}${os.EOL}`
            ],
            sentryExceptionSpy = sandbox.spy(dummySentry, 'captureException'),
            collectors = [
                new WinstonConsoleCollector(),
                new WinstonFileCollector({ file: LOG_FILE + 4 }),
                new WinstonSentryCollector({ sentry: dummySentry })
            ],
            fooLogger = new Originator({ collectors: collectors, origin: 'foo' });

        fooLogger.setLogLevel('info');

        stdMocks.use();
        fooLogger.info('foo');
        fooLogger.warn('bar');
        fooLogger.error('foobar', { crash: true });
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(3);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);

        // sentry should be called only once (for error)
        expect(sentryExceptionSpy.callCount).to.equal(1);

        // Reading file immediately will throw error
        setTimeout(() => {
            fileString = fs.readFileSync(LOG_FILE + 4, { encoding: 'utf-8' });
            try {
                expect(fileString).to.equal(expectedFileOut);
                done();
            }
            catch (e) {
                done(e);
            }
        }, 1000);
    });

    it('should log warn and higher messages after changing log level to warn', (done) => {
        let fileString = '',
            expectedFileOut = `[123456][123456][foo][warn]["bar"]${os.EOL}[123456][123456][foo][error]["foobar",{"crash":true}]${os.EOL}`, // eslint-disable-line max-len
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = [
                `123456 foo warn "bar"${os.EOL}`,
                `123456 foo error "foobar",{"crash":true}${os.EOL}`
            ],
            sentryExceptionSpy = sandbox.spy(dummySentry, 'captureException'),
            collectors = [
                new WinstonConsoleCollector(),
                new WinstonFileCollector({ file: LOG_FILE + 5 }),
                new WinstonSentryCollector({ sentry: dummySentry })
            ],
            fooLogger = new Originator({ collectors: collectors, origin: 'foo' });

        fooLogger.setLogLevel('warn');

        stdMocks.use();
        fooLogger.info('foo');
        fooLogger.warn('bar');
        fooLogger.error('foobar', { crash: true });
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(2);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);

        // sentry should be called only once (for error)
        expect(sentryExceptionSpy.callCount).to.equal(1);

        // Reading file immediately will throw error
        setTimeout(() => {
            fileString = fs.readFileSync(LOG_FILE + 5, { encoding: 'utf-8' });
            try {
                expect(fileString).to.equal(expectedFileOut);
                done();
            }
            catch (e) {
                done(e);
            }
        }, 1000);
    });

    it('should log error and higher messages after changing log level to error', (done) => {
        let fileString = '',
            expectedFileOut = `[123456][123456][foo][error]["foobar",{"crash":true}]${os.EOL}`,
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = [
                `123456 foo error "foobar",{"crash":true}${os.EOL}`
            ],
            sentryExceptionSpy = sandbox.spy(dummySentry, 'captureException'),
            collectors = [
                new WinstonConsoleCollector(),
                new WinstonFileCollector({ file: LOG_FILE + 6 }),
                new WinstonSentryCollector({ sentry: dummySentry })
            ],
            fooLogger = new Originator({ collectors: collectors, origin: 'foo' });

        fooLogger.setLogLevel('error');

        stdMocks.use();
        fooLogger.info('foo');
        fooLogger.warn('bar');
        fooLogger.error('foobar', { crash: true });
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(1);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);

        // sentry should be called only once (for error)
        expect(sentryExceptionSpy.callCount).to.equal(1);

        // Reading file immediately will throw error
        setTimeout(() => {
            fileString = fs.readFileSync(LOG_FILE + 6, { encoding: 'utf-8' });
            try {
                expect(fileString).to.equal(expectedFileOut);
                done();
            }
            catch (e) {
                done(e);
            }
        }, 1000);
    });

    it('should log messages as per set log level', (done) => {
        let fileString = '',
            expectedFileOut = `[123456][123456][foo][warn]["bar"]${os.EOL}[123456][123456][foo][error]["foobar",{"crash":true}]${os.EOL}[123456][123456][foo][info]["foo"]${os.EOL}[123456][123456][foo][warn]["bar"]${os.EOL}[123456][123456][foo][error]["foobar",{"context":{"api":"foo","domain":"bar"},"crash":true}]${os.EOL}[123456][123456][foo][error]["foobar"]${os.EOL}`, // eslint-disable-line max-len
            output = {},
            stdout = [],
            stderr = [],
            expectedStdout = [
                `123456 foo warn "bar"${os.EOL}`,
                `123456 foo error "foobar",{"crash":true}${os.EOL}`,
                `123456 foo info "foo"${os.EOL}`,
                `123456 foo warn "bar"${os.EOL}`,
                `123456 foo error "foobar",{"context":{"api":"foo","domain":"bar"},"crash":true}${os.EOL}`,
                `123456 foo error "foobar"${os.EOL}`
            ],
            sentryExceptionSpy = sandbox.spy(dummySentry, 'captureException'),
            collectors = [
                new WinstonConsoleCollector(),
                new WinstonFileCollector({ file: LOG_FILE + 7 }),
                new WinstonSentryCollector({ sentry: dummySentry })
            ],
            fooLogger = new Originator({ collectors: collectors, origin: 'foo' });

        stdMocks.use();
        fooLogger.setLogLevel('warn');

        fooLogger.info('foo');
        fooLogger.warn('bar');
        fooLogger.error('foobar', { crash: true });

        fooLogger.setLogLevel('info');

        fooLogger.info('foo');
        fooLogger.warn('bar');
        fooLogger.error('foobar', { context: { api: 'foo', domain: 'bar' }, crash: true });

        fooLogger.setLogLevel('error');

        fooLogger.info('foo');
        fooLogger.warn('bar');
        fooLogger.error('foobar');
        stdMocks.restore();

        output = stdMocks.flush();
        stdout = output.stdout;
        stderr = output.stderr;

        expect(stdout).to.be.an('array');
        expect(stdout).to.have.lengthOf(6);
        expect(stdout).to.eql(expectedStdout);
        expect(stderr).to.be.an('array');
        expect(stderr).to.have.lengthOf(0);

        // sentry should be called only once (for error)
        expect(sentryExceptionSpy.callCount).to.equal(2);

        // Reading file immediately will throw error
        setTimeout(() => {
            fileString = fs.readFileSync(LOG_FILE + 7, { encoding: 'utf-8' });
            try {
                expect(fileString).to.equal(expectedFileOut);
                done();
            }
            catch (e) {
                done(e);
            }
        }, 1000);
    });

    it('should return a context object when getContext() is called', (done) => {
        let collectors = [
                new WinstonConsoleCollector()
            ],
            fooLogger = new Originator({ collectors: collectors, origin: 'foo' });

        expect(fooLogger.getContext('foo', 'bar')).to.be.a('object').and.be.eql({
            api: 'foo',
            domain: 'bar'
        });
        done();
    });

});
