/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.fn.Prototype;

import java.text.Collator;

/**
 *  Generic part of all value comparisons. Each particular operator (eq gt ...)
 *  has only a simple decision maker.
 */
public abstract class ValueComparison extends Comparison {

    public ValueComparison( Expression expr1, Expression expr2 ) {
        super(expr1, expr2);
    }

    static Prototype[] protos = { 
        Prototype.op("value-comparison", Type.BOOLEAN, ExecD.class).hidden()
	  .arg("op1", Type.DOUBLE).arg("op2", Type.UNTYPED_ATOMIC),
        Prototype.op("value-comparison", Type.BOOLEAN, ExecD.class).hidden()
	  .arg("op1", Type.UNTYPED_ATOMIC).arg("op2", Type.DOUBLE),
        Prototype.op("value-comparison", Type.BOOLEAN, ExecI.class)
	    .arg("op1", Type.INTEGER).arg("op2", Type.INTEGER),
        Prototype.op("value-comparison", Type.BOOLEAN, ExecD.class)
	    .arg("op1", Type.DOUBLE).arg("op2", Type.DOUBLE),
        Prototype.op("value-comparison", Type.BOOLEAN, ExecS.class)
	    .arg("op1", Type.STRING).arg("op2", Type.STRING),
        Prototype.op("value-comparison", Type.BOOLEAN, ExecAny.class)
            .arg("op1", Type.ITEM.opt).arg("op2", Type.ITEM.opt)
    };
    public Prototype[] getProtos() { return protos; }

    public static class ExecI extends Comparison.Exec {

	public boolean evalAsBoolean( Focus focus, EvalContext context ) throws XQueryException {
	    long e1 = args[0].evalAsInteger(focus, context);
	    long e2 = args[1].evalAsInteger(focus, context);
	    context.at(this);
	    return decision.make(e1 < e2 ? -1 : e1 > e2 ? 1 : 0);
	}
    }

    public static class ExecD extends Comparison.Exec {

	public boolean evalAsBoolean( Focus focus, EvalContext context ) throws XQueryException {
	    double d1 = args[0].evalAsDouble(focus, context);
	    double d2 = args[1].evalAsDouble(focus, context);
	    context.at(this);
	    return decision.make(d1 < d2 ? -1 : d1 > d2 ? 1 : 0);
	}
    }

    public static class ExecS extends Comparison.Exec {

	public boolean evalAsBoolean( Focus focus, EvalContext context ) throws XQueryException {
	    String s1 = args[0].evalAsString(focus, context);
	    String s2 = args[1].evalAsString(focus, context);
	    context.at(this);
	    Collator coll = context.getCollator(null);
	    int cmp = coll != null ? coll.compare(s1, s2) : s1.compareTo(s2);
	    return decision.make(cmp < 0 ? -1 : cmp > 0 ? 1 : 0);
	}
    }

    public static class ExecAny extends Comparison.Exec {

	public boolean evalAsBoolean( Focus focus, EvalContext context ) throws XQueryException {
	    Value v1 = args[0].eval(focus, context);
	    Value v2 = args[1].eval(focus, context);
	    if(!v1.next() || !v2.next())
		context.error(this, "empty sequence not allowed in value comparison");
	    context.at(this);
	    int cmp = v1.compareTo( v2, context.getCollator(null), context.getImplicitTimezone() );
	    if(Math.abs(cmp) == Item.INCOMPARABLE)
		context.error(this, "values are not comparable");
	    return decision.make(cmp);
	}
    }
}
