/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.dm.XMLEventReceiver;
import net.xfra.qizxopen.dm.DataModelException;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.*;
import net.xfra.qizxopen.xquery.dt.QNameValue;

/**
 *  Superclass of Element, Attribute, PI constructors.
 */
public abstract class NamedConstructor extends NodeConstructor {

    public Expression  name;

    public NamedConstructor( Expression name ) {
        contents = new Expression[0];
	this.name = name;
    }

    public void dump( ExprDump d ) {
	d.header( this, "NamedConstructor" );
        d.display("name", name);
        d.display("contents", contents);
    }

    public Expression staticCheck( StaticContext context ) {
	super.staticCheck(context);
	if(name != null) {
	    name = context.staticCheck( name, 0 );
	    if( !Type.QNAME.accepts(name.getType()) &&
	        !Type.STRING.accepts(name.getType())  ) {
		context.getLog().error( name.module, name.location,
			 "bad type for constructor name: must be xs:string or xs:QName");
	    }
	}
	return this;
    }

    public Expression addTextItem( String chunk ) {
	Expression last = null;
	TextLiteral result = null;
	if(contents.length > 0 &&
	   (last = contents[contents.length - 1]) instanceof TextLiteral)
	{
	    result = (TextLiteral) last;
	    result.value += chunk;
	}
	else addItem(result = new TextLiteral(chunk));
	return result;
    }

    QName evalName( XMLEventReceiver output, Focus focus, EvalContext context)
	throws XQueryException {
	Value n = name.eval(focus, context);
	if( !n.next() )
	    context.error(this, Type.ERR_EMPTY_UNEXPECTED);
	if(Type.QNAME.accepts(name.getType()))
	   return ((QNameValue) n).getValue();
        try {
	    String pname = n.asString();
	    String prefix = NSPrefixMapping.extractPrefix(pname);
	    String ncname = NSPrefixMapping.extractLocalName(pname);
	    if(prefix.length() == 0)
		return QName.get(Namespace.NONE, ncname);
	    // The draft is unclear: only static in-scope NS or NS in-scope for the current node?
	    QName qname = null;
	    String uri = (output == null) ? null : output.resolvePrefix(prefix);
	    if(uri != null)
		qname = QName.get(uri, ncname);
	    else	
		qname = context.getStaticContext().getInScopeNS().expandName( pname );
	    if(qname == null)
		context.error(this, "no namespace found for prefix "+prefix);
	    return qname;
        }
        catch (Exception e) {
            context.error(this, "error converting string to QName: "+e.getMessage());
	    return null;
        }
    }

    String evalContents(Focus focus, EvalContext context) throws XQueryException {
	StringBuffer buf = new StringBuffer();
	for(int c = 0; c < contents.length; c++) {
	    Value part = contents[c].eval(focus, context);
	    if(part.next())
		buf.append( part.asString() );
	    for( ; part.next(); ) {
		buf.append( ' ' );
		buf.append( part.asString() );
	    }
	}
	return buf.toString();
    }

    void evalContents( Node node, Focus focus, EvalContext context )
	throws XQueryException {
	for(int c = 0; c < contents.length; c++) {
	    Value part = contents[c].eval(focus, context);
	    if(part.next())
		node.addText( part.asString() );
	    for( ; part.next(); ) {
		node.addText( " " );
		node.addText( part.asString() );
	    }
	}
    }
}
