/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dt.SingleString;
import net.xfra.qizxopen.xquery.dt.GenericValue;


/**
 *  class FilterExpr: 
 * 
 */
public class FilterExpr extends Expression {

    public Expression  source;
    public Expression[] predicates;

    public FilterExpr( Expression source ) {
        this.source = source;
        predicates = new Expression[0];
    }

    public void addPredicate( Expression predic ) {
	predicates = addExpr(predicates, predic);
    }

    public void dump( ExprDump d ) {
	d.header( this, "FilterExpr" );
        d.display("source", source);
        d.display("predicates", predicates);
    }

    public boolean visit( Visitor v ) {
	return v.examine(this) && source.visit(v) && v.visit(predicates);
    }

    public int getFlags() {
	return source.getFlags();
    }

    public Expression staticCheck( StaticContext context ) {
	source = context.staticCheck( source, 0 );
	context.pushDotType(source.getType());
	for(int p = 0; p < predicates.length; p++) {
	    Expression pred = predicates[p] = context.staticCheck( predicates[p], 0 );
	    // CAUTION: use 'isSuperType', not 'accepts' which is too lax
	    ItemType predType = pred.getType().getItemType();
	    
	    // recognize numeric predicates : wrapped into a special construct
	    if( !( pred instanceof net.xfra.qizxopen.xquery.fn.Last.Exec ) && 
		Type.NUMERIC.isSuperType(predType))
		if( pred.findSubExpression( net.xfra.qizxopen.xquery.fn.Position.Exec.class )
		    != null ||
		    pred.findSubExpression( net.xfra.qizxopen.xquery.fn.Last.Exec.class )
		    != null ||
		    pred.findSubExpression( SelfStep.class ) != null )
		    predicates[p] = context.staticCheck(new ValueEqOp(pred,
				new FunctionCall(QName.get(Namespace.FN, "position"))), 0);
		else
		    predicates[p] = new PosTest(pred);
	    //TODO: recognize "slices":
	    //position() >= M and position() <= N | position = M to N
	}
	// manage naturalStepOrder: 
	if(source instanceof BasicStep)
	    ((BasicStep) source).naturalStepOrder = true;
	context.popDotType();
	type = source.getType();
	// filtering can make a sequence empty:
	// transform '+' into '*' and one-occ into '?' :
	if(Type.isRepeatable(type.getOccurrence()))
	    type = type.getItemType().star;
	else
	    type = type.getItemType().opt;
	// EBV of predicates is used : no need of type check
	return this;
    }

    public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	Value src = source.eval( focus, context );
	for(int p = 0; p < predicates.length; p++) {
	    Expression pred = predicates[p];
	    if( pred instanceof PosTest) {
		Expression index = ((PosTest) pred).index;
		src = new Index( src, (long) index.evalAsDouble(focus, context) );
	    }
	    else if(pred instanceof net.xfra.qizxopen.xquery.fn.Last.Exec)
		src = new Last(src);
	    else // general predicate
		src = new Sequence(src, pred, context);
	}
	return src;
    }

    private static class PosTest extends Expression {
	Expression index;

	PosTest(Expression index) { this.index = index; }

	public boolean visit( Visitor v ) {
	    return v.examine(this) && index.visit(v);
	}

	public boolean evalEffectiveBooleanValue( Focus focus, EvalContext context )
	    throws XQueryException {
	    return focus.getPosition() == index.evalAsDouble(focus, context);
	}
    }

    /**
     *	General filtering sequence, where the predicate is evaluated for each
     *  input item.
     */
    public static class Sequence extends GenericValue implements Focus
    {
	Value source;
	Expression predicate;
	EvalContext evalContext;  // for evaluation of the predicate
	int   position = 0;
	int   last = -1;	// caching: if >= 0 , it is already evaluated

	Sequence( Value source, Expression predicate, EvalContext evalContext ) {
	    this.evalContext = evalContext;
	    this.predicate = predicate;
	    this.source = source;
	}

	public boolean next() throws XQueryException {
	    for (;;) {
		evalContext.at(predicate);
		if(!source.next())
		    return false;
		item = source.asItem();
		++ position;
		// change the focus:
		if( predicate.evalEffectiveBooleanValue( this, evalContext ) )
		    return true;
		
	    }
	}

	public Value  bornAgain() {
	    return new Sequence( source.bornAgain(), predicate, evalContext );
	}

	// implementation of Focus 

	public Item getItem() {
	    return item;
	}

	public long getItemAsInteger() throws XQueryException {
	    return source.asInteger();
	}

	public int  getPosition() {
	    return position;
	}

	public int  getLast() throws XQueryException {
	    if(last < 0) {
		Value sba = source.bornAgain();
		for(last = 0; sba.next(); )
		    ++ last;
	    }
	    return last;
	}
    } // end of class Sequence

    public static class Last extends Sequence {
	private boolean started = false;

	public Last(Value source) {
	    super(source, null, null);
	}

	public Value  bornAgain() {
	    return new Last( source.bornAgain() );
	}

	public boolean next() throws XQueryException {
	    if(started)
		return false;
	    started = true;
	    for (; source.next(); )
		item = source.asItem();
	    return item != null;
	}
    }

    public static class Index extends Sequence {
	long index;

	public Index(Value source, long index) {
	    super(source, null, null);
	    this.index = index;
	    
	}

	public Value  bornAgain() {
	    return new Index( source.bornAgain(), index );
	}

	public boolean next() throws XQueryException {
	    for (;;) {
		++ position;
		if(!source.next() || position > index)
		    return false;
		if( position == index) {
		    item = source.asItem();
		    return true;
		}
	    }
	}
    }

    public static class Slice extends Index {
	long indexHi;

	public Slice(Value source, long index, long indexHi) {
	    super(source, index);
	    this.indexHi = indexHi;
	}

	public Value  bornAgain() {
	    return new Slice( source.bornAgain(), index, indexHi );
	}

	public boolean next() throws XQueryException {
	    for (;;) {
		++ position;
		if(!source.next() || position > indexHi)
		    return false;
		if( position >= index) {
		    item = source.asItem();
		    return true;
		}
	    }
	}
    }
}
