/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.dt.SingleDecimal;
import net.xfra.qizxopen.xquery.fn.Prototype;

import java.math.BigDecimal;

/**
 *  Implementation of operator div. 
 */
public class DivOp extends NumericOp {

    /**
     *	Precision of decimal division.
     */
    final static int DECIMAL_SCALE = 8;

    public DivOp( Expression expr1, Expression expr2 ) {
        super( expr1, expr2 );
    }

    static Prototype[] protos = { 
        Prototype.op("div", Type.DOUBLE, ExecD.class).hidden()
	  .arg("op1", Type.DOUBLE).arg("op2", Type.UNTYPED_ATOMIC),
        Prototype.op("div", Type.DOUBLE, ExecD.class).hidden()
	  .arg("op1", Type.UNTYPED_ATOMIC).arg("op2", Type.DOUBLE),
        Prototype.op("div", Type.DECIMAL, ExecDec.class)
	  .arg("op1", Type.DECIMAL).arg("op2", Type.DECIMAL),
        Prototype.op("div", Type.FLOAT, ExecF.class)
	  .arg("op1", Type.FLOAT).arg("op2", Type.FLOAT),
        Prototype.op("div", Type.DOUBLE, ExecD.class)
	  .arg("op1", Type.DOUBLE).arg("op2", Type.DOUBLE),
    };
    public Prototype[] getProtos() { return protos; }

    public void dump( ExprDump d ) {
	d.header( this, "div" );
        d.display("expr1", operands[0]);
        d.display("expr2", operands[1]);
    }

    public static class ExecDec extends Function.Call {

	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    try {
		Value e1 = args[0].eval(focus, context);
		Value e2 = args[1].eval(focus, context);
		context.at(this);
		return new SingleDecimal( e1.asDecimal().divide( e2.asDecimal(),
					      DECIMAL_SCALE, BigDecimal.ROUND_HALF_EVEN ));
	    }
	    catch(ArithmeticException e) {
		return context.error(this, e.getMessage());
	    }
	    catch(TypeException te) {
		return context.error(this, te);
	    }
	}
    }

    public static class ExecF extends Function.FloatCall {

	public float evalAsFloat( Focus focus, EvalContext context ) throws XQueryException {
	    float e1 = args[0].evalAsFloat(focus, context);
	    float e2 = args[1].evalAsFloat(focus, context);
	    context.at(this);
	    return e1 / e2;
	}
    }

    public static class ExecD extends Function.DoubleCall {

	public double evalAsDouble( Focus focus, EvalContext context ) throws XQueryException {
	    double e1 = args[0].evalAsDouble(focus, context);
	    double e2 = args[1].evalAsDouble(focus, context);
	    context.at(this);
	    return e1 / e2;
	}
    }
}
