/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.fn;
import net.xfra.qizxopen.xquery.impl.*;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.Util;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.op.Expression;
import net.xfra.qizxopen.xquery.op.GlobalVariable;
import net.xfra.qizxopen.xquery.dm.Node;
import net.xfra.qizxopen.xquery.dt.*;

import java.lang.reflect.*;
import java.util.Enumeration;
import java.util.Vector;
import java.util.HashSet;
import java.util.ArrayList;

/**
 *	Function bound to a Java method.
 *	<p>The binding mechanism is based on the qname of the function: the uri part
 *	has the form java:<fully_qualified_class_name> (e.g. "java:java.lang.Math")
 *	and the local part matches the Java method name. For example the following code
 *	calls java.lang.Math.log :
 *	<pre>declare namespace math = "java:java.lang.Math"
 *	math:log(1)</pre>
 *	<p>The function name can be in XPath style with hyphens: it is converted
 *	into 'camelCase' style automatically. For example "list-files" is converted
 *	to "listFiles".
 *	<p>
 */
public class JavaFunction extends Function
{
    Prototype[] prototypes;
    public static boolean trace = false;

    // recognized types:
    final static int NONE = 0;
    final static int STRING = 1;
    final static int BOOLEAN = 2;
    final static int DOUBLE = 3;
    final static int FLOAT = 4;
    final static int INTEGER = 5;
    final static int INT = 6;
    final static int SHORT = 7;
    final static int BYTE = 8;
    final static int NODE = 9 ;
    final static int WRAPPED_OBJECT = 10;
    final static int STRING_ARRAY = 11;
    final static int CHAR_ARRAY = 12;
    final static int DOUBLE_ARRAY = 13;
    final static int FLOAT_ARRAY = 14;
    final static int INTEGER_ARRAY = 15;
    final static int INT_ARRAY = 16;
    final static int SHORT_ARRAY = 17;
    final static int BYTE_ARRAY = 18;
    final static int NODE_ARRAY = 19;
    final static int WRAPPED_OBJECT_ARRAY = 20;
    final static int CHAR = 21 ;
    final static int ENUMERATION = 22;
    final static int VECTOR = 23;
    final static int ARRAYLIST = 24;

    // mapping of Java types to XQ types:
    final static Type typeTable[] = {
	Type.NONE, Type.STRING.opt, Type.BOOLEAN, Type.DOUBLE, Type.FLOAT, Type.INTEGER,
	Type.INT, Type.SHORT, Type.BYTE, Type.NODE.opt, Type.WRAPPED_OBJECT.opt, 

	Type.STRING.star, Type.INTEGER.star, Type.DOUBLE.star, /*float*/Type.DOUBLE.star,
	// INT* SHORT* BYTE* are mapped to integer*
	Type.INTEGER.star, Type.INTEGER.star, Type.INTEGER.star, Type.INTEGER.star,
	Type.NODE.star, Type.WRAPPED_OBJECT.star, Type.INTEGER,
	Type.WRAPPED_OBJECT.star, Type.WRAPPED_OBJECT.star, Type.WRAPPED_OBJECT.star
    };

    public static class Prototype extends net.xfra.qizxopen.xquery.fn.Prototype
    {
	Method method;			// method and constructor are exclusive
	Constructor constructor;
	int[]  conversions;	
	int    resultConversion;
	QName  autoArg0;	// name of a global automatically added as first arg.

	public Prototype( QName qname, Type returnType,
			  Method method, Constructor constructor) {
	    super(qname, returnType, Call.class);
	    this.method = method;
	    this.constructor = constructor;
	}

	public Function.Call newInstance(StaticContext ctx, Expression[] actualArguments) {
	    Call call = (Call) super.newInstance(ctx, actualArguments);

	    if(autoArg0 != null && !Modifier.isStatic(method.getModifiers())) {
		// add a first arg which a reference to a global
		GlobalVariable global = ctx.lookforGlobalVariable(autoArg0);
		if(global == null)
		    throw new RuntimeException("no such auto arg "+autoArg0);
		call.autoArg0 = global;
	    }
	    return call;
	}
    }
    
    public static class Plugger implements PredefinedModule.FunctionPlugger {
	HashSet allowedClasses;

	public void authorizeClass( String className ) {
	    if(allowedClasses == null)
		allowedClasses = new HashSet();
	    allowedClasses.add(className);
	}

	public Function plug( QName qname ) throws SecurityException {
	    String uri = qname.getURI();
	    if(!uri.startsWith("java:"))
		return null;
	    String className = uri.substring(5);

	    // detect auto arg0 specification:	java:className?localname=uri
	    QName autoArg0 = null;
	    int mark = className.indexOf('?');
	    if(mark > 0) {
		int eq = className.indexOf('=', mark);
		String argName = className.substring(mark+1, eq);
		String argUri = className.substring(eq + 1);
		autoArg0 = QName.get(argUri, argName);
		className = className.substring(0, mark);
	    }

	    if(allowedClasses != null && !allowedClasses.contains(className))
		throw new SecurityException("security restriction on class "+className);

	    try {
		Class fclass = Class.forName(className);
		Prototype[] protos;
		int kept = 0;
		String name = Util.camelCase(qname.getLocalName(), false);
		if(trace)
		    System.err.println("found Java class "+fclass+" for function "+name);
		if(name.equals("new")) {
		    // look for constructors 
		    int mo = fclass.getModifiers();
		    if( ! Modifier.isPublic(mo) ||
			Modifier.isAbstract(mo) || Modifier.isInterface(mo))
			return null;	// instantiation will fail

		    Constructor[] constructors = fclass.getConstructors();
		    protos = new Prototype[constructors.length];
		    for (int c = 0; c < constructors.length; c++) {
			protos[kept] = convertConstructor( qname, constructors[c] );
			if(protos[kept] != null) {
			    if(trace)
				System.err.println("detected constructor "+protos[kept]);
			    ++ kept;
			}
		    }
		}
		else {
		    Method[] methods = fclass.getMethods();
		    protos = new Prototype[methods.length];
		    for(int m = 0; m < methods.length; m++)
			// match name without conversion:
			if( methods[m].getName().equals(name)) {
			    protos[kept] = convertPrototype(qname, methods[m], autoArg0);
			    if(protos[kept] != null) {
				if(trace)
				    System.err.println("detected method "+protos[kept]);
				++ kept;
			    }
			}
		}
		if(kept > 0)
		    return new JavaFunction(protos, kept);
	    }
	    catch (ClassNotFoundException e) {
		if(trace) System.err.println("*** class not found "+className);
	    }
	    catch (Exception ie) {
		ie.printStackTrace();	// abnormal: report
	    }
	    return null;
	}
    }

    static Prototype convertPrototype( QName qname, Method method, QName autoArg0 ) {
	if( !Modifier.isPublic(method.getModifiers()) )
	    return null;
	Class[] params = method.getParameterTypes();
	Prototype proto = new Prototype(qname, null, method, null);
	proto.autoArg0 = autoArg0;

	int paramCnt = params.length, cv = 0;
	if(autoArg0 == null && !Modifier.isStatic(method.getModifiers())) {
	    // add a first argument of type wrappedObject
	    proto.arg("p0", Type.WRAPPED_OBJECT);
	    paramCnt ++;
	}
	proto.conversions = new int[paramCnt];
	if(paramCnt > params.length)
	    proto.conversions[ cv++ ] = WRAPPED_OBJECT;
	for(int p = 0; p < params.length; p++) {
	    int type = convertType(params[p]);
	    proto.conversions[ cv++ ] = type;
	    proto.arg("p"+(p+1), typeTable[type] );
	}
	proto.resultConversion = convertType( method.getReturnType() );
	proto.returnType = typeTable[proto.resultConversion];
	return proto;
    }

    static Prototype convertConstructor( QName qname, Constructor constr ) {
	if( !Modifier.isPublic(constr.getModifiers()) )
	    return null;
	Class[] params = constr.getParameterTypes();
	Prototype proto = new Prototype(qname, null, null, constr);
	proto.conversions = new int[params.length];
	for(int p = 0; p < params.length; p++) {
	    int type = convertType(params[p]);
	    proto.conversions[ p ] = type;
	    proto.arg("p"+(p+1), typeTable[type]);
	}
	proto.returnType = Type.WRAPPED_OBJECT;
	return proto;
    }

    public static int convertType(Class javaType) {
	if(javaType == String.class)
	    return STRING;
	if(javaType == Node.class)
	    return NODE;
	if(javaType == boolean.class || javaType == Boolean.class)
	    return BOOLEAN;
	if(javaType == double.class || javaType == Double.class)
	    return DOUBLE;
	if(javaType == float.class || javaType == Float.class)
	    return FLOAT;
	if(javaType == long.class || javaType == Long.class)
	    return INTEGER;
	if(javaType == int.class || javaType == Integer.class)
	    return INT;
	if(javaType == short.class || javaType == Short.class)
	    return SHORT;
	if(javaType == byte.class || javaType == Byte.class)
	    return BYTE;
	if(javaType == char.class || javaType == Character.class )
	    return CHAR;
	if(javaType == void.class)
	    return NONE;
	if(javaType == Enumeration.class)
	    return ENUMERATION;
	if(javaType == Vector.class)
	    return VECTOR;
	if(javaType == ArrayList.class)
	    return ARRAYLIST;

	if(javaType.isArray()) {
	    // apparently no way to get the array item type:
	    String name = javaType.getName();
	    if(name.equals("[Ljava.lang.String;"))
		return STRING_ARRAY;
	    if(name.equals("[Lnet.xfra.qizxopen.xquery.dm.Node;"))
		return NODE_ARRAY;
	    if(name.equals("[D"))
		return DOUBLE_ARRAY;
	    if(name.equals("[F"))
		return FLOAT_ARRAY;
	    if(name.equals("[J"))
		return INTEGER_ARRAY;
	    if(name.equals("[I"))
		return INT_ARRAY;
	    if(name.equals("[S"))
		return SHORT_ARRAY;
	    if(name.equals("[B"))
		return BYTE_ARRAY;
	    if(name.equals("[C"))
		return CHAR_ARRAY;
	    return WRAPPED_OBJECT_ARRAY;
	}
	return WRAPPED_OBJECT;
    }

    public static Value convertResult( Object value, int conversion) {
	if(value == null)
	    return Value.empty;
	switch(conversion) {
	    case NONE:
		return Value.empty;   
	    case STRING:
		return new SingleString( (String) value );
	    case BOOLEAN:
		return new SingleBoolean( ((Boolean) value).booleanValue() );
	    case DOUBLE:
		return new SingleDouble( ((Double) value).doubleValue() );
	    case FLOAT: 
		return new SingleFloat( ((Float) value).floatValue() );
	    case INTEGER:
		return new SingleInteger( ((Long) value).longValue() );
	    case INT:
		return new SingleInteger( ((Integer) value).intValue(), Type.INT );
	    case SHORT:
		return new SingleInteger( ((Short) value).intValue(), Type.SHORT );
	    case BYTE:
		return new SingleInteger( ((Byte) value).intValue(), Type.BYTE );
	    case CHAR:
		return new SingleInteger( ((Byte) value).intValue(), Type.INTEGER );
	    case NODE:
		return new SingleNode( (Node) value );
	    case WRAPPED_OBJECT:
		return new SingleWrappedObject( value );
		
	    case STRING_ARRAY: {
		String[] result = (String[]) value;
		return new StringArraySequence(result, result.length);
	    }
	    case DOUBLE_ARRAY: {
		double[] result = (double[]) value;
		return new FloatArraySequence(result, result.length);
	    }
	    case FLOAT_ARRAY: {
		float[] result = (float[]) value;
		return new FloatArraySequence(result, result.length);
	    }
	    case INTEGER_ARRAY: {
		long[] result = (long[]) value;
		return new IntegerArraySequence(result, result.length);
	    }
	    case INT_ARRAY: {
		int[] result = (int[]) value;
		return new IntegerArraySequence(result, result.length);
	    }
	    case SHORT_ARRAY: {
		short[] result = (short[]) value;
		return new IntegerArraySequence(result, result.length);
	    }
	    case BYTE_ARRAY: {
		byte[] result = (byte[]) value;
		return new IntegerArraySequence(result, result.length);
	    }
	    case CHAR_ARRAY: {
		char[] result = (char[]) value;
		return new IntegerArraySequence(result, result.length);
	    }
	    case NODE_ARRAY: {
		Node[] result = (Node[]) value;
		return result == null ? Value.empty
		    : new ArraySequence(result, result.length);
	    }
	    case WRAPPED_OBJECT_ARRAY: {
		Object[] result = (Object[]) value;
		return result == null ? Value.empty
		    : new ObjectArraySequence(result, result.length);
	    }
	    case ENUMERATION: {
		return new ObjectArraySequence((Enumeration) value);
	    }
	    case VECTOR: {
		return new ObjectArraySequence((Vector) value);
	    }
	    case ARRAYLIST: {
		return new ObjectArraySequence((ArrayList) value);
	    }
	    default:
		throw new IllegalArgumentException("bad result conversion");
	}
    }

    

    public JavaFunction( Prototype[] protos, int count ) {
        prototypes = new Prototype[count];
	System.arraycopy(protos, 0, prototypes, 0, count);
    }

    public net.xfra.qizxopen.xquery.fn.Prototype[] getProtos()
	{ return prototypes; } // dummy here


    public static class Call extends Function.Call
    {
	GlobalVariable autoArg0;

	public void dump( ExprDump d ) {
	    d.header( this, getClass().getName() );
	    d.display("prototype", prototype.toString());
	    if(autoArg0 != null) d.display("auto arg", autoArg0);
	    d.display("actual arguments", args);
	}

	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    
	    Prototype proto = (Prototype) prototype;

	    if(proto.method != null) {
		boolean isStatic = Modifier.isStatic(proto.method.getModifiers())
		                   || autoArg0 != null;
		
		Object target = null;
		if(!isStatic) 
		    target = convertArg(args[0], proto.conversions[0], focus, context);
		else if(autoArg0 != null) {
		    Value v = context.loadGlobal(autoArg0);
		    target = ((SingleWrappedObject) v).getObject();
		}

		int argcnt = isStatic? proto.argCnt : (proto.argCnt - 1);
		int shift = isStatic? 0 : 1;
		Object[] params = new Object[argcnt];
		for(int a = 0; a < argcnt; a++)
		    params[a] = convertArg( args[a + shift], proto.conversions[a + shift],
					    focus, context);
		try {
		    Object result = proto.method.invoke(target, params);
		    if(trace) System.err.println("calling Java method: "+ proto.method +
						 "\n for "+ proto);
		    return convertResult(result, proto.resultConversion);
		}
		catch (InvocationTargetException iex) {
		    //ex.printStackTrace();
		    Exception cause = (Exception) iex.getCause();
		    context.error(this,
		     new EvalException("exception in extension function: "+cause, cause));
		}
		catch (Exception ex) {
		    context.error(this,
		       new EvalException("invocation of extension function: "+ex, ex));
		}
	    }
	    else {
		Object[] params = new Object[proto.argCnt];
		for(int a = 0; a < proto.argCnt; a++)
		    params[a] = convertArg( args[a], proto.conversions[a], focus, context);
		try {
		    if(trace)
			System.err.println("calling Java constructor: "+proto.constructor+
					   "\n for "+ proto);
		    Object result = proto.constructor.newInstance(params);
		    return convertResult(result, WRAPPED_OBJECT);
		}
		catch (Exception ex) {
		    //ex.printStackTrace();
		    context.error(this, "invocation of extension constructor: "+ex);
		}
	    }
	    return Value.empty;
	}

	Object convertArg( Expression arg, int conversion,
			   Focus focus, EvalContext context ) throws XQueryException {
	    switch(conversion) {
	    case STRING:
		return arg.evalAsString(focus, context);
	    case BOOLEAN:
		return new Boolean( arg.evalAsBoolean(focus, context) );
	    case DOUBLE:
		return new Double( arg.evalAsDouble(focus, context) );
	    case FLOAT: 
		return new Float( arg.evalAsFloat(focus, context) );
	    case INTEGER:
		return new Long( arg.evalAsInteger(focus, context) );
	    case INT:
		return new Integer( (int) arg.evalAsInteger(focus, context) );
	    case SHORT:
		return new Short( (short) arg.evalAsInteger(focus, context) );
	    case BYTE:
		return new Byte( (byte) arg.evalAsInteger(focus, context) );
	    case NODE:
		return arg.evalAsNode(focus, context);
	    case WRAPPED_OBJECT:
		Item item = arg.evalAsItem(focus, context);
		if(item instanceof StringValue)
		    return ((StringValue) item).asString();
		return ((WrappedObjectValue) item).getObject();
	    case CHAR:
		return new Character( (char) arg.evalAsInteger(focus, context) );

	    case STRING_ARRAY:
		return StringArraySequence.expand( arg.eval(focus, context) );
	    case DOUBLE_ARRAY:
		return FloatArraySequence.expandDoubles( arg.eval(focus, context) );
	    case FLOAT_ARRAY:
		return FloatArraySequence.expandFloats( arg.eval(focus, context) );
	    case INTEGER_ARRAY:
		return IntegerArraySequence.expandIntegers( arg.eval(focus, context) );
	    case INT_ARRAY:
		return IntegerArraySequence.expandInts( arg.eval(focus, context) );
	    case SHORT_ARRAY:
		return IntegerArraySequence.expandShorts( arg.eval(focus, context) );
	    case BYTE_ARRAY:
		return IntegerArraySequence.expandBytes( arg.eval(focus, context) );
	    case CHAR_ARRAY:
		return IntegerArraySequence.expandChars( arg.eval(focus, context) );
	    case NODE_ARRAY:
		return ArraySequence.expandNodes( arg.eval(focus, context) );
	    case WRAPPED_OBJECT_ARRAY:
		return ObjectArraySequence.expand( arg.eval(focus, context) );
	    default:
		throw new IllegalArgumentException("bad conversion");
	    }
	}
    }
} // end of class JavaFunction
