/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.fn;
import net.xfra.qizxopen.xquery.impl.*;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.dm.XMLEventReceiver;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.*;
import net.xfra.qizxopen.xquery.op.Expression;
import net.xfra.qizxopen.xquery.dt.SingleBoolean;
import net.xfra.qizxopen.xquery.dt.SingleInteger;
import net.xfra.qizxopen.xquery.dt.SingleFloat;
import net.xfra.qizxopen.xquery.dt.SingleDouble;
import net.xfra.qizxopen.xquery.dt.SingleString;
import net.xfra.qizxopen.xquery.dt.SingleNode;

/**
 *	Function descriptor.
 *	Each built-in function is a subclass of Function and defines a list of
 *	implemented signatures.
 */
public abstract class Function {

    /**
     *	Checks the arguments and return a function call instantiated with actual arguments.
     *  <p>Specific to each function.
     */
    public Expression  staticCheck( StaticContext context, Expression[] arguments,
				    Expression subject ) {
        return context.resolve(getProtos(), arguments, subject);
    }

    /**
     *	Instantiation of the function/operator call.
     */
    public abstract static class Call extends Expression {
	public Expression[] args;	// actual args
	public Prototype prototype;

	public boolean visit( Visitor v ) {
	    return v.examine(this) && v.visit(args);
	}

	public void dump( ExprDump d ) {
	    d.header( this, "Call "+prototype );
	    d.display("args", args);
	}

	public void compilationHook() {
	}
    }

    public QName getName() {
	return getProtos()[0].qname;
    }

    // redefined in each implementation
    public abstract Prototype[] getProtos();

    public abstract static class BoolCall extends Call {

	public abstract boolean evalAsBoolean( Focus focus, EvalContext context)
	    throws XQueryException;

	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    return new SingleBoolean( evalAsBoolean( focus, context ) );
	}

	public boolean evalAsEffectiveBoolean(Focus focus, EvalContext context)
	    throws XQueryException {
	    return evalAsBoolean( focus, context );
	}
    }

    public abstract static class IntegerCall extends Call {
	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    return new SingleInteger( evalAsInteger( focus, context ) );
	}
	public abstract long evalAsInteger( Focus focus, EvalContext context)
	    throws XQueryException;
    }

    public abstract static class FloatCall extends Call {
	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    return new SingleFloat( evalAsFloat( focus, context ) );
	}
	public abstract float evalAsFloat( Focus focus, EvalContext context)
	    throws XQueryException;
    }

    public abstract static class DoubleCall extends Call {
	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    return new SingleDouble( evalAsDouble( focus, context ) );
	}
	public abstract double evalAsDouble( Focus focus, EvalContext context)
	    throws XQueryException;
    }

    public abstract static class StringCall extends Call {

	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    return new SingleString( evalAsString( focus, context ) );
	}

	public abstract String evalAsString( Focus focus, EvalContext context)
	    throws XQueryException;
    }

    public abstract static class OptStringCall extends Call {

	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    String s = evalAsOptString( focus, context );
	    return s == null ? Value.empty : new SingleString(s);
	}

	public abstract String evalAsOptString( Focus focus, EvalContext context)
	    throws XQueryException;

	public String evalAsString( Focus focus, EvalContext context)
	    throws XQueryException {
	    String s = evalAsOptString( focus, context );
	    if(s == null)
		context.error(this, Type.ERR_EMPTY_UNEXPECTED);
	    return s;
	}
    }

    public abstract static class TreeCall extends Call {

	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    Node node = evalAsNode( focus, context );
	    return node == null ? Value.empty : new SingleNode(node);
	}

	public Node evalAsNode( Focus focus, EvalContext context )
	    throws XQueryException {
	    EventDrivenBuilder builder = new EventDrivenBuilder();
	    evalAsEvents( builder, focus, context );
	    return builder.crop();
	}

	public abstract void evalAsEvents( XMLEventReceiver output, Focus focus,
					   EvalContext context) throws XQueryException;
    }

}

