/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.dm;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.dm.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dt.SingleString;
import net.xfra.qizxopen.xquery.dt.NodeType;
import java.text.Collator;
import java.util.Date;

/**
 *	Basic in-core Implementation of XQuery Data Model.
 *	Used for constructed tree fragments.
 */
public class CoreDataModel
{
    public static int CreatedElements = 0, CreatedText = 0;


    static abstract class SNode extends NodeBase implements Node {

	public Element  parent;
	public SNode    nextSibling;
	public int      order = -1;		//TODO

	public String toString() {
	    return "Core "+getNodeKind()+" "+
		(getNodeName() != null ? (" name="+getNodeName()) : getStringValue());
	}

	public String getBaseURI() {
	    QName base = QName.get(Namespace.XML, "base");
	    for(SNode n = this; n != null; n = n.parent) {
		Node attr  = n.getAttribute(base);
		if(attr != null)
		    return attr.getStringValue();
	    }
	    return null;
	}

	public String getDocumentURI() {
	    return null;
	}

	public QName  getNodeName() {
	    return null;
	}

	public Node getParent() {
	    return parent;
	}

	public Value  getTypedValue() { 
	    return new SingleString( getStringValue() ); //TODO schemas	
	}

	public ItemType  getType() {
	    return NodeType.getTypeByKind( getNature() );
	}

	SNode getFirstChild() {
	    return null;
	}

	public Node  getDocument() {
	    return getTop();
	}

	public int orderCompare( net.xfra.qizxopen.dm.Node node ) {
	    if(node == this)
		return 0;
	    if( !(node instanceof SNode) )
		return docPosition() - node.docPosition();
	    SNode that = (SNode) node, top = getTop(), thatTop = that.getTop();
	    if(top != thatTop)
		return top.docPosition() - thatTop.docPosition();
	    // if possible, use order stamps:
	    if(order >= 0 && that.order >= 0)
		return Util.comparison(order - that.order);
	    // inefficient, but not supposed to be used heavily:
	    for( ; that != null; that = that.getNodeNext())
		if(that == this)
		    return 1;
	    return -1;
	}

	public boolean contains( net.xfra.qizxopen.dm.Node node ) {
	    if( !(node instanceof SNode) )
		return false;
	    
	    for( SNode n = (SNode) node; n != null; n = n.parent)
		if(n == this)
		    return true;
	    return false;
	}

	public boolean equals( Object that ) {
	    return this == that;
	}

	public boolean deepEqual( Item item, Collator collator ) {
	    if(! (item instanceof Node) )
		return false;
	    try {
		return deepEq((Node) item, collator);
	    } catch (Exception e) {
		return false;	//should not happen
	    }
	}

	public boolean deepEq( net.xfra.qizxopen.dm.Node that, Collator collator )
	    throws XQueryException {
	    int kind = getNature();
	    if(kind != that.getNature() || getNodeName() != that.getNodeName() )
		return false;
	    if(kind == Node.DOCUMENT)
		return ((Element) this).contentEq((Element) that, collator);
	    else if(kind == Node.ELEMENT)
		return ((Element) this).attributesEq((Element) that, collator)
		    && ((Element) this).contentEq((Element) that, collator);
	    else // compare by string value
		return compareTo( (Node) that, collator, 0) == 0;
	}

	public  SNode  getNodeNext() {	// in document order
 	    SNode kid = getFirstChild();
 	    return kid != null ? kid : getNodeAfter();
	}

	public  SNode  getNodeAfter() {	// in document order
	    SNode node = this, nxt;
	    while((nxt = node.nextSibling) == null) {
		SNode parent = node.parent;
		if (parent == null)
		    return null;
		node = parent;
	    }
	    return nxt;
	}

	SNode getTop() {
	    SNode n = this;
	    for( ; n.parent != null; )
		n = n.parent;
	    return n;
	}

	public int docPosition() {
	    return getTop().hashCode();	    
	}

	public int getDefinedNSCount() {
	    return 0;
	}

	public void addNamespace( String prefix, String value ) {
	    throw new RuntimeException("not an element");
	}

	

	public Value getAncestors( NodeTest nodeTest ) {
	    return new AncestorsOrSelf(this, nodeTest);
	}

	public Value getAncestorsOrSelf( NodeTest nodeTest ) {
	    return new AncestorsOrSelf(this, nodeTest);
	}

	public Value getParent( NodeTest nodeTest ) {
	    return new Parent(this, nodeTest);
	}

	public Value getChildren( NodeTest nodeTest ) {
	    return new Children(this, nodeTest);
	}

	public Value getDescendants( NodeTest nodeTest ) {
	    return new Descendants(this, nodeTest);
	}

	public Value getDescendantsOrSelf( NodeTest nodeTest ) {
	    return new DescendantsOrSelf(this, nodeTest);
	}

	public Value getAttributes( NodeTest nodeTest ) {
	    return new Attributes(this, nodeTest);
	}

	public Value getFollowingSiblings( NodeTest nodeTest ) {
	    return new FollowingSiblings(this, nodeTest);
	}

	public Value getPrecedingSiblings( NodeTest nodeTest ) {
	    return new PrecedingSiblings(this, nodeTest);
	}

	public Value getFollowing( NodeTest nodeTest ) {
	    return new Following(this, nodeTest);
	}

	public Value getPreceding( NodeTest nodeTest ) {
	    return new Preceding(this, nodeTest);
	}

	public void setNextSibling( SNode sibling ) {
	    nextSibling = sibling;
	    if(sibling != null)
		sibling.parent = parent;
	}

	

	public net.xfra.qizxopen.dm.Node  document() {
	    return getDocument();
	}

	public net.xfra.qizxopen.dm.Node   parent() {
	    return getParent();
	}

	public net.xfra.qizxopen.dm.Node  attribute( QName name ) {
	    return getAttribute(name);
	}

	public int compareStringValues( net.xfra.qizxopen.dm.Node that, Collator collator ) {
	    String s1 = this.getStringValue(), s2 = that.getStringValue();
	    return collator != null ? collator.compare(s1, s2) : s1.compareTo(s2);
	}

	public NodeSequence children() {
	    return (NodeSequence) getChildren();
	}

	public NodeSequence attributes() {
	    return new Attributes(this, null);
	}

	public NodeSequence ancestors( NodeTest nodeTest ) {
	    return new AncestorsOrSelf(this, nodeTest);
	}

	public NodeSequence ancestorsOrSelf( NodeTest nodeTest ) {
	    return new AncestorsOrSelf(this, nodeTest);
	}

	public NodeSequence parent( NodeTest nodeTest ) {
	    return new Parent(this, nodeTest);
	}

	public NodeSequence children( NodeTest nodeTest ) {
	    return new Children(this, nodeTest);
	}

	public NodeSequence descendants( NodeTest nodeTest ) {
	    return new Descendants(this, nodeTest);
	}

	public NodeSequence descendantsOrSelf( NodeTest nodeTest ) {
	    return new DescendantsOrSelf(this, nodeTest);
	}

	public NodeSequence attributes( NodeTest nodeTest ) {
	    return new Attributes(this, nodeTest);
	}

	public NodeSequence namespaces( boolean inScope ) {
	    return new Namespaces(this);
	}

	public NodeSequence followingSiblings( NodeTest nodeTest ) {
	    return new FollowingSiblings(this, nodeTest);
	}

	public NodeSequence precedingSiblings( NodeTest nodeTest ) {
	    return new PrecedingSiblings(this, nodeTest);
	}

	public NodeSequence following( NodeTest nodeTest ) {
	    return new Following(this, nodeTest);
	}

	public NodeSequence preceding( NodeTest nodeTest ) {
	    return new Preceding(this, nodeTest);
	}

	

	public char[] getChars() {
	    throw new RuntimeException("getChars of a non-atom");
	}

	public Object getAtomValue() {
	    return this;
	}

	public int   getAtomType() {
	    Object value = getAtomValue();
	    if(value instanceof String)
		return TYPE_STRING;
	    if(value instanceof Long)
		return TYPE_INTEGER;
	    if(value instanceof Date)
		return TYPE_DATETIME;
	    if(value instanceof Double)
		return TYPE_DOUBLE;
	    if(value instanceof Boolean)
		return TYPE_BOOLEAN;
	    if(value.getClass().isArray() )	// OOPS
		return TYPE_BINARY;
	    throw new RuntimeException("improper atom value "+value);
	}

	public double getDoubleValue() throws DataModelException {
	    Object obj = getAtomValue();
	    if(obj instanceof Double)
		return ((Double) obj).doubleValue();
	    try {
		return Double.parseDouble(getStringValue());
	    }
	    catch (NumberFormatException e) {
		throw new DataModelException("atom cannot be converted to double", e);
	    }
	}

	public long getIntegerValue() throws DataModelException {
	    Object obj = getAtomValue();
	    if(obj instanceof Long)
		return ((Long) obj).longValue();
	    try {
		return Long.parseLong(getStringValue());
	    }
	    catch (NumberFormatException e) {
		throw new DataModelException("atom cannot be converted to integer", e);
	    }
	}
    }

    static class Element extends SNode {

	QName name;
	SNode firstChild;
	Attribute attributes;

	Element( QName name ) {
	    ++ CreatedElements;
	    this.name = name;
	}

	public String getNodeKind() {
	    return "element";
	}

	public String getStringValue() {
	    StringBuffer buf = new StringBuffer();
	    for(SNode kid = firstChild; kid != null; kid = kid.nextSibling)
		recStringValue(kid, buf);
	    return buf.toString();
	}

	static void recStringValue(SNode node, StringBuffer buf) {
	    if(node.isElement()) {
		for(SNode kid = ((Element) node).firstChild; kid != null; kid = kid.nextSibling)
		    recStringValue(kid, buf);
	    }
	    else if(node.getNature() == Node.TEXT)	// ignore comments and PI
		buf.append( node.getStringValue() );
	}

	public Value getChildren() {
	    return new SSequence(firstChild);
	}

	public QName  getNodeName() {
	    return name;
	}

	SNode getFirstChild() {
	    return firstChild;
	}

	public Value getAttributes() {
	    return new Attributes(this, null);
	}

	public Value getNamespaces( boolean inScope ) {
	    //TODO (or maybe not) implement inScope
	    return new Namespaces(this);
	}

	public int getDefinedNSCount() {
	    int cnt = 0;
	    for(SNode attr = attributes; attr != null; attr = attr.nextSibling)
		if(attr.getNature() == NAMESPACE)
		    ++ cnt;
	    return cnt;
	}

	// return first NSnode on element or enclosing elements
	// (NS are expanded on nodes that define at least one NS)
	private SNode getFirstNS() {
	    for(Element e = this; e != null; e = e.parent) {
		for(SNode attr = e.attributes; attr != null; attr = attr.nextSibling)
		    if(attr instanceof NSNode)
			return attr;
	    }
	    return null;
	}

	public String getNsPrefix( String nsuri ) {
	    for(SNode attr = getFirstNS(); attr != null; attr = attr.nextSibling)
		if(attr instanceof NSNode) {
		    NSNode ns = (NSNode) attr;
		    if(nsuri.equals(ns.value))
			return ns.name.getLocalName();
		}
	    return null;
	}

	public String getNsUri( String prefix ) {
	    for(SNode attr = getFirstNS(); attr != null; attr = attr.nextSibling)
		if(attr instanceof NSNode) {
		    NSNode ns = (NSNode) attr;
		    if(prefix.equals(ns.name.getLocalName()))
			return ns.getStringValue();
		}
	    return null;
	}

	

	public int   getNature() {
	    return Node.ELEMENT;
	}

	public boolean isElement() {
	    return true;
	}

	public boolean isWhitespace() {
	    return false;
	}

	boolean attributesEq( Element that, Collator collator ) throws XQueryException {
	    Value oattr = that.getAttributes();
	    int acnt = 0;
	    for(; oattr.next(); ++ acnt) {
		Node oatt = oattr.asNode();
		SNode att = (SNode) getAttribute(oatt.getNodeName());
		if( att == null || att.compareTo( oatt, collator, 0) != 0)
		    return false;
	    }
	    return getAttributeCount() == acnt;
	}

	boolean contentEq( Element that, Collator collator ) throws XQueryException {
	    Value okids = that.getChildren();
	    for(SNode kid = firstChild; kid != null; kid = kid.nextSibling) {
		int kidkind = kid.getNature(), okidkind;
		if(kidkind == Node.COMMENT || kidkind == Node.PROCESSING_INSTRUCTION)
		    continue;
		Node okid;
		for( ; ; ) {
		    if(!okids.next())
			return false;
		    okid = okids.asNode();
		    if( (okidkind = okid.getNature()) != Node.COMMENT &&
			 okidkind != Node.PROCESSING_INSTRUCTION)
			break;
		}
		// we have 2 comparable kids:
		if(!kid.deepEq( okid, collator ))
		   return false;
	    }
	    // just check that there are no more kids in 'that'
	    return !okids.next();
	}

	public int getAttributeCount() {
	    int cnt = 0;
	    for(SNode attr = attributes; attr != null; attr = attr.nextSibling)
		if(attr.getNature() == ATTRIBUTE)
		    ++ cnt;
	    return cnt;
	}

	public Node  getAttribute( QName name ) {
	    for(SNode attr = attributes; attr != null; attr = attr.nextSibling)
		if( attr.getNodeName() == name && attr.getNature() == Node.ATTRIBUTE )
		    return attr;
	    return null;
	}

	public void setAttributes( Attribute attr ) {
	    attributes = attr;
	    for( ; attr != null; attr = (Attribute) attr.nextSibling)
		attr.parent = this;
	}

	// caution: used also for NS nodes
	public void addAttribute( Attribute attr ) {
	    boolean hasNS = false;
	    int nnat = attr.getNature();
	    SNode inscopeNS = (nnat == Node.NAMESPACE)? getFirstNS() : null;
	    if( attributes == null )
		attributes = attr;
	    else // need to check duplicates anyway
		for(Attribute last = attributes; ; last = (Attribute) last.nextSibling) {
		    int nat = last.getNature();
		    if(nat == Node.NAMESPACE)
			hasNS = true;
		    if(last.name == attr.name && nat == nnat) {
			last.value = attr.value;
			return;
		    }
		    if(last.nextSibling == null) {
			last.nextSibling = attr;
			break;
		    }
		}
	    if(!hasNS && nnat == Node.NAMESPACE) {
		// need to copy in-scope NS: (without overwriting the new one)
		for( ; inscopeNS != null; inscopeNS = inscopeNS.nextSibling)
		    if(inscopeNS.getNature() == Node.NAMESPACE) {
			NSNode ns = (NSNode) inscopeNS;
			if(ns.name == attr.name)
			    continue;
			NSNode nns = new NSNode(ns.name, ns.getStringValue());
			nns.nextSibling = attributes;
			attributes = nns;
			nns.parent = this;
		    }
	    }
	    attr.parent = this;
	}

	public void addText( String value ) {
	    throw new RuntimeException("addText on an element");
	}
    }

    static class Document extends Element {
	Document() {
	    super(null);
	}

	public String getNodeKind() {
	    return "document";
	}
	
	public int   getNature() {
	    return Node.DOCUMENT;
	}
    }

    static class TextNode extends SNode
    {
	Object value; 

	TextNode( Object value ) {
	    this.value = value;
	    ++ CreatedText;
	}

	TextNode() {
	    this("");
	}

	public String getNodeKind() {
	    return "text";
	}

	public int getNature() {
	    return Node.TEXT;
	}

	public void addText( String value ) {
	    String thisText = getStringValue();
	    if( thisText == null || thisText.length() == 0)
		this.value = value;
	    else
		this.value = thisText + value;
	}

	public String getStringValue() {
	    return value.toString();
	}

	public Value getChildren() {
	    return SSequence.empty;
	}

	public Value getAttributes() {
	    return SSequence.empty;
	}

	public Value getNamespaces( boolean inScope ) {
	    return SSequence.empty;
	}

	public String getNsPrefix( String nsuri ) {
	    return parent == null ? null : parent.getNsPrefix(nsuri);
	}

	public String getNsUri( String prefix ) {
	    return parent == null ? null : parent.getNsUri(prefix);
	}

	public boolean isElement() {
	    return false;
	}

	public boolean isWhitespace() {
	    return false;	//TODO
	}

	public Node  getAttribute( QName name ) {
	    return null;
	}
    }

    static class PINode extends TextNode
    {
	String target = "";

	PINode( String target, String value ) {
	    super(value);
	    this.target = target;
	}

	public String getNodeKind() {
	    return "processing-instruction";
	}

	public int getNature() {
	    return Node.PROCESSING_INSTRUCTION;
	}

	public QName  getNodeName() {
	    return QName.get(Namespace.NONE, target);
	}
    }

    static class CommentNode extends TextNode
    {
	CommentNode( String value ) {
	    super(value);
	}

	public String getNodeKind() {
	    return "comment";
	}

	public int getNature() {
	    return Node.COMMENT;
	}
    }

    // Attribute and namespace nodes: 
    static class Attribute extends TextNode
    {
	QName name;

	Attribute(QName name) {
	    this.name = name;
	}

	public QName  getNodeName() {
	    return name;
	}

	public String getNodeKind() {
	    return "attribute";
	}

	public int getNature() {
	    return Node.ATTRIBUTE;
	}
    }

    // namespace Node:
    static class NSNode extends Attribute {

	NSNode(QName name, String value) {
	    super(name); this.value = value;
	}

	NSNode(String prefix) {
	    super(QName.get(Namespace.NONE, prefix));
	}

	public String getNodeKind() {
	    return "namespace";
	}

	public int getNature() {
	    return Node.NAMESPACE;
	}
    }

    // sibling sequence
    static class SSequence extends NodeSequenceBase implements NodeSequence
    {
	boolean started = false;
	SNode current, first;

	SSequence( SNode first ) {
	    this.first = this.current = first;
	}

	public Value  bornAgain() {
	    return new SSequence( first );
	}

	public boolean next() {
	    if(started && current != null)
		current = current.nextSibling;
	    started = true;
	    return current != null;
	}

	public Node   asNode() {
	    return current;
	}

	

	public boolean nextNode() {
	    return next();
	}

	public net.xfra.qizxopen.dm.Node   currentNode() {
	    return current;
	}

	public NodeSequence  reborn() {
	    return (NodeSequence) bornAgain();
	}

	static SSequence empty = new SSequence(null);
    }

    static class Namespaces extends SSequence {
	Namespaces( SNode first ) {
	    super(first);
	    current = ((Element) first).attributes;
	}
	
	public boolean next() {
	    for(; current != null; ) {
		if(started)
		    current = current.nextSibling;
		started = true;
		if(current instanceof NSNode)
		    return true;
	    }
	    return false;
	}

	public Value  bornAgain() {
	    return new Namespaces( first );
	}
    }

    static abstract class TypedSequence extends SSequence {
	NodeTest nodeTest;

	TypedSequence( SNode first, NodeTest nodeTest ) {
	    super(first);
	    this.nodeTest = nodeTest;
	}

	boolean checkNode() {
	    return current != null &&
		(nodeTest == null || nodeTest.accepts(current));
	}
    }

    static class Parent extends TypedSequence
    {
	Parent( SNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	}

	public boolean next() {
	    if(started)
		return false;
	    started = true;
	    current = current.parent;
	    return checkNode();
	}
	
	public Value  bornAgain() {
	    return new Parent( first, nodeTest );
	}
    }

    static class AncestorsOrSelf extends TypedSequence
    {
	boolean started = false;

	AncestorsOrSelf( SNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	}

	public boolean next() {
	    for(; current != null; ) {
		if(started)
		    current = current.parent;
		started = true;
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public Value  bornAgain() {
	    return new AncestorsOrSelf( first, nodeTest );
	}
    }

    static class Ancestors extends AncestorsOrSelf
    {
	Ancestors( SNode first, NodeTest nodeTest ) {
	    super( first, nodeTest);
	    started = true;
	}

	public Value  bornAgain() {
	    return new Ancestors( first, nodeTest );
	}
    }

    static class Children extends TypedSequence
    {
	Children( SNode first, NodeTest nodeTest ) {
	    super( first, nodeTest);
	}

	public boolean next() {
	    for(; current != null; ) {
		if(!started)
		     current = current.getFirstChild();
		else current = current.nextSibling;
		started = true;
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public Value  bornAgain() {
	    return new Children( first, nodeTest );
	}
    }

    static class DescendantsOrSelf extends TypedSequence
    {
	Node lastNode;
	boolean started = false;

	DescendantsOrSelf( SNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	    lastNode = first.getNodeAfter();
	}

	public boolean next() {
	    for(; current != null; ) {
		if(started)
		    current = current.getNodeNext();
		started = true;
		if(current == lastNode)
		    current = null;
		else if(checkNode())
		    return true;
	    }
	    return false;
	}

	public Value  bornAgain() {
	    return new DescendantsOrSelf( first, nodeTest );
	}
    }

    static class Descendants extends DescendantsOrSelf
    {
	Descendants( SNode first, NodeTest nodeTest ) {
	    super( first, nodeTest);
	    started = true;
	}

	public Value  bornAgain() {
	    return new Descendants( first, nodeTest );
	}
    }

    static class FollowingSiblings extends TypedSequence
    {
	FollowingSiblings( SNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	}

	public boolean next() {
	    for( ; current != null; ) {
		current = current.nextSibling;
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public Value  bornAgain() {
	    return new FollowingSiblings( first, nodeTest );
	}
    }

    static class Following extends TypedSequence
    {
	Following( SNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	}

	public boolean next() {
	    for( ; current != null; ) {
		current = current.getNodeNext();
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public Value  bornAgain() {
	    return new Following( first, nodeTest );
	}
    }

    static class PrecedingSiblings extends TypedSequence
    {
	PrecedingSiblings( SNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	    current = first.parent == null ? null : first.parent.firstChild;
	}

	public boolean next() {
	    for( ; current != null; ) {
		if(started) 
		    current = current.nextSibling;
		started = true;
		if(current == first) {
		    current = null; return false;
		}
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public Value  bornAgain() {
	    return new PrecedingSiblings( first, nodeTest );
	}
    }

    static class Preceding extends TypedSequence
    {
	Preceding( SNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	    current = (SNode) first.getDocument();
	}

	public boolean next() {
	    for( ; current != null; ) {
		if(started) 
		    current = current.getNodeNext();
		started = true;
		if(current == first) {
		    current = null; return false;
		}
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public Value  bornAgain() {
	    return new Preceding( first, nodeTest );
	}
    }

    static class Attributes extends TypedSequence
    {
	Attributes( SNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	    if(first.getNature() == Node.ELEMENT)
		current = ((Element) first).attributes;
	}

	public Value  bornAgain() {
	    return new Attributes( first, nodeTest );
	}

	public boolean next() {
	    for( ; current != null; ) {
		if(started) 
		    current = current.nextSibling;
		started = true;
		if(current != null && current.getNature() == Node.ATTRIBUTE && checkNode())
		    return true;
	    }
	    return false;
	}
    }

    // --------------- static construction ------------------------------------------

    public static Document newDocumentNode() {
	return new Document();
    }

    public static Element newElement( QName name ) {
	return new Element(name);
    }

    public static Attribute newAttribute( QName name ) {
	return new Attribute(name);
    }

    public static NSNode newNSNode( String prefix ) {
	return new NSNode(prefix);
    }

    public static TextNode newTextNode() {
	return new TextNode();
    }

    public static PINode newPINode( String target ) {
	return new PINode(target, "");
    }

    public static CommentNode newCommentNode() {
	return new CommentNode("");
    }
}
