/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.util.time.DateTimeException;
import net.xfra.qizxopen.util.FileUtil;
import net.xfra.qizxopen.dm.*;
import net.xfra.qizxopen.xquery.impl.*;
import net.xfra.qizxopen.xquery.dm.Node;
import net.xfra.qizxopen.xquery.dt.*;

import java.io.PrintWriter;
import java.io.Writer;
import java.io.Reader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Vector;
import java.util.HashMap;
import java.util.HashSet;
import java.text.Collator;

/**
 *	Main interface to XML Query services.
 *	
 * <p>XQueryProcessor provides a static environment to compile a query from
 * some text source, and a dynamic environment (in particular a Document Manager)
 * to execute this query.
 * <p>The standard way for using it is as follows:<ul>
 * <li>Instantiate a XQueryProcessor using one of the constructors.
 * <li>compile a query using one form of <code>compileQuery</code>.
 * A message {@link Log} has to be instantiated and passed to these methods.
 * This yields a compiled <code>Query</code> object.
 * <li>Execute the query one or several times using one of the <code>executeQuery</code>
 * methods.
 *</ul>
 * <p>Miscellaneous optional informations can be set through specialized methods
 * for use in the static or dynamic query contexts:
 * global variable values, default collation, implicit timezone, document input, 
 * base URI, default serialization output.
 * <p>XQueryProcessors can share a common {@link ModuleManager} or
 * {@link DocumentManager}. It avoids redundant compilation or document loading in
 * an environment where many processors use the same queries or documents.
 * The way to achieve this is to create a "master" processor with its own ModuleManager and
 * DocumentManager and to use it in the proper constructor for each needed instance.
 */
public class XQueryProcessor
{
    // a shared predefined module with Qizx extensions:
    public static String    EXTENSIONS_URI = "net.xfra.qizxopen.ext.Qizx";
    public static Namespace EXTENSIONS_NS = Namespace.get(EXTENSIONS_URI);

    // **** to define *after* EXTENSIONS_URI!
    static PredefinedModule QizxPredefined = newQizxPredefinedModule();

    protected DocumentManager docMan;
    protected ModuleManager   moduleMan;
    protected String baseURI = ".";
    protected Value input;
    protected PredefinedModule predefined = QizxPredefined;
    protected HashMap globals = new HashMap();	// specified values for globals
    protected HashMap collations;
    protected HashMap properties = new HashMap();
    protected String defaultCollation = null;
    protected String implicitTimezone = null;
    protected PrintWriter defaultOutput = new PrintWriter(System.out, true);
    protected Log log;
    protected NSPrefixMapping extraNS;

    /**
     *	Creation without Module Manager and Document Manager.<p>These objects must be
     *	specified before execution.
     */
    public XQueryProcessor() {
	init();
    }
    
    /**
     *	Simple creation with private Module Manager and Document Manager.
     *	@param baseURI default base URI (also used for Document Manager)
     *	@param moduleBaseURI base URI used to resolve module locations
     *	@throws IOException thrown by ModuleManager or DocumentManager constructors.
     */
    public XQueryProcessor( String moduleBaseURI, String baseURI ) throws IOException { 
        setDocumentManager( new DocumentManager(baseURI) );
        setModuleManager( new ModuleManager(moduleBaseURI) );
	init();
    }
    
    /**
     *	Creates a new XQueryProcessor from a "master" processor, inheriting and sharing
     *	the document manager, module manager, predefined functions and global variables.
     *	@param master a XQueryProcessor used as template.
     */
    public XQueryProcessor( XQueryProcessor master ) {
        docMan = master.getDocumentManager();
        moduleMan = master.getModuleManager();
        predefined = master.predefined;
        collations = master.collations;
        defaultCollation = master.defaultCollation;
        implicitTimezone = master.implicitTimezone;
	properties = (HashMap) master.properties.clone();
	extraNS = (master.extraNS == null)? null : master.extraNS.copy();
	setSysProperty(":processor", this);
    }

    private void init() {
	// qizx: and xfn: are prefix aliases for extension functions
	predefineNamespace("qizx", EXTENSIONS_NS.getURI() );
	predefineNamespace("x", EXTENSIONS_NS.getURI() );
	setSysProperty(":processor", this);
	setSysProperty("version", "1.0");	// of XQuery
	setSysProperty("vendor", "Xavier Franc");
	setSysProperty("vendor-url", "http://www.xfra.net/qizxopen/");
	setSysProperty("product-name", "Qizx/open");
	setSysProperty("product-version", Version.get());
    }

    /**
     *	Gets the current version of the XML Query engine.
     */
    public String getVersion() {
        return Version.get();
    }
    
    /**
    *	Sets up the Module Manager.
    */
    public void setModuleManager( ModuleManager moduleManager ) {
        moduleMan = moduleManager;
    }
    /**
    *	Returns the current Module Manager.
    */
    public ModuleManager getModuleManager( ) {
        return moduleMan;
    }
    
    /**
    *	Defines the Document Manager.
    */
    public void setDocumentManager( DocumentManager documentManager ) {
        docMan = documentManager;
    }
    /**
    *	Returns the current Document Manager.
    */
    public DocumentManager getDocumentManager() {
        return docMan;
    }
    
    /**
    *	Defines the input() sequence by a document URI.
    *	@param docURI uri of a document to be opened or parsed and used as implicit input.
    *	This URI can be relative to the document base URI (if the default
    *	Document Manager is used).
    */
    public void setDocumentInput( String docURI ) throws XQueryException {
        if( docMan == null )
            throw new EvalException("no Document Manager defined");
        setInput( docMan.findDocument(docURI) );
    }
    
    /**
    *	Defines the input() sequence by a collection URI (not yet implemented).
    */
    public void setCollectionInput( String url ) {
        throw new RuntimeException("no setCollectionInput! "+url);
    }
    
    /**
    *	Defines a single node as the root of a document returned by
    *	the XQuery function input(). For internal usage.
    */
    public void setInput( Node inputDoc ) {
        input = new SingleNode( inputDoc );
    }
    
    /**
    *	Defines the default output channel for serialization.
    */
    public void setDefaultOutput( PrintWriter output ) {
        this.defaultOutput = output;
    }
    
    /**
    *	Defines a runtime log.
    */
    public void setLog(Log log) {
        this.log = log;
    }

    /**
     *	Defines a property, a named object that can be retrieved by the
     *	extension function x:get-app-property(name) or by any application.
     */
    public void setSysProperty(String name, Object property) {
	properties.put(name, property);
    }

    /**
     *	Retrieves a previously defined system property.
     */
    public Object getSysProperty(String name) {
	return properties.get(name);
    }

    /**
     *	Clears all predefined variables added, and initial values for globals.
     */
    public void resetDeclarations() {
	predefined = QizxPredefined;
	globals = new HashMap();
    }

    /**
    * Defines a global variable in the predefined static context.
    * The initial value must then be set by a variant of initGlobal() suitable to the type.
    *	@param varName local name of the variable.
    *	@param type assigned to the variable.
    */
    public void predefineGlobal( QName varName, Type type ) {
        if(predefined == QizxPredefined)
            predefined = newQizxPredefinedModule();	// ie clone
        predefined.defineGlobal(varName, type);
    }

    private static PredefinedModule newQizxPredefinedModule() {
	PredefinedModule pdm = new PredefinedModule();
	pdm.registerFunctionPlugger(
	 new PredefinedModule.BasicFunctionPlugger(EXTENSIONS_NS, EXTENSIONS_URI+ "%C"));
	return pdm;
    }
    
    /**
    * Defines a global variable (with 'local' namespace) in the predefined static context.
    * The initial value must then be set by a variant of initGlobal() suitable to the type.
    *	@param varName local name of the variable.
    *	@param type assigned to the variable.
    */
    public void predefineGlobal( String varName, Type type ) {
        predefineGlobal(QName.get(Module.LOCAL_NS, varName), type);
    }
    
    /**
     *	Sets an initial value for a global variable. This is not attached to a particular
     *	query, so it does not raise an error if the variable doesn't exist.
     *	@param varName qualified name of the variable (can be in the namespace
     *	of a module or 'local').
     *	@param value provided initial value: must be compatible with the declared
     *	type (not checked immediately). If the value is a string, an attempt to cast to
     *	the proper type will be made.
     */
    public void initGlobal( QName varName, Value value ) {
        globals.put(varName, value);
    }
    
    /**
    *	Utility for use with predefineGlobal and initGlobal: convert a NCName to a QName
    *	in 'local' namespace.
    */
    public static QName toLocalNS( String name ) {
        return QName.get(Module.LOCAL_NS, name);
    }
    
    /**
    *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
    */
    public void initGlobal( QName varName, boolean value ) {
        initGlobal(varName, new SingleBoolean(value));
    }
    
    /**
    *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
    */
    public void initGlobal( QName varName, long value ) {
        initGlobal(varName, new SingleInteger(value));
    }
    
    /**
    *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
    */
    public void initGlobal( QName varName, double value ) {
        initGlobal(varName, new SingleDouble(value));
    }
    
    /**
    *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
    */
    public void initGlobal( QName varName, String value ) {
        initGlobal(varName, new SingleString(value));
    }
    
    /**
    *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
    */
    public void initGlobal( QName varName, String[] value ) {
        SingleString[] v = new SingleString[value.length];
        for(int i = 0; i < value.length; i++)
            v[i] = new SingleString(value[i]);
        initGlobal( varName, new ArraySequence(v, v.length));
    }
    
    /**
    *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
    */
    public void initGlobal( QName varName, Object value ) {
        initGlobal( varName, new SingleWrappedObject(value));
    }
    
    /**
    *	Defines a namespace mapping visible by queries compiled with this processor.
    */
    public void predefineNamespace( String prefix, String uri) {
        if(extraNS == null)
            extraNS = new NSPrefixMapping();
        extraNS.addMapping(prefix, uri);
    }
    
    /**
    *	Registers a custom collation for use in the processed queries.
    */
    public void registerCollation(String uri, Collator collator) {
        if(collations == null)
            collations = new HashMap();
        collations.put(uri, collator);
    }
    
    /**
    *	Defines the URI of the default collation.
    */
    public void setDefaultCollation(String uri) {
        defaultCollation = uri;
    }
    
    /**
    *	Defines the implicit timezone in xs:duration format.
    *	@param duration for example "PT4H30M" or "-PT5H".
    *	If not specified, the implicit timezone is taken from the system default.
    */
    public void setImplicitTimezone(String duration) {
        implicitTimezone = duration;
    }
    
    /**
     *	Allow a Java class to be used as extension (more precisely, its public methods
     *	can be called as extension functions).
     *	<p><b>Caution:</b> using this method enforces an explicit control: all classes
     *	to be used as extensions must then be explicitly declared. This is a security
     *	feature.
     *	@param className fully qualified name of Java class,
     *	for example <pre>java.io.File</pre>
     */
    public void authorizeClass(String className) {
        predefined.authorizeJavaClass(className);
    }
    
    /**
    *	Parses and checks a query from a text input.
    *  Errors are reported through the log.
    *	@param textInput the actual text to parse
    *	@param uri of the query source (for messages), or null if not applicable.
    *	@param log message collector
    *  @return the parsed and type-checked query if no error is detected.
    *  @throws SyntaxException as soon as a syntax error is detected.
    *  @throws XQueryException at end of compilation if static errors have been
    *  detected (error details are reported through the Log).
    */
    public Query compileQuery( CharSequence textInput, String uri, Log log )
    throws XQueryException {
        if(textInput == null)
            throw new IllegalArgumentException("null textInput");
        if(uri == null)
            throw new IllegalArgumentException("null URI");
        if(log == null)
            throw new IllegalArgumentException("null log");
        if(moduleMan == null)
            throw new XQueryException("no Module Manager specified");
        if(docMan == null)
            throw new XQueryException("no Document Manager specified");
        Parser parser = new Parser(moduleMan);
        parser.setPredefinedModule(predefined);
        parser.setCollations(collations);
        if(extraNS != null)
            parser.setPredefinedNamespaces(extraNS);
	try {
	    int initialErrCnt = log.getErrorCount();
	    Query query = parser.parseQuery( textInput, uri, log );
	    query.setBaseURI( docMan.getBaseURI() );
	    int errCnt = log.getErrorCount() - initialErrCnt;
	    if(errCnt == 0)
		query.staticCheck( moduleMan, log );
	    log.flush();	// proper error display before raising an exception
	    errCnt = log.getErrorCount() - initialErrCnt;
	    if(errCnt > 0)
		throw new XQueryException("static analysis: "+ errCnt
					  + " error"+ (errCnt > 1? "s":""));
	    return query;
	} catch (SyntaxException e) {
	    throw new XQueryException("syntax: "+ e.getMessage());
	}
    }
    
    /**
    *	Helper for compiling a query from a stream. See {@link #compileQuery}
    *	@param input query source
    *	@param uri uri of the source (can be a dummy value)
    *	@param log message collector
    */
    public Query compileQuery( Reader input, String uri, Log log )
	throws XQueryException, IOException {
        StringBuffer buffer = new StringBuffer(1000);
        int count;
        char[] chars = new char[4096];
        while ((count = input.read(chars, 0, chars.length)) > 0)
            buffer.append(chars, 0, count);
        return compileQuery(buffer, uri, log);
    }
    
    /**
    *	Helper for compiling a query from a file. See {@link #compileQuery}
    *	@param input query source
    *	@param log message collector
    */
    public Query compileQuery( File input, Log log )
	throws XQueryException, IOException {
        return compileQuery(new FileReader(input), input.getAbsolutePath(), log);
    }
    
    private DefaultEvalContext prepareQuery(Query query ) throws XQueryException {
        if(query == null)
            throw new IllegalArgumentException("null query");
        if( docMan == null )
            throw new EvalException("no Document Manager defined");
        DefaultEvalContext ctx = new DefaultEvalContext(query, query.getLocalSize());
	ctx.setProperties(properties);
        ctx.setDocumentManager(docMan);
        ctx.setDefaultOutput(defaultOutput);
        ctx.setLog(log);
        try {
            if(implicitTimezone != null)
                ctx.setImplicitTimezone(implicitTimezone);
        }
        catch (DateTimeException e) {
            throw new XQueryException("implicit timezone: "+e.getMessage());
        }
        if(defaultCollation != null)
            query.setDefaultCollation(defaultCollation);
        if(input != null)
            ctx.setInput(input);
        query.initGlobals(ctx, globals);

	Integer timeout = (Integer) getSysProperty(":timeout");
	if(timeout != null) {
	    final DefaultEvalContext fctx = ctx;
	    Timer.request( timeout.intValue(), new Timer.Handler() {
		    public void timeEvent( Timer.Request r ) {
			fctx.timeOut(true);
		    }
		});
	}
        return ctx;
    }
    
    /**
    *	Executes a query in the static and dynamic environment provided by this processor.
    *	The Document Manager is used for access to documents by the XQuery function doc(),
    *  the {@link #setDocumentInput} or {@link #setCollectionInput} methods define
    *  the data accessible by the XQuery function input().
    *  @param query a query compiled with compileQuery. May be used by several threads.
    *  @return the evaluated value. Enumerates items and their value.
    *  @throws EvalException run-time exception. A stack trace can be obtained
    *  from the exception.
    *  @throws XQueryException other exception. Happens only in serious cases.
    */
    public Value executeQuery( Query query ) throws XQueryException {
        DefaultEvalContext ctx = prepareQuery(query);
	return query.eval( null, ctx );	// no initial focus
    }
    
    /**
    *	Executes a query with direct output to a serial XML event receiver (SAX or
    *	XML stream). The query must evaluate as a single Node, otherwise an EvalException
    *	is raised.
    *  <p>See {@link #executeQuery(Query query)} for more details
    *  @param query a query compiled with compileQuery. May be used by several threads.
    *  @param receiver a handler for generated events. In practice a XMLSerializer, 
    *  HTMLSerializer or SAXEventReceiver.
    */
    public void executeQuery( Query query, XMLEventReceiver receiver )
	throws XQueryException {
        DefaultEvalContext ctx = prepareQuery(query);
        
        receiver.reset();
        receiver.definePrefixHints( query.getInScopeNS() );
        try {
	    query.evalAsEvents( receiver, null, ctx );	// no initial focus
	    receiver.terminate();
        }
        catch (DataModelException e) {
            ctx.error(query.body, new EvalException(e.getMessage(), e));
        }
    }
    
    /**
    *	Executes a query with direct output to a serial XML stream (with the default
    *	serialization options).
    *	The query must evaluate as a single Node, otherwise an EvalException
    *	is raised.
    *  <p>See {@link #executeQuery(Query query)} for more details
    */
    public void executeQuery( Query query, Writer output ) throws XQueryException {
        XMLSerializer serialOut = new XMLSerializer();
        serialOut.setOutput(output);
        executeQuery(query, serialOut);
    }
}
