/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */
package net.xfra.qizxopen.util;
/****
 *	Timer event generator (supports simple one-shot alarms).
 */
public class Timer extends Thread
{
    Request requests_;	// queue

    /****
     *	Interface that must be implemented by timer-event targets.
     */
    public interface Handler {
	/****
	 *	Called when the timer event is fired. 
	 */
	void  timeEvent( Request request );
    }

    public Timer() {
	setDaemon(true);
    }

    /****
     *	Arm a timer request with a duration and an event-handler.
     *  @param duration in milliseconds.
     *  @param target receiver of the timer event.
     */
    public synchronized Request newRequest( int duration, Handler handler)
    {
	Request nrq = new Request();
	
	nrq.expiration = System.currentTimeMillis() + duration;
	nrq.handler = handler;
	// insert at proper place in list:
	if (requests_ == null)
	    requests_ = nrq;
	else {
	    Request r = requests_, last = null;
	    for( ; r != null; last = r, r = r.next)
		if (r.expiration > nrq.expiration) {
		    nrq.next = r;
		    if (last == null)
			requests_ = nrq;
		    else last.next = nrq;
		    break;
		}
	    if (r == null)
		last.next = nrq;
	}
	// wake up the service thread, to make it adjust itself:
	notify();
	return nrq;
    }

    static Timer StaticTimer = null;

    /**
     *	Static convenience.
     */
    public static synchronized Request request( int duration, Handler handler) {
	if(StaticTimer == null) {
	    StaticTimer = new Timer();
	    StaticTimer.start();
	}
	return StaticTimer.newRequest(duration, handler);
    }

    public void run()
    {
	long wtime;
	while(true)
	    synchronized (this) {
		wtime = 1000;
		if (requests_ != null) {
		    long now = System.currentTimeMillis();
		    if (now >= requests_.expiration) {
			// fire the event
			Request r = requests_;
			requests_ = r.next;
			if(!r.cancelled)
			    r.handler.timeEvent(r);	//TODO: unlock before calling
			continue;	// dont wait: maybe another
		    }
		    else wtime = requests_.expiration - now;
		}
		try {
		    wait(wtime);
		} catch(InterruptedException e) { }
	    }
    }

    public static class Request
    {
	long  expiration;
	boolean cancelled;
	Request next;
	Handler handler;

	public void cancel() {
	    cancelled = true;
	}
    }

    
/*
    public static void main(String args[]) {
	Beat b1 = new Beat(1300);
	Beat b2 = new Beat(1000);
	Beat b3 = new Beat(400);
	b1.timeEvent(null);
	b2.timeEvent(null);
	b3.timeEvent(null);
	try {
	    Thread.sleep(2000);
	} catch(InterruptedException e) { }
    }

    static class Beat implements Timer.Handler {
	int cycle;
	Timer.Request curReq;

	Beat(int cycle) {
	    this.cycle = cycle;
	}

	public void timeEvent(Timer.Request req) {
	    System.out.println("beat " + cycle + " at " + System.currentTimeMillis());
	    curReq = Timer.request(cycle, this);
	    //curReq.cancel();
	}
    } */
}
