/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.util;

import java.util.HashMap;

/**
 *	Representation of a Qualified Name: namespace URI + localName.
 *	<p>There is no notion of prefix here.
 *	<p>A QName has a unique representation: comparison can be done with ==
 */
public final class QName
{
    private static HashMap names = new HashMap();
    private static QName probe = new QName(null, null);

    public static QName XML_LANG;
    public static QName XML_SPACE;
    public static QName XML_BASE;
    public static QName XSI_TYPE;
    public static QName XSI_NIL;
    public static QName XSI_SCHEMA_LOCATION;
    public static QName XSI_NO_NAMESPACE_SCHEMA_LOCATION;

    static {
	XML_LANG = get(Namespace.XML, "lang");
	XML_SPACE = get(Namespace.XML, "space");
	XML_BASE = get(Namespace.XML, "base");
	XSI_TYPE = get(Namespace.XSI, "type");
	XSI_NIL = get(Namespace.XSI, "nil");
	XSI_SCHEMA_LOCATION = get(Namespace.XSI, "schemaLocation");
	XSI_NO_NAMESPACE_SCHEMA_LOCATION = get(Namespace.XSI, "noNamespaceSchemaLocation");
    }

    private Namespace namespace; 
    private String localName; 

    // ------------------------------------------------------------------------

    public static synchronized QName get(String localName) {
	return get(Namespace.NONE, localName);
    }

    public static synchronized QName get(String namespaceURI, String localName) {
	return get(Namespace.get(namespaceURI), localName);
    }

    public static synchronized QName get(Namespace namespace, String localName) {
	if (namespace == null)
	    throw new IllegalArgumentException("null namespace");
	if (localName == null)
	    throw new IllegalArgumentException("null localName");

	probe.namespace = namespace;
	probe.localName = localName;
	QName name = (QName) names.get(probe);
	if (name == null) {
	    name = new QName(namespace, localName);
	    names.put(name, name);
	}
	return name;
    }

    public boolean equals(Object other) {
	if (other == null || !(other instanceof QName))
	    return false;
	QName n = (QName) other;
	return (namespace == n.namespace && localName.equals(n.localName));
    }

    public int hashCode() {
	// identityHashCode returns 0 for null.
	return (System.identityHashCode(namespace) ^ localName.hashCode());
    }

    private QName(Namespace namespace, String localName) {
	this.namespace = namespace;
	this.localName = localName == null ? null : localName.intern();
    }

    public Namespace getNamespace() {
	return namespace;
    }

    public String getURI() {
	return namespace.getURI();
    }

    public String getLocalName() {
	return localName;
    }

    public String toString() {
	if (namespace == Namespace.NONE)
	    return localName;
	return '{' + namespace.getURI() + '}' + localName;
    }

    public int compareTo(QName other) {
	if (namespace == other.namespace)
	    return localName.compareTo(other.localName);
	else
	    return namespace.compareTo(other.namespace);
    }
}
