/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.util;

/**
 *	Mapping of Namespace prefixes.
 *	<p>This simple implementation with linear search is sufficient if
 *	we consider that rarely more than a few NS are defined in practice.
 */
public class NSPrefixMapping {

    private int ptr = 0;
    private String[] prefixes;	// allocated on demand
    private Namespace[] namespaces;

    public NSPrefixMapping copy() {
	NSPrefixMapping nm = new NSPrefixMapping();
	nm.ptr = ptr;
	nm.prefixes = (String[]) prefixes.clone();
	nm.namespaces = (Namespace[]) namespaces.clone();
	return nm;
    }

    public void addMapping( String prefix, String uri ) {
	addMapping( prefix, Namespace.get(uri) );
    }

    public void addMapping( String prefix, Namespace ns ) {
	if(prefixes == null) {
	    prefixes = new String[4];
	    namespaces = new Namespace[4];
	}
	else if(ptr >= prefixes.length) {
	    String[] oldp = prefixes;
	    prefixes = new String[2 * oldp.length];
	    System.arraycopy(oldp, 0, prefixes, 0, oldp.length);
	    Namespace[] oldns = namespaces;
	    namespaces = new Namespace[2 * oldns.length];
	    System.arraycopy(oldns, 0, namespaces, 0, oldns.length);
	}
	prefixes[ ptr ] = prefix;
	namespaces[ ptr ] = ns;
	++ ptr ;
    }

    public int getMappingCount() {
	return ptr;
    }

    public void removeMappings( int count ) {
	ptr -= count;
    }

    /**
     *	Returns the most recent NS matching the prefix.
     */
    public Namespace convertToNamespace( String prefix ) {
	for(int p = ptr; --p >= 0; ) {
	    if(prefixes[ p ].equals(prefix))
		return namespaces[ p ];
	}
	return null;
    }

    /**
     *	Returns the most recent prefix matching the NS.
     */
    public String convertToPrefix( Namespace ns ) {
	for(int p = ptr; --p >= 0; )
	    if(namespaces[ p ] == ns)
		return prefixes[ p ];
	return null;
    }

    public static String extractPrefix( String name ) {
	int colon = name.indexOf(':');
	String prefix = colon < 0 ? "" : name.substring(0, colon);
	if(prefix.length() > 0 && !XMLUtil.isNCName(prefix))
	    throw new IllegalArgumentException("illegal prefix "+prefix);
	return prefix;
    }

    public static String extractLocalName( String name ) {
	int colon = name.indexOf(':');
	String ncname = name.substring(colon + 1);
	if(!XMLUtil.isNCName(ncname))
	    throw new IllegalArgumentException("illegal local name "+ncname);
	return ncname;
    }

    /**
     *	Converts a prefixed name into a QName.
     *	@return null if the prefix is not defined, the expanded QName otherwise.
     */
    public QName expandName( String name ) {
	String prefix = extractPrefix(name);
	Namespace ns = prefix.length() == 0 ? Namespace.NONE : convertToNamespace(prefix);
	return ns == null ? null : QName.get(ns, extractLocalName(name));
    }

    /**
     *	Converts a QName into a prefixed name.
     */
    public String prefixedName( QName name ) {
	if(name.getNamespace() == Namespace.NONE)
	    return name.getLocalName();
	return convertToPrefix(name.getNamespace()) +":"+ name.getLocalName();
    }

    /**
     *	Returns a declared prefix. The last declared prefix has rank 1, the previous 2 etc.
     */
    public String getLastPrefix( int rankFromTop ) {
	return prefixes[ ptr - rankFromTop ];
    }
    /**
     *	Returns a declared namespace.
     *	The last declared namespace has rank 1, the previous 2 etc.
     */
    public Namespace getLastNamespace( int rankFromTop ) {
	return namespaces[ ptr - rankFromTop ];
    }

    public static NSPrefixMapping std;
    static {
	std = new NSPrefixMapping();
	std.addMapping("xml", Namespace.XML );
	std.addMapping("xs", Namespace.XSD );
	std.addMapping("xsi", Namespace.XSI );
	std.addMapping("fn", Namespace.FN );
	std.addMapping("op", Namespace.OP );
    }

} // end of class NSPrefixMapping

