/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.ext;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.LimitReachedException;
import net.xfra.qizxopen.dm.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.impl.*;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.fn.Prototype;
import net.xfra.qizxopen.xquery.op.Expression;
import net.xfra.qizxopen.xquery.op.StringLiteral;
import net.xfra.qizxopen.xquery.dm.Node;
import net.xfra.qizxopen.xquery.dm.EventDrivenBuilder;
import net.xfra.qizxopen.xquery.dt.ArraySequence;

import java.io.StringWriter;
import java.io.PrintWriter;

/**
 *	Parses and evaluates a XQuery expression, returns the results in diverse forms
 *	according to the options.
 *	<p>Options:<ul>
 *	<li>escaped: if true, each item is serialized to XML, otherwise
 *	<li>max-output: maximum size of output in bytes.
 *	<li>max-time: maximum execution time in milliseconds.
 *	</ul>
 */
public class QizxEval extends Function
{
    static QName qfname = QName.get(XQueryProcessor.EXTENSIONS_NS, "eval");
    static Prototype[] protos = { 
        new Prototype(qfname, Type.NODE.star, Exec.class)
	 .arg("query", Type.STRING).arg("options", Type.NODE.star),
        new Prototype(qfname, Type.NODE.star, Exec.class)
	 .arg("query", Type.STRING)
    };
    public Prototype[] getProtos() { return protos; }

    public static class Exec extends Function.Call {

        public Value eval(Focus focus, EvalContext context)
	    throws XQueryException {
            context.at(this);
	    String querySrc = args[0].evalAsString(focus, context);
	    querySrc = querySrc.replace('\r', ' ');
	    Node options = (args.length<2)? null : args[1].evalAsOptNode(focus, context);

	    QName headerElem = QName.get("pre"), itemElem = QName.get("pre");
	    QName classAttr = QName.get("class");
	    int maxTime = -1, maxItems = -1;
	    int maxOutSize = -1;
	    boolean escaped = true, traceExceptions = false;

	    StringWriter out = null;
	    PrintWriter pwout = null;
	    Log log = null;
	    EventDrivenBuilder builder = new EventDrivenBuilder();
	    ArraySequence rseq = new ArraySequence(4);
	    try {
		if(options != null) {	
		    for(Value list = options.getAttributes(); list.next(); ) {
			Node attr = list.asNode();
			String option = attr.getNodeName().getLocalName();
			if(option.equals("escaped"))
			    escaped = attr.getStringValue().equals("true");
			else if(option.equals("output-max"))
			    maxOutSize = Integer.parseInt(attr.getStringValue());
			else if(option.equals("time-max"))
			    maxTime = Integer.parseInt(attr.getStringValue());
			else if(option.equals("item-max"))
			    maxItems = Integer.parseInt(attr.getStringValue());
		    }
		} 
		out = new StringWriter();
		pwout = new PrintWriter(out);
		log = new Log(pwout);
		XQueryProcessor pro = (XQueryProcessor)context.getProperty(":processor");
		Query q = pro.compileQuery(querySrc, "query", log);
		if(maxOutSize > 0)
		    builder.setMaxVolume(maxOutSize);
		if(maxTime > 0)
		    pro.setSysProperty(":timeout", new Integer(maxTime));

		try {
		    long T0 = System.currentTimeMillis();
		    Value result = pro.executeQuery(q);
		    result.setLazy(true);
		    int count = 0;
		    for(; result.next(); )
			++ count;
		    long T1 = System.currentTimeMillis();
		    //builder.startDocument();
		    builder.startElement(headerElem);
		    builder.attribute(classAttr, "header");
		    builder.text("Query executed in "+(T1-T0)+" milliseconds, returns ");
		    if(count == 0)
			builder.text("empty sequence");
		    else builder.text(count+" item"+((count > 1)?"s":"")+":");
		    builder.endElement(headerElem);
		    //builder.endDocument();
		    rseq.addItem( builder.crop() );

		    XMLSerializer serialDisplay = new XMLSerializer();
		    if(maxOutSize > 0)
			serialDisplay.setMaxVolume(maxOutSize);
		    serialDisplay.definePrefixHints( q.getInScopeNS() );

		    if(escaped) {
			builder.reset();
			builder.startElement(itemElem);
			builder.attribute(classAttr, "items");
		    }
		    result = result.bornAgain();
		    for(int it = 1; result.next(); ++it) {
			if(maxItems > 0 && it > maxItems)
			    throw new LimitReachedException("too many items");
			Item item = result.asItem();
			Node node = null;
			if( item.isNode() && !escaped &&
			    (node = item.asNode()).getNature() != Node.TEXT) {
			    rseq.addItem(node);
			}
			else {	// serialize each item inside an 'item' element:
			    if(item.isNode()) {	// escaped
				out = new StringWriter();
				pwout = new PrintWriter(out);
				serialDisplay.setOutput(pwout);
				serialDisplay.reset();
				serialDisplay.traverse(item.asNode());
				serialDisplay.terminate();
				builder.text(trimString(out.getBuffer()));
			    }
			    else builder.atom( result.asString() );
			}
		    }
		    if(escaped) {
			builder.endElement(itemElem);
			builder.terminate();
			rseq.addItem(builder.crop());
		    }
		}
		catch(LimitReachedException lime) {
		    builder.setMaxVolume(-1);
		    builder.endElement(itemElem);
		    builder.terminate();
		    rseq.addItem(builder.crop());
		    builder.reset();
		    builder.startElement(headerElem);
		    builder.attribute(classAttr, "runtime-error");
		    builder.text("** truncated results: "+ lime.getMessage() +" **");
		    builder.endElement(headerElem);
		    rseq.addItem(builder.crop());
		}
		catch (EvalException ee) {
		    ee.printStack(log, 20);
		    if(ee.getCause() != null && traceExceptions) {
			log.info("  caused by: " + ee.getCause());
		    }
		    log.flush();
		    builder.startElement(headerElem);
		    builder.attribute(classAttr, "runtime-error");
		    builder.text(out.toString());
		    builder.endElement(headerElem);
		    rseq.addItem(builder.crop());
		}
	    }
	    catch (Exception e) {
		try {
		    log.flush();
		    builder.startElement(headerElem);
		    builder.attribute(classAttr, "error");
		    pwout.println("*** "+e.getMessage());
		    pwout.flush();
		    builder.text(out.toString());
		    builder.endElement(headerElem);
		    rseq.addItem(builder.crop());
		}
		catch (DataModelException de) {
		    context.error(this, "serialization error: "+de.getMessage());
		}
	    }
	    return rseq;
	}
    }

    // boring problem with \r generated by PrintWriter on Windows
    private static String trimString(StringBuffer buf) {
	int out = 0;
	for(int i = 0, L = buf.length(); i < L; i++)
	    if(buf.charAt(i) != '\r')
		buf.setCharAt(out++, buf.charAt(i));
	return buf.substring(0, out);
    }
} 
