/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.NSPrefixMapping;
import net.xfra.qizxopen.xquery.XQueryException;

/**
 *	An abstract receiver of SAX-like events. 
 *	<p>Used for event-style evaluation of XQuery expressions.
 */
public interface XMLEventReceiver
{
    /**
     *	Resets the state prior to use startDocument() or startElement().
     */
    public void reset();

    /**
     *	Terminates a document or simple subtree. May perform consistency checks.
     */
    public void terminate() throws DataModelException;

     /**
     *	Starts a document.
     *	<p>It is not necessarily called, if only a fragment is generated.
     */
    public void startDocument() throws DataModelException;

    /**
     *	Ends a document. Must be balanced by a corresponding startDocument().
     */
    public void endDocument() throws DataModelException;

    /**
     *	Starts an element. Must be balanced by a corresponding endElement().
     */
    public void startElement(QName name) throws DataModelException;

    /**
     * Must follow startElement and precede any child. May be interleaved with attribute().
     */
    public void namespace(String prefix, String uri) throws DataModelException;

    /**
     * Must follow startElement and precede any child. May be interleaved with namespace().
     */
    public void attribute(QName name, String value) throws DataModelException;

    /**
     *	Ends an element. Removes NS mappings.
     */
    public void endElement(QName name) throws DataModelException;

    /**
     *	Emits a text chunk. No space before or after.
     */
    public void text(String value) throws DataModelException;

    /**
     *	Emits the text of an atom. The difference with text() is that a space in requested
     *	between two atoms.
     */
    public void atom(String value) throws DataModelException;

    /**
     *	Emits a processing-instruction.
     */
    public void pi(String target, String value) throws DataModelException;

    /**
     *	Emits a comment.
     */
    public void comment(String value) throws DataModelException;

    /**
     *	Traverses and generates a subtree into this receiver.
     *	@param inScopeNS if true, copy all in-scope namespace nodes (not only those defined
     *	in the node itself).
     */
    public void traverse( Node node, boolean inScopeNS ) throws DataModelException ;

    /**
     *	Resolves a prefix to an URI in the context of the current node.
     *	@return null if the prefix cannot be resolved.
     */
    String resolvePrefix( String prefix );

    /**
     *	Optional: define preferred prefix->NS mapping for serialization.
     */
    void  definePrefixHints( NSPrefixMapping prefixes );
} 
