/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.QName;
import org.xml.sax.*;
import org.xml.sax.ext.LexicalHandler;

/**
 *	Adapter of tree events to SAX2 handlers.
 *	
 */
public class SAXEventReceiver extends XMLEventReceiverBase
    implements org.xml.sax.Attributes 
{
    public boolean nsPrefixes = !false;
    ContentHandler contentHandler;
    LexicalHandler lexicalHandler;
    ErrorHandler   errorHandler;
    protected char[] charBuf;

    public SAXEventReceiver( )  {
    }

    public ContentHandler getContentHandler() {
	return contentHandler;
    }

    public void setContentHandler(ContentHandler handler) {
	contentHandler = handler;
    }

    public LexicalHandler getLexicalHandler() {
	return lexicalHandler;
    }

    public void setLexicalHandler(LexicalHandler handler) {
	lexicalHandler = handler;
    }

    public ErrorHandler getErrorHandler() {
	return errorHandler;
    }

    public void setErrorHandler(ErrorHandler handler) {
	errorHandler = handler;
    }

    private void convertSaxException(SAXException e) throws DataModelException {
	throw new DataModelException("SAX exception: "+e.getMessage());
    }

    public void  namespace(String prefix, String uri) throws DataModelException {
	super.namespace(prefix, uri);
        try {	// OK to emit NS immediately, because tag and attributes are buffered.
	    if(contentHandler != null)
		contentHandler.startPrefixMapping(prefix, uri);
        } catch (SAXException e) {
            convertSaxException(e);
        }
    }

    public void  startDocument() throws DataModelException {
	super.startDocument();
        try {
	    if(contentHandler != null)
		contentHandler.startDocument();
        } catch (SAXException e) {
            convertSaxException(e);
        }
    }

    public void  endDocument() throws DataModelException {
        try {
	    if(contentHandler != null)
		contentHandler.endDocument();
        } catch (SAXException e) {
            convertSaxException(e);
        }
	super.endDocument();
    }

    public void  flushElement( boolean empty ) throws DataModelException {
	if(!elementStarted)
	    return;
	elementStarted = false;
        try {
	    String qname = nsPrefixes ? prefixes.prefixedName(tagName) : null;
	    if(contentHandler != null)
		contentHandler.startElement(tagName.getURI(), tagName.getLocalName(),
					    qname, this);
        } catch (SAXException e) {
            convertSaxException(e);
        }
    }

    public void  endElement(QName name) throws DataModelException {
	if(elementStarted)
	    flushElement(true);
        if(contentHandler != null)
	    try {
		String qname = nsPrefixes ? prefixes.prefixedName(tagName) : null;
		contentHandler.endElement(tagName.getURI(), tagName.getLocalName(), qname);
		// remove mappings
		for(int ns = 1; ns <= nsCnt; ns++)
		    contentHandler.endPrefixMapping( prefixes.getLastPrefix(ns) );
	    } catch (SAXException e) {
		convertSaxException(e);
	    }
	super.endElement(name);
     }

    public void  text(String value) throws DataModelException {
        try {
	    if(elementStarted)
		flushElement(false);
		
	    int vlen = value.length();
	    if(charBuf == null || vlen > charBuf.length)
		charBuf = new char[vlen];
	    value.getChars(0, vlen, charBuf, 0);
	    if(contentHandler != null)
		contentHandler.characters(charBuf, 0, vlen);
	    spaceNeeded = false;
        } catch (SAXException e) {
            convertSaxException(e);
        }
    }

    public void  atom(String value) throws DataModelException {
        try {
	    if(elementStarted)
		flushElement(false);
	    if(spaceNeeded && contentHandler != null)
		contentHandler.characters(blank, 0, 1);
		
	    int vlen = value.length();
	    if(charBuf == null || vlen > charBuf.length)
		charBuf = new char[vlen];
	    value.getChars(0, vlen, charBuf, 0);
	    if(contentHandler != null)
		contentHandler.characters(charBuf, 0, vlen);
	    spaceNeeded = true;
        } catch (SAXException e) {
            convertSaxException(e);
        }
    }

    public void  pi(String target, String value) throws DataModelException {
        try {
	    if(elementStarted)
		flushElement(false);
	    if(contentHandler != null)
		contentHandler.processingInstruction(target, value);
        } catch (SAXException e) {
            convertSaxException(e);
        }
    }

    public void  comment(String value) throws DataModelException {
        try {
	    if(elementStarted)
		flushElement(false);
	    int vlen = value.length();
	    if(charBuf == null || vlen > charBuf.length)
		charBuf = new char[vlen];
	    value.getChars(0, vlen, charBuf, 0);
	    if(lexicalHandler != null)
		lexicalHandler.comment(charBuf, 0, vlen);
        } catch (SAXException e) {
            convertSaxException(e);
        }
    }


    

    public int getIndex(QName qName) {
	for(int i = attrCnt; --i >= 0; )
	    if(attrNames[i] == qName)
		return i;
	return -1;
    }

    public int getIndex(String qName) {
	return getIndex(prefixes.expandName(qName));
    }

    public int getIndex(String uri, String localName) {
	return getIndex(QName.get(uri, localName));
    }

    public int getLength() {
	return attrCnt;
    }

    public String getLocalName(int index) {
	return index < 0 || index >= attrCnt ? null : attrNames[index].getLocalName();
    }

    public String getURI(int index) {
	return index < 0 || index >= attrCnt ? null : attrNames[index].getURI();
    }

    public String getQName(int index) {
	return index < 0 || index >= attrCnt ? null : prefixes.prefixedName(attrNames[index]);
    }

    public String getType(int index) {
	return "CDATA";
    }
    public String getType(String qName) {
	return "CDATA";
    }
    public String getType(String uri, String localName) {
	return "CDATA";
    }

    public String getValue(int index) {
	return index < 0 || index >= attrCnt ? null : attrValues[index];
    }

    public String getValue(String qName) {
	return getValue(getIndex(qName));
    }

    public String getValue(String uri, String localName) {
	return getValue(getIndex(uri, localName));
    }

} // end of class SAXEventReceiver
