/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.Namespace;
import java.text.Collator;

/**
 *	Node as defined by the XQuery1/XPath2 Data Model, with extensions.
 */
public interface Node
{
    /**
     *	Accessor dm:node-kind().
     *	@return the node kind as a string: "element", "attribute" etc.
     */
    String getNodeKind();
    /**
     *	Accessor dm:node-name().
     *	@return null if the node has no name (document, text, comment, pi).
     */
    QName  getNodeName();
    /**
     *	Accessor dm:parent().
     *	@return null if the node has no parent.
     */
    Node   parent();
    /**
     *	Accessor dm:string-value().
     */
    String getStringValue();
    /**
     *	Accessor dm:base-uri().
     */
    String getBaseURI();

    String getDocumentURI();

    /**
     *	Accessor dm:children.
     *	Only for Document and Elements: other kinds yields an empty sequence.
     */
    NodeSequence children();

    /**
     *	Accessor dm:attributes. Only for Elements: other kinds yields an empty sequence.
     */
    NodeSequence attributes();

    /**
     *	Accessor dm:namespaces(). Returns namespaces visible by this node.
     *	On non-element nodes, always returns the empty sequence.
     *  @param inScope if true, return the closure of in-scope namespaces,
     *  else namespaces defined specifically on this node. 
     */
    NodeSequence namespaces( boolean inScope );

    // ---- extensions for internal use: ------------------------------------------------

    static int	DOCUMENT = 1;
    static int	ELEMENT = 2;
    static int	ATTRIBUTE = 3;
    static int	TEXT = 4;
    static int	PROCESSING_INSTRUCTION = 5;
    static int	COMMENT = 6;
    static int	NAMESPACE = 7;
    /**
     *	Returns a numeric node kind: DOCUMENT, ELEMENT etc.
     */
    int   getNature();

    /**
     *	Returns the document node if any, else the top-level node.
     */
    Node  document();

    /**
     *	Returns an arbitrary value that is stable on the whole document (for order
     *	comparison).
     */
    int docPosition();

    /**
     *	Convenience: quick test of elements.
     */
    boolean isElement();

    /**
     *	get attribute by name.
     */
    Node  attribute( QName name );

    /**
     *	get text as char array (atomic nodes only).
     */
    char[]  getChars();

    /**
     *	Number of Namespaces defined on this node.
     */
    int     getDefinedNSCount();
    /**
     *	Returns a matching prefix for the Namespace by looking up the namespace nodes.
     *  @return null if none is found, the first suitable prefix otherwise.
     */
    String getNsPrefix( String nsuri );
    /**
     *	Returns a matching Namespace for the prefix by looking up the namespace nodes.
     *  @return null if none is found, the first suitable namespace URI otherwise.
     */
    String getNsUri( String prefix );

    /**
     *	Returns -1 if this node is strictly before the argument node in document order,
     *  0 if nodes are identical, 1 if after the argument node. 
     *  If the two nodes belong to different documents, returns
     *  an arbitrary but stable "order of documents".
     */
    int orderCompare( Node node );

    int compareStringValues( Node node, Collator collator );

    /**
     *	Returns true if this node is an ancestor of the node, or the node itself.
     */
    boolean contains( Node node );

    // ---- extensions for efficient XPath steps:

    /**
     *	Returns an iterator on ancestors that match the node test.
     *  @param nodeTest concrete classes are BaseNodeTest(node-kind?, nsuri?, ncname?),
     *  UnionNodeTest, DocumentTest. 
     */
    NodeSequence ancestors( NodeTest nodeTest );

    NodeSequence ancestorsOrSelf( NodeTest nodeTest );

    /**
     *	Returns a sequence of length 1 (if the parent matches the nodeTest) else 0.
     */
    NodeSequence parent( NodeTest nodeTest );

    NodeSequence children( NodeTest nodeTest );

    NodeSequence descendants( NodeTest nodeTest );

    NodeSequence descendantsOrSelf( NodeTest nodeTest );

    NodeSequence attributes( NodeTest nodeTest );

    NodeSequence followingSiblings( NodeTest nodeTest );
    /**
     *	Preceding siblings in document order.
     */
    NodeSequence precedingSiblings( NodeTest nodeTest );

    NodeSequence following( NodeTest nodeTest );
    /**
     *	Preceding nodes in document order.
     */
    NodeSequence preceding( NodeTest nodeTest );

    // ----- Modification (optional): -------------------------------------------------

    /**
     *	Adds text inside the node. For an element, creates a text child node if necessary.
     */
    void addText( String text );

    // ----- Extensions for typed atomic values ---------------------------------------

    int TYPE_NODE = 0;		// non-atomic
    int TYPE_BINARY = 1;	// array of bytes
    int TYPE_BOOLEAN = 2;
    int TYPE_DATETIME = 3;  
    int TYPE_DOUBLE = 4;
    int TYPE_INTEGER = 5;
    int TYPE_STRING = 6;

    /**
     *	returns the real type of the atom value: TYPE_STRING, TYPE_BINARY, etc.
     */
    int   getAtomType();
    /**
     *	Returns the real value of the atom: byte[], Boolean, Date, Double, Long, String.
     */
    Object   getAtomValue();	// 
    /**
     * Returns the double value (attempts to convert).
     * @throws DataModelException if not convertible to double.
     */
    double   getDoubleValue() throws DataModelException;
    /**
     * Returns the integer value (attempts to convert).
     * @throws DataModelException if not convertible to integer.
     */
    long     getIntegerValue() throws DataModelException;
}
