/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.*;
import java.text.Collator;
import java.util.Date;


/**
 *	Basic in-core implementation of Data Model, which can be
 *	used independently of XQuery.
 *	<p>Used for temporarily constructed tree fragments. TODO: builder!
 */
public class CoreDM
{
    public static int CreatedElements = 0, CreatedText = 0;

    /**	[Internal]. */
    public static abstract class BaseNode implements Node {

	public Element  parent;
	public BaseNode nextSibling;
	public int      order = -1;		//TODO

	public String toString() {
	    return "Core "+getNodeKind()+" "+
		(getNodeName() != null ? (" name="+getNodeName()) : getStringValue());
	}

	public String getBaseURI() {
	    QName base = QName.get(Namespace.XML, "base");
	    for(BaseNode n = this; n != null; n = n.parent) {
		Node attr  = n.attribute(base);
		if(attr != null)
		    return attr.getStringValue();
	    }
	    return null;
	}

	public String getDocumentURI() {
	    return null;
	}

	public QName  getNodeName() {
	    return null;
	}

	public Node  parent() {
	    return parent;
	}

	BaseNode getFirstChild() {
	    return null;
	}

	public Node  document() {
	    return getTop();
	}

	public int orderCompare( Node node ) {
	    if(node == this)
		return 0;
	    if( !(node instanceof BaseNode) )
		return docPosition() - node.docPosition();
	    BaseNode that = (BaseNode) node, top = getTop(), thatTop = that.getTop();
	    if(top != thatTop)
		return top.docPosition() - thatTop.docPosition();
	    // if possible, use order stamps:
	    if(order >= 0 && that.order >= 0)
		return Util.comparison(order - that.order);
	    // inefficient, but not supposed to be used heavily:
	    for( ; that != null; that = that.nodeNext())
		if(that == this)
		    return 1;
	    return -1;
	}

	public boolean contains( Node node ) {
	    if( !(node instanceof BaseNode) )
		return false;
	    
	    for( BaseNode n = (BaseNode) node; n != null; n = n.parent)
		if(n == this)
		    return true;
	    return false;
	}

	public boolean equals( Object that ) {
	    return this == that;
	}

	public boolean deepEqual( Node node, Collator collator ) {
	    return deepEq(node, collator);
	}

	public boolean deepEq( Node that, Collator collator ) {
	    int kind = getNature();
	    if(kind != that.getNature() || getNodeName() != that.getNodeName() )
		return false;
	    if(kind == Node.DOCUMENT)
		return ((Element) this).contentEq((Element) that, collator);
	    else if(kind == Node.ELEMENT)
		return ((Element) this).attributesEq((Element) that, collator)
		    && ((Element) this).contentEq((Element) that, collator);
	    else return compareStringValues(that, collator) == 0;
	}

	public int compareStringValues( Node that, Collator collator ) {
	    String s1 = this.getStringValue(), s2 = that.getStringValue();
	    return collator != null ? collator.compare(s1, s2) : s1.compareTo(s2);
	}


	public  BaseNode  nodeNext() {	// in document order
	    BaseNode kid = getFirstChild();
	    return kid != null ? kid : nodeAfter();
	}

	public  BaseNode  nodeAfter() {	// in document order
	    BaseNode node = this, nxt;
	    while((nxt = node.nextSibling) == null) {
		BaseNode parent = node.parent;
		if (parent == null)
		    return null;
		node = parent;
	    }
	    return nxt;
	}

	BaseNode getTop() {
	    BaseNode n = this;
	    for( ; n.parent != null; )
		n = n.parent;
	    return n;
	}

	public int docPosition() {
	    return getTop().hashCode();	    
	}

	public int getDefinedNSCount() {
	    return 0;
	}

	public void addNamespace( String prefix, String value ) {
	    throw new RuntimeException("not an element");
	}

	

	public char[] getChars() {
	    throw new RuntimeException("getChars of a non-atom");
	}

	public Object getAtomValue() {
	    return this;
	}

	public int   getAtomType() {
	    Object value = getAtomValue();
	    if(value instanceof String)
		return TYPE_STRING;
	    if(value instanceof Long)
		return TYPE_INTEGER;
	    if(value instanceof Date)
		return TYPE_DATETIME;
	    if(value instanceof Double)
		return TYPE_DOUBLE;
	    if(value instanceof Boolean)
		return TYPE_BOOLEAN;
	    if(value.getClass().isArray() )	// OOPS
		return TYPE_BINARY;
	    throw new RuntimeException("improper atom value "+value);
	}

	public double getDoubleValue() throws DataModelException {
	    Object obj = getAtomValue();
	    if(obj instanceof Double)
		return ((Double) obj).doubleValue();
	    try {
		return Double.parseDouble(getStringValue());
	    }
	    catch (NumberFormatException e) {
		throw new DataModelException("atom cannot be converted to double", e);
	    }
	}

	public long getIntegerValue() throws DataModelException {
	    Object obj = getAtomValue();
	    if(obj instanceof Long)
		return ((Long) obj).longValue();
	    try {
		return Long.parseLong(getStringValue());
	    }
	    catch (NumberFormatException e) {
		throw new DataModelException("atom cannot be converted to integer", e);
	    }
	}

	

	public NodeSequence ancestors( NodeTest nodeTest ) {
	    return new AncestorsOrSelf(this, nodeTest);
	}

	public NodeSequence ancestorsOrSelf( NodeTest nodeTest ) {
	    return new AncestorsOrSelf(this, nodeTest);
	}

	public NodeSequence parent( NodeTest nodeTest ) {
	    return new Parent(this, nodeTest);
	}

	public NodeSequence children( NodeTest nodeTest ) {
	    return new Children(this, nodeTest);
	}

	public NodeSequence descendants( NodeTest nodeTest ) {
	    return new Descendants(this, nodeTest);
	}

	public NodeSequence descendantsOrSelf( NodeTest nodeTest ) {
	    return new DescendantsOrSelf(this, nodeTest);
	}

	public NodeSequence attributes( NodeTest nodeTest ) {
	    return new Attributes(this, nodeTest);
	}

	public NodeSequence followingSiblings( NodeTest nodeTest ) {
	    return new FollowingSiblings(this, nodeTest);
	}

	public NodeSequence precedingSiblings( NodeTest nodeTest ) {
	    return new PrecedingSiblings(this, nodeTest);
	}

	public NodeSequence following( NodeTest nodeTest ) {
	    return new Following(this, nodeTest);
	}

	public NodeSequence preceding( NodeTest nodeTest ) {
	    return new Preceding(this, nodeTest);
	}

	public void setNextSibling( BaseNode sibling ) {
	    nextSibling = sibling;
	    if(sibling != null)
		sibling.parent = parent;
	}
    }

    static class Element extends BaseNode {

	QName name;
	BaseNode firstChild;
	Attribute attributes;

	Element( QName name ) {
	    ++ CreatedElements;
	    this.name = name;
	}

	public String getNodeKind() {
	    return "element";
	}

	public String getStringValue() {
	    StringBuffer buf = new StringBuffer();
	    for(BaseNode kid = firstChild; kid != null; kid = kid.nextSibling)
		recStringValue(kid, buf);
	    return buf.toString();
	}

	static void recStringValue(BaseNode node, StringBuffer buf) {
	    if(node.isElement()) {
		BaseNode kid = ((Element) node).firstChild;
		for( ; kid != null; kid = kid.nextSibling)
		    recStringValue(kid, buf);
	    }
	    else if(node.getNature() == Node.TEXT)	// ignore comments and PI
		buf.append( node.getStringValue() );
	}

	public NodeSequence children() {
	    return new SSequence(firstChild);
	}

	public QName  getNodeName() {
	    return name;
	}

	BaseNode getFirstChild() {
	    return firstChild;
	}

	public NodeSequence attributes() {
	    return new Attributes(this, null);
	}

	public NodeSequence namespaces( boolean inScope ) {
	    //TODO (or maybe not) implement inScope
	    return new Namespaces(this);
	}

	public int getDefinedNSCount() {
	    int cnt = 0;
	    for(BaseNode attr = attributes; attr != null; attr = attr.nextSibling)
		if(attr.getNature() == NAMESPACE)
		    ++ cnt;
	    return cnt;
	}

	public String getNsPrefix( String nsuri ) {
	    for(Element e = this; e != null; e = e.parent)
		for(BaseNode attr = e.attributes; attr != null; attr = attr.nextSibling)
		    if(attr instanceof NSNode) {
			NSNode ns = (NSNode) attr;
			if(nsuri.equals(ns.value))
			    return ns.name.getLocalName();
		    }
	    return null;
	}

	public String getNsUri( String prefix ) {
	    for(Element e = this; e != null; e = e.parent) {
		for(BaseNode attr = e.attributes; attr != null; attr = attr.nextSibling)
		    if(attr instanceof NSNode) {
			NSNode ns = (NSNode) attr;
			if(prefix.equals(ns.name.getLocalName()))
			    return ns.getStringValue();
		    }
	    }
	    return null;
	}

	

	public int   getNature() {
	    return Node.ELEMENT;
	}

	public boolean isElement() {
	    return true;
	}

	public boolean isWhitespace() {
	    return false;
	}

	boolean attributesEq( Element that, Collator collator ) {
	    NodeSequence oattr = that.attributes();
	    int acnt = 0;
	    for(; oattr.nextNode(); ++ acnt) {
		Node oatt = oattr.currentNode();
		BaseNode att = (BaseNode) attribute(oatt.getNodeName());
		if( att == null || att.compareStringValues( oatt, collator ) != 0)
		    return false;
	    }
	    return attributeCount() == acnt;
	}

	boolean contentEq( Element that, Collator collator ) {
	    NodeSequence okids = that.children();
	    for(BaseNode kid = firstChild; kid != null; kid = kid.nextSibling) {
		int kidkind = kid.getNature(), okidkind;
		if(kidkind == Node.COMMENT || kidkind == Node.PROCESSING_INSTRUCTION)
		    continue;
		Node okid;
		for( ; ; ) {
		    if(!okids.nextNode())
			return false;
		    okid = okids.currentNode();
		    if( (okidkind = okid.getNature()) != Node.COMMENT &&
			 okidkind != Node.PROCESSING_INSTRUCTION)
			break;
		}
		// we have 2 comparable kids:
		if(!kid.deepEq( okid, collator ))
		   return false;
	    }
	    // just check that there are no more kids in 'that'
	    return !okids.nextNode();
	}

	public int attributeCount() {
	    int cnt = 0;
	    for(BaseNode attr = attributes; attr != null; attr = attr.nextSibling)
		if(attr.getNature() == ATTRIBUTE)
		    ++ cnt;
	    return cnt;
	}

	public Node  attribute( QName name ) {
	    for(BaseNode attr = attributes; attr != null; attr = attr.nextSibling)
		if( attr.getNodeName() == name && attr.getNature() == Node.ATTRIBUTE )
		    return attr;
	    return null;
	}

	public void setAttributes( Attribute attr ) {
	    attributes = attr;
	    for( ; attr != null; attr = (Attribute) attr.nextSibling)
		attr.parent = this;
	}

	public void addAttribute( Attribute attr ) {
	    if( attributes == null )
		attributes = attr;
	    else // need to check duplicates anyway
		for(Attribute last = attributes; ; last = (Attribute) last.nextSibling) {
		    if(last.name == attr.name) {
			last.value = attr.value;
			return;
		    }
		    if(last.nextSibling == null) {
			last.nextSibling = attr;
			break;
		    }
		}
	    attr.parent = this;
	}

	public void addText( String value ) {
	    throw new RuntimeException("addText on an element");
	}
    }

    static class Document extends Element {
	Document() {
	    super(null);
	}

	public String getNodeKind() {
	    return "document";
	}
	
	public int   getNature() {
	    return Node.DOCUMENT;
	}
    }

    static class TextNode extends BaseNode
    {
	Object value;

	TextNode( Object value ) {
	    this.value = value;
	    ++ CreatedText;
	}

	TextNode() {
	    this("");
	}

	public String getNodeKind() {
	    return "text";
	}

	public int getNature() {
	    return Node.TEXT;
	}

	public void addText( String value ) {
	    String thisText = getStringValue();
	    if( thisText == null || thisText.length() == 0)
		this.value = value;
	    else
		this.value = thisText + value;
	}

	public String getStringValue() {
	    return value.toString();
	}

	public NodeSequence children() {
	    return SSequence.empty;
	}

	public NodeSequence attributes() {
	    return SSequence.empty;
	}

	public NodeSequence namespaces( boolean inScope ) {
	    return SSequence.empty;
	}

	public String getNsPrefix( String nsuri ) {
	    return parent == null ? null : parent.getNsPrefix(nsuri);
	}

	public String getNsUri( String prefix ) {
	    return parent == null ? null : parent.getNsUri(prefix);
	}

	public boolean isElement() {
	    return false;
	}

	public boolean isWhitespace() {
	    return false;	//TODO
	}

	public Node  attribute( QName name ) {
	    return null;
	}

	public char[] getChars() {
	    return getStringValue().toCharArray();
	}

	public Object getAtomValue() {
	    return value;
	}
    }

    static class PINode extends TextNode
    {
	String target = "";

	PINode( String target, String value ) {
	    super(value);
	    this.target = target;
	}

	public String getNodeKind() {
	    return "processing-instruction";
	}

	public int getNature() {
	    return Node.PROCESSING_INSTRUCTION;
	}

	public QName  getNodeName() {
	    return QName.get(Namespace.NONE, target);
	}
    }

    static class CommentNode extends TextNode
    {
	CommentNode( String value ) {
	    super(value);
	}

	public String getNodeKind() {
	    return "comment";
	}

	public int getNature() {
	    return Node.COMMENT;
	}
    }

    // Attribute and namespace nodes: 
    static class Attribute extends TextNode
    {
	QName name;

	Attribute(QName name) {
	    this.name = name;
	}

	public QName  getNodeName() {
	    return name;
	}

	public String getNodeKind() {
	    return "attribute";
	}

	public int getNature() {
	    return Node.ATTRIBUTE;
	}
    }

    // namespace Node:
    static class NSNode extends Attribute {
	boolean local;	// defined locally (not simply inscope)

	NSNode(String prefix) {
	    super(QName.get(Namespace.NONE, prefix));
	}

	public String getNodeKind() {
	    return "namespace";
	}

	public int getNature() {
	    return Node.NAMESPACE;
	}
    }

    // sibling sequence
    static class SSequence implements NodeSequence
    {
	boolean started = false;
	BaseNode current, first;

	SSequence( BaseNode first ) {
	    this.first = this.current = first;
	}

	public NodeSequence  reborn() {
	    return new SSequence( first );
	}

	public boolean nextNode() {
	    if(started && current != null)
		current = current.nextSibling;
	    started = true;
	    return current != null;
	}

	public Node   currentNode() {
	    return current;
	}

	static SSequence empty = new SSequence(null);
    }

    static class Namespaces extends SSequence {
	Namespaces( BaseNode first ) {
	    super(first);
	    current = ((Element) first).attributes;
	}
	
	public boolean nextNode() {
	    for(; current != null; ) {
		if(started)
		    current = current.nextSibling;
		started = true;
		if(current instanceof NSNode)
		    return true;
	    }
	    return false;
	}
    }

    static abstract class TypedSequence extends SSequence {
	NodeTest nodeTest;

	TypedSequence( BaseNode first, NodeTest nodeTest ) {
	    super(first);
	    this.nodeTest = nodeTest;
	}

	boolean checkNode() {
	    return current != null &&
		   (nodeTest == null || nodeTest.accepts(current));
	}
    }

    static class Parent extends TypedSequence
    {
	Parent( BaseNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	}

	public boolean nextNode() {
	    if(started)
		return false;
	    started = true;
	    current = current.parent;
	    return checkNode();
	}
	
	public NodeSequence  reborn() {
	    return new Parent( first, nodeTest );
	}
    }

    static class AncestorsOrSelf extends TypedSequence
    {
	boolean started = false;

	AncestorsOrSelf( BaseNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	}

	public boolean nextNode() {
	    for(; current != null; ) {
		if(started)
		    current = current.parent;
		started = true;
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public NodeSequence  reborn() {
	    return new AncestorsOrSelf( first, nodeTest );
	}
    }

    static class Ancestors extends AncestorsOrSelf
    {
	Ancestors( BaseNode first, NodeTest nodeTest ) {
	    super( first, nodeTest);
	    started = true;
	}

	public NodeSequence  reborn() {
	    return new Ancestors( first, nodeTest );
	}
    }

    static class Children extends TypedSequence
    {
	Children( BaseNode first, NodeTest nodeTest ) {
	    super( first, nodeTest);
	}

	public boolean nextNode() {
	    for(; current != null; ) {
		if(!started)
		     current = current.getFirstChild();
		else current = current.nextSibling;
		started = true;
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public NodeSequence  reborn() {
	    return new Children( first, nodeTest );
	}
    }

    static class DescendantsOrSelf extends TypedSequence
    {
	Node lastNode;
	boolean started = false;

	DescendantsOrSelf( BaseNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	    lastNode = first.nodeAfter();
	}

	public boolean nextNode() {
	    for(; current != null; ) {
		if(started)
		    current = current.nodeNext();
		started = true;
		if(current == lastNode)
		    current = null;
		else if(checkNode())
		    return true;
	    }
	    return false;
	}

	public NodeSequence  reborn() {
	    return new DescendantsOrSelf( first, nodeTest );
	}
    }

    static class Descendants extends DescendantsOrSelf
    {
	Descendants( BaseNode first, NodeTest nodeTest ) {
	    super( first, nodeTest);
	    started = true;
	}

	public NodeSequence  reborn() {
	    return new Descendants( first, nodeTest );
	}
    }

    static class FollowingSiblings extends TypedSequence
    {
	FollowingSiblings( BaseNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	}

	public boolean nextNode() {
	    for( ; current != null; ) {
		current = current.nextSibling;
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public NodeSequence  reborn() {
	    return new FollowingSiblings( first, nodeTest );
	}
    }

    static class Following extends TypedSequence
    {
	Following( BaseNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	}

	public boolean nextNode() {
	    for( ; current != null; ) {
		current = current.nodeNext();
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public NodeSequence  reborn() {
	    return new Following( first, nodeTest );
	}
    }

    static class PrecedingSiblings extends TypedSequence
    {
	PrecedingSiblings( BaseNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	    current = first.parent == null ? null : first.parent.firstChild;
	}

	public boolean nextNode() {
	    for( ; current != null; ) {
		if(started) 
		    current = current.nextSibling;
		started = true;
		if(current == first) {
		    current = null; return false;
		}
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public NodeSequence  reborn() {
	    return new PrecedingSiblings( first, nodeTest );
	}
    }

    static class Preceding extends TypedSequence
    {
	Preceding( BaseNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	    current = (BaseNode) first.document();
	}

	public boolean nextNode() {
	    for( ; current != null; ) {
		if(started) 
		    current = current.nodeNext();
		started = true;
		if(current == first) {
		    current = null; return false;
		}
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public NodeSequence  reborn() {
	    return new Preceding( first, nodeTest );
	}
    }

    static class Attributes extends TypedSequence
    {
	Attributes( BaseNode first, NodeTest nodeTest ) {
	    super(first, nodeTest);
	    if(first.getNature() == Node.ELEMENT)
		current = ((Element) first).attributes;
	}

	public NodeSequence  reborn() {
	    return new Attributes( first, nodeTest );
	}

	public boolean nextNode() {
	    for( ; current != null; ) {
		if(started) 
		    current = current.nextSibling;
		started = true;
		if(current != null && current.getNature() == Node.ATTRIBUTE && checkNode())
		    return true;
	    }
	    return false;
	}
    }
}
