/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.app;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.dm.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.impl.DefaultEvalContext;
import net.xfra.qizxopen.xquery.dt.QNameValue;
import net.xfra.qizxopen.xquery.fn.JavaFunction;

import java.io.*;
import java.util.Vector;
import java.net.URL;

/**
 *	Non-Regression Test engine.
 *	<p>
 *	Works by interpreting special test files .xqt: an XQT file basically
 *	contains a series of groups like:<ul>
 *	<li>XML query,
 *	<li>followed by a special marker '=>' 
 *	<li>followed by the expected result of the query (in text form)
 *	<li>followed by '##'. </ul> 
 *	<p>Each query is executed, and its result is compared to the expected result, 
 *	if there are differences, they are displayed, and the test fails.
 *	<p>An XQT file may also contain directives:<ul>
 *	<li>#include <file>	
 *	<li>#docinput <url>	define input() as a document
 *	<li>#collinput <url>	define input() as a collection
 *	<li>#defvar name \n expr \n##	define/set a global in the predefined
 *	module (so that it is accessible to all queries)
 *	
 */
public class NonReg extends XQueryProcessor
{
    static public void usage( )
    {
        System.err.println("usage: NonReg <options> <testfile>... ");
        System.err.println("options:");
        System.err.println("  -v           verbose mode");
        System.err.println("  -q           quiet mode");
        System.err.println("  -e           dump query");
        System.err.println("  -diff        no diff: show actual output");
        System.err.println("  -jt          trace Java bindings");
        System.err.println("  -tex         trace exceptions");
    }

    static public void main( String args[] )
    {
        try {
	    NonReg tester = new NonReg(".");

            for(int a = 0; a < args.length; a++) {
                String arg = args[a];
                if(arg.equals("-path")) {
                    if(++ a >= args.length) usage();
		    tester = new NonReg(args[a]);
                }
                else if(arg.equals("-v")) {
                    tester.verbose = true;
                }
                else if(arg.equals("-q")) {
                    tester.verbose = false;
                }
		else if(arg.equals("-i")) {
		    tester.nodeDisplay.setIndent(-1);
		}
		else if(arg.equals("-diff")) {
		    tester.displayDiff = false;
		}
		else if(arg.equals("-e")) {
		    tester.exprDump = true;
		}
		else if(arg.equals("-jt")) {
		    JavaFunction.trace = true;
		}
		else if(arg.equals("-tex")) {
		    tester.traceExceptions = true;
		}
		else {
		    tester.execFile( arg );
		    System.err.println(arg+" done");
		}
            }
            
        } catch (Exception e) {
            e.printStackTrace();
	    System.err.println("ERROR: "+e);
        }
    }

    public NonReg( String dataPath ) throws IOException {
	super( dataPath, dataPath );
	nodeDisplay = new XMLSerializer();
	nodeDisplay.setIndent(4);
	nodeDisplay.setDepth(-1);
	setDefaultOutput(output);
	nodeDisplay.setOutput(output);
	this.dataPath = dataPath;
    }

    String dataPath = ".";
    XMLSerializer nodeDisplay;
    boolean display = true, exprDump = false, verbose = false, displayDiff = true;
    Input currentInput;
    String curQuery = "";
    String currentTestName = null;
    int    queryLine = -1;
    String queryFile;
    boolean firstDiag;
    boolean wrappedDisplay = true;
    boolean traceExceptions = false;
    Vector refData = new Vector();
    StringWriter stringOut = new StringWriter(200);
    PrintWriter output = new PrintWriter(stringOut);
    Log log = new Log(output);

    final static String INCLUDE_DIRECTIVE = "#include ";
    final static String DOCINPUT_DIRECTIVE = "#docinput ";
    final static String COLLINPUT_DIRECTIVE = "#collinput ";
    final static String DEFVAR_DIRECTIVE = "#defvar ";
    final static String OPTION_DIRECTIVE = "#option ";
    final static String TESTNAME_DIRECTIVE = "#test ";
    final static String OUTPUT_MARK = "=>";
    final static String EOOUTPUT_MARK = "##";
    final static String COMMENT_MARK = "//";

    static class Input {
	Input( Input up, String uri ) {
	    this.up = up; this.uri = uri;
	}
	Input up;
	BufferedReader reader;
	int line = 0;
	String uri;
    }

    /**
     *	Reads and interprest a XQT file.
     */
    public void execFile( String uri ) throws IOException, XQueryException {
	resetDeclarations();	// weird bug
	pushInput(uri);
	for(;;) {
	    String L = readLine();
	    if(L == null)
		break;
	    if(L.length() == 0 || L.startsWith(COMMENT_MARK))
		continue;
	    if(L.startsWith(INCLUDE_DIRECTIVE))
		pushInput(L.substring(INCLUDE_DIRECTIVE.length()).trim());
	    else if(L.startsWith(DOCINPUT_DIRECTIVE))
		documentInput(L.substring(DOCINPUT_DIRECTIVE.length()).trim());
	    else if(L.startsWith(COLLINPUT_DIRECTIVE))
		collectionInput(L.substring(COLLINPUT_DIRECTIVE.length()).trim());
	    else if(L.startsWith(DEFVAR_DIRECTIVE)) {
		String varname = L.substring(DEFVAR_DIRECTIVE.length()).trim();
		declareGlobal(varname, "TEST");
	    }
	    else if(L.startsWith(OPTION_DIRECTIVE)) {
		String[] args = L.substring(DEFVAR_DIRECTIVE.length()).split("[ \t\r]+");
		if(args.length == 0)
		    continue;
		//for(int j = 0; j < args.length; j++) System.err.println(args[j]+"|");
		if(args[0].equals("wrapped"))
		    wrappedDisplay = (args.length == 1 || args[1].equals("yes"));
	    }
	    else if(L.startsWith(TESTNAME_DIRECTIVE)) {
		currentTestName = L.substring(TESTNAME_DIRECTIVE.length()).trim();
	    }
	    else if(L.startsWith(OUTPUT_MARK)) {
		refData.setSize(0);
		for(;;) {
		    L = readLine();
		    if(L == null || L.trim().equals(EOOUTPUT_MARK))
			break;
		    refData.add(L);
		}
		if(L == null)
		   error("missing '##'");
		try {
		    execQuery(curQuery, "TEST");
		    
		} catch (XQueryException e) {
		    //error(e.toString());
		} catch (Exception e) {
		    error(e.toString());
		    e.printStackTrace();
		}
		compare( getOutput(), refData );
		clearOutput();
		curQuery = "";
		currentTestName = null;
		queryLine = -1;
	    }
	    else if(L.charAt(0) != '#') {
		// part of executable query
		curQuery += L + '\n';
		if(queryLine < 0) {
		    queryLine = currentInput.line;
		    queryFile = currentInput.uri;
		}
	    }
	    else error("bad line '"+L+"'");
	}
	if(curQuery.trim().length() > 0)
	    error("non-executed query at end? ");
	
    }

    void error( String msg ) {
	System.err.println("*** at line "+currentInput.line+" in "+currentInput.uri+": "+msg);
    }

    String readLine() throws IOException {
	if(currentInput == null)
	    return null;
	String L = currentInput.reader.readLine();
	while(L == null && currentInput.up != null) {
	    currentInput = currentInput.up;
	    L = currentInput.reader.readLine();
	}
	++ currentInput.line;
	return L;
    }

    void pushInput( String uri ) throws IOException {
	File furi = new File(uri);
	File file = furi.isAbsolute()? furi : new File(new File(dataPath), uri);
	// source code in Lat1:
	Reader rd = new InputStreamReader(new FileInputStream(file), "ISO8859_1");
	currentInput = new Input(currentInput, uri);
	currentInput.reader = new BufferedReader(rd);
    }

    void documentInput( String uri ) throws IOException, XQueryException {
	setDocumentInput(uri);
    }

    void collectionInput( String uri ) {
	System.err.println("collection Input |"+uri+"|");
    }

    void declareGlobal( String varname, String URI ) throws IOException {
	String def = "", L;
	for(;;) {
	    L = readLine();
	    if(L == null || L.equals(EOOUTPUT_MARK))
		break;
	    if(L.startsWith(INCLUDE_DIRECTIVE))
		pushInput(L.substring(INCLUDE_DIRECTIVE.length()).trim());
	    else
		def += L + '\n';
	}
	if(L == null)
	    error("missing '##'");
	

	try {
	    log.reset();
	    Query query = compileQuery( def, URI, log );
	    if(log.getErrorCount() != 0) {
		System.err.println("*** variable "+varname+" not defined");
		return;
	    }
	    Value v = executeQuery( query );
	    predefineGlobal(QName.get(varname), query.getType());
	    initGlobal(QName.get(varname), v);
	}
	catch (XQueryException e) {
	    error(e.getMessage());
	    e.printStackTrace();
	}
    }

    void execQuery( String textInput, String URI ) throws XQueryException {
	//error("execute "+textInput);

	try {
	    long T0 = System.currentTimeMillis();
	    log.reset();
	    Query query = compileQuery( textInput, URI, log );

	    if(exprDump && query != null)
		query.dump(new ExprDump());
	    long T1 = System.currentTimeMillis() - T0;
	    if(verbose)
		System.err.println("parsing & static check time(ms): "+ T1);
	    if(log.getErrorCount() != 0) {
		if(verbose)
		    System.err.println(log.getErrorCount() + " parsing/static error(s)");
		return;
	    }

	    nodeDisplay.definePrefixHints( query.getInScopeNS() );

	    T0 = System.currentTimeMillis();
	    Value v = executeQuery( query );
	    T1 = System.currentTimeMillis();
	    int itemX = 0;
	    if(wrappedDisplay) output.println("<results>");
	    for( ; v.next(); ) {
		++ itemX;
		if(display) {
		    Item item = v.asItem();
		    Type type = item.getType();
		    if(wrappedDisplay) output.print(" <item type='"+type+"'>");
		    
		    if( Type.NODE.accepts(item.getType()) ) {
			nodeDisplay.output(v.asNode());
		    }
		    else if(item.getType() == Type.QNAME) {
			QName qname = ((QNameValue) item).getValue();
			output.print(query.prefixedName(qname));
		    }
		    else		
			output.print(v.asString());
		    if(wrappedDisplay) output.println("</item>");
		}
	    }
	    if(wrappedDisplay) output.println("</results>");
	    long T2 = System.currentTimeMillis();
	    output.flush();
	    if(verbose) System.err.println("eval time(ms): "+ (T1-T0)+ " display time "+(T2-T1));
	}
	catch (EvalException e) {
	    if(traceExceptions)
		e.printStackTrace();
	    e.printStack(log, 20);
	    if(e.getCause() != null && !traceExceptions) {
		System.err.println("  caused by: " + e.getCause());
	    }
	}
	catch (SyntaxException lexe) {
	    output.println("** Syntax: " + lexe.getMessage());
	}
	catch (DataModelException dme) {
	    dme.printStackTrace();	// should not happen ?
	}
    }

    void diag(String pref, String msg) {
	if(firstDiag) {
	    System.out.print("====== test");
	    if(currentTestName != null)
		System.out.print(" "+currentTestName);
	    System.out.println(" line "+queryLine+" in "+queryFile+" fails: ======");

	    firstDiag = false;
	}
	System.out.print(pref);
	System.out.println(msg);
    }

    void compare( String output, Vector reference ) {
	firstDiag = true;
	boolean ok = true, crlf = (output.indexOf("\r\n") >= 0);
	String[] sout = output.split(crlf ? "\r\n" : "\n");
	int L = Math.min(sout.length, reference.size()), i = 0;
	for( ; i < L; i++) {
	    String ref = (String) reference.get(i);
	    if(!sout[i].equals(ref)) 
		if (displayDiff) {
		    diag("LINE "+i+": ", sout[i]);
		    diag("EXPECT: ", ref);
		}
		else ok = false;
	}
	if(sout.length > L) 
	    if (displayDiff) {
		diag("## ", "UNEXPECTED:");
		for( ; i < sout.length; i++)
		    diag("", sout[i]);
	    }
	    else ok = false;
	else if(reference.size() > L)
	    if (displayDiff) {
		diag("## ", "MISSING:");
		for( ; i < reference.size(); i++)
		    diag("", (String) reference.get(i));
	    }
	    else ok = false;
	if(!ok && !displayDiff) {
	    diag("Actual output", ":");
	    System.err.print(output);
	}
    }

    String getOutput() {
	output.flush();
	return stringOut.toString();
    }

    void clearOutput() {
	stringOut.getBuffer().setLength(0);
    }

    public class Log extends net.xfra.qizxopen.xquery.Log
    {
	Log( PrintWriter output ) {
	    this.output = output;
	    printSource = false;
	}

	public void info( String message ) {	// discard info messages
	    //if(!displayDiff)
		super.info(message);
	}

    } // end of class Log


} // end of class NonReg

