/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.KeyEvent.*;
import static javax.swing.KeyStroke.getKeyStroke;
import static net.argius.stew.ui.window.Resource.getString;

import java.awt.*;
import java.awt.event.*;
import java.util.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.undo.*;

/**
 * ANṼ[eBeBB
 * R|[lg̃ANVĂȕ։B
 */
final class ActionUtility {

    private static final String UNDO = "undo";
    private static final String REDO = "redo";
    private static final String CUT_TO_CLIPBOARD = "cut-to-clipboard";
    private static final String COPY_TO_CLIPBOARD = "copy-to-clipboard";
    private static final String PASTE_FROM_CLIPBOARD = "paste-from-clipboard";
    private static final String SELECT_ALL = "select-all";

    private final JComponent c;
    private final ActionMap amap;
    private final InputMap imap;

    private ActionUtility(JComponent c) {
        this.c = c;
        this.amap = c.getActionMap();
        this.imap = c.getInputMap();
    }

    /**
     * CX^X̎擾B
     * @param c ΏۃR|[lg
     * @return CX^X
     */
    static ActionUtility getInstance(JComponent c) {
        return new ActionUtility(c);
    }

    /**
     * ANV蓖ĂB
     * @param action ANV
     * @return ANVL[
     */
    Object bindAction(Action action) {
        return bindAction(action, action.getValue(Action.NAME), null);
    }

    /**
     * ANV蓖ĂB
     * @param action ANV
     * @param actionKey ANVL[
     * @return ANVL[
     */
    Object bindAction(Action action, Object actionKey) {
        return bindAction(action, actionKey, null);
    }

    /**
     * ANV蓖ĂB
     * @param action ANV
     * @param actionKey ANVL[
     * @param keyStroke V[gJbgL[
     * @return ANVL[
     */
    Object bindAction(Action action, KeyStroke keyStroke) {
        return bindAction(action, action.getValue(Action.NAME), keyStroke);
    }

    /**
     * ANV蓖ĂB
     * @param action ANV
     * @param actionKey ANVL[
     * @param keyStroke V[gJbgL[
     * @return ANVL[
     */
    Object bindAction(Action action, Object actionKey, KeyStroke keyStroke) {
        amap.put(actionKey, action);
        imap.put(keyStroke, actionKey);
        return actionKey;
    }

    /**
     * ANV蓖ĂB
     * @param listener ANṼfBXpb`
     * @param cmdString R}h(ANVL[)
     * @param keyStrokes V[gJbgL[
     * @return ANVL[
     */
    Object bindAction(ActionCommandListener listener, String cmdString, KeyStroke... keyStrokes) {
        return bindAction(c, listener, cmdString, keyStrokes);
    }

    /**
     * ANV蓖ĂB
     * @param c ΏۃR|[lg
     * @param listener ANV̓]惊Xi
     * @param cmdString R}h(ANVL[)
     * @param keyStrokes V[gJbgL[
     * @return ANVL[
     */
    static Object bindAction(JComponent c,
                             final ActionCommandListener listener,
                             final String cmdString,
                             KeyStroke... keyStrokes) {
        InputMap im = c.getInputMap();
        for (final KeyStroke ks : keyStrokes) {
            im.put(ks, cmdString);
        }
        ActionMap am = c.getActionMap();
        Action action = new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
                listener.actionCommandPerform(e, cmdString);
            }
        };
        am.put(cmdString, action);
        if (keyStrokes.length == 0 && (c instanceof AbstractButton)) {
            ((AbstractButton)c).addActionListener(action);
        }
        return cmdString;
    }

    interface ActionCommandListener extends EventListener {
        void actionCommandPerform(ActionEvent e, String cmd);
    }

    /**
     * ̃R|[lg̃ANVƃV[gJbgJMenuItemɃRs[B
     * @param actionKey ANVL[
     * @param dst Rs[JMenuItem
     * @return dstƓIuWFNg
     */
    JMenuItem copyActionTo(Object actionKey, JMenuItem dst) {
        final JComponent c = this.c;
        final Action action = amap.get(actionKey);
        if (action != null) {
            dst.addActionListener(new ActionListener() {

                public void actionPerformed(ActionEvent e) {
                    e.setSource(c);
                    action.actionPerformed(e);
                }

            });
        }
        for (KeyStroke keyStroke : imap.allKeys()) {
            Object k = imap.get(keyStroke);
            if (k != null && k.equals(actionKey)) {
                dst.setAccelerator(keyStroke);
                break;
            }
        }
        return dst;
    }

    /**
     * ReLXgj[ANVXg琶Đݒ肷B
     * @param actionNames ANṼXg
     */
    void setContextMenu(String[] actionNames) {
        setContextMenu(actionNames, null);
    }

    /**
     * ReLXgj[ANVXg琶Đݒ肷B
     * @param actionNames ANṼXg
     * @param mnemonics j[jbN
     */
    void setContextMenu(String[] actionNames, char[] mnemonics) {
        c.setComponentPopupMenu(createPopupMenu(actionNames, mnemonics));
        bindContextMenuKey();
    }

    /**
     * ReLXgj[̕\AvP[VL[Ɋ蓖ĂB
     */
    private void bindContextMenuKey() {
        bindAction(new AbstractAction("context-menu") {

            public void actionPerformed(ActionEvent e) {
                final Object src = e.getSource();
                if (src instanceof JComponent) {
                    JComponent c = (JComponent)src;
                    Point p = c.getMousePosition();
                    if (p != null) {
                        c.getComponentPopupMenu().show(c, p.x, p.y);
                    }
                }
            }

        }, getKeyStroke(VK_CONTEXT_MENU, 0));
    }

    /**
     * ReLXgj[𐶐B
     * @param actionNames ANṼXg
     * @param mnemonics j[jbN
     * @return ReLXgj[
     */
    JPopupMenu createPopupMenu(String[] actionNames, char[] mnemonics) {
        return createPopupMenu(actionNames, mnemonics, null);
    }

    /**
     * ReLXgj[𐶐B
     * @param actionNames ANṼXg
     * @param mnemonics j[jbN
     * @param mapping ANVMenuItem̃}bsO
     * @return ReLXgj[
     */
    JPopupMenu createPopupMenu(String[] actionNames,
                               char[] mnemonics,
                               Map<String, JMenuItem> mapping) {
        JPopupMenu pmenu = new JPopupMenu();
        int index = 0;
        for (final String actionName : actionNames) {
            if (actionName == null || actionName.length() == 0) {
                pmenu.addSeparator();
            } else {
                final char mnemonic = (mnemonics == null) ? ' ' : mnemonics[index];
                JMenuItem item = new JMenuItem(getString("Action." + actionName, mnemonic));
                item.setMnemonic(mnemonic);
                pmenu.add(copyActionTo(actionName, item));
                if (mapping != null) {
                    mapping.put(actionName, item);
                }
            }
            ++index;
        }
        return pmenu;
    }

    /**
     * AhDEhDANVݒ肷B
     * @return UndoManager
     */
    UndoManager setUndoAction() {
        if (!(c instanceof JTextComponent)) {
            throw new IllegalComponentStateException(c.getClass().getName());
        }
        JTextComponent text = (JTextComponent)c;
        final UndoManager um = new UndoManager();
        text.getDocument().addUndoableEditListener(um);
        ActionMap amap = text.getActionMap();
        InputMap imap = text.getInputMap();
        final int shortcutKey = Resource.getMenuShortcutKeyMask();
        amap.put(UNDO, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                if (um.canUndo()) {
                    um.undo();
                }
            }

        });
        imap.put(getKeyStroke(VK_Z, shortcutKey), UNDO);
        amap.put(REDO, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                if (um.canRedo()) {
                    um.redo();
                }
            }

        });
        imap.put(getKeyStroke(VK_Y, shortcutKey), REDO);
        return um;
    }

    /**
     * AhDEhDANVݒ肷B
     * @param c Ώ
     * @return UndoManager
     */
    static UndoManager setUndoAction(JTextComponent c) {
        return new ActionUtility(c).setUndoAction();
    }

    /**
     * eLXgR|[lg̃ANVݒ肷B
     */
    void setActionForTextComponent() {
        if (!(c instanceof JTextComponent)) {
            throw new IllegalComponentStateException(c.getClass().getName());
        }
        final JTextComponent text = (JTextComponent)c;
        final UndoManager um = setUndoAction();
        char[] mnemonics = "UR TCPA".toCharArray();
        String[] actionNames = {UNDO, REDO, "", CUT_TO_CLIPBOARD, COPY_TO_CLIPBOARD,
                                PASTE_FROM_CLIPBOARD, SELECT_ALL,};
        final Map<String, JMenuItem> m = new HashMap<String, JMenuItem>();
        JPopupMenu pmenu = createPopupMenu(actionNames, mnemonics, m);
        pmenu.addPopupMenuListener(new PopupMenuListener() {

            public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
                m.get(UNDO).setEnabled(um.canUndo());
                m.get(REDO).setEnabled(um.canRedo());
                final boolean textSelected = text.getSelectionEnd() > text.getSelectionStart();
                m.get(CUT_TO_CLIPBOARD).setEnabled(textSelected);
                m.get(COPY_TO_CLIPBOARD).setEnabled(textSelected);
            }

            public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
                // empty
            }

            public void popupMenuCanceled(PopupMenuEvent e) {
                // empty
            }

        });
        text.setComponentPopupMenu(pmenu);
        bindContextMenuKey();
    }

    /**
     * eLXgR|[lg̃ANVݒ肷B
     * @param text eLXgR|[lg
     */
    static void setActionForTextComponent(JTextComponent text) {
        new ActionUtility(text).setActionForTextComponent();
    }

}
