/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.KeyEvent.*;
import static javax.swing.JOptionPane.*;
import static javax.swing.KeyStroke.getKeyStroke;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.util.List;

import javax.swing.*;

import net.argius.stew.*;
import net.argius.stew.ui.window.ActionUtility.ActionCommandListener;
import net.argius.stew.ui.window.TextSearch.Matcher;

/**
 * eLXgplB
 */
final class TextSearchPanel extends JPanel implements ActionCommandListener {

    private static final ResourceManager res = ResourceManager.getInstance(TextSearchPanel.class);
    private static final String cmdClose = "close";
    private static final String cmdSearch = "search";
    private static final String cmdForward = "forward";
    private static final String cmdBackward = "backward";

    private final List<TextSearch> targets;
    private final JTextField text;
    private final JCheckBox useRegexCheck;
    private final JCheckBox ignoreCaseCheck;

    private TextSearch currentTarget;
    private boolean searchBackward;

    TextSearchPanel(JFrame frame) {
        // [CX^X]
        this.targets = new ArrayList<TextSearch>();
        this.text = new JTextField(16);
        this.useRegexCheck = new JCheckBox(res.get("useregex"));
        this.ignoreCaseCheck = new JCheckBox(res.get("ignorecase"));
        setLayout(new FlowLayout(FlowLayout.LEFT, 1, 1));
        setVisible(false);
        final JButton close = new JButton(Resource.getImageIcon("close.png"));
        final JTextField text = this.text;
        final JButton forward = new JButton(res.get("forward"));
        final JButton backward = new JButton(res.get("backward"));
        final JCheckBox useRegexCheck = this.useRegexCheck;
        final JCheckBox ignoreCaseCheck = this.ignoreCaseCheck;
        // [CAEg]
        close.setToolTipText(res.get("close"));
        close.setMargin(new Insets(0, 0, 0, 0));
        text.setMargin(new Insets(1, 2, 1, 2));
        forward.setMargin(new Insets(0, 7, 0, 7));
        backward.setMargin(new Insets(0, 7, 0, 7));
        changeFontSize(useRegexCheck, 0.8f);
        changeFontSize(ignoreCaseCheck, 0.8f);
        add(createFiller(2));
        add(close);
        add(createFiller(2));
        add(new JLabel(res.get("label")));
        add(text);
        add(createFiller(1));
        add(forward);
        add(backward);
        add(useRegexCheck);
        add(ignoreCaseCheck);
        // [tH[JXݒ]
        final FocusTraversalPolicy parentPolicy = frame.getFocusTraversalPolicy();
        frame.setFocusTraversalPolicy(new LayoutFocusTraversalPolicy() {
            @Override
            public Component getComponentAfter(Container focusCycleRoot, Component component) {
                if (component == ignoreCaseCheck) {
                    return close;
                }
                return parentPolicy.getComponentAfter(focusCycleRoot, component);
            }
            @Override
            public Component getComponentBefore(Container focusCycleRoot, Component component) {
                if (component == close) {
                    return ignoreCaseCheck;
                }
                return parentPolicy.getComponentBefore(focusCycleRoot, component);
            }
        });
        // [Cxgݒ]
        for (JComponent c : new JComponent[]{close, text, forward, backward, useRegexCheck,
                                             ignoreCaseCheck}) {
            ActionUtility.bindAction(c, this, cmdClose, getKeyStroke(VK_ESCAPE, 0));
        }
        ActionUtility.bindAction(close, this, cmdClose);
        ActionUtility textau = ActionUtility.getInstance(text);
        textau.setActionForTextComponent();
        textau.bindAction(this, cmdSearch, getKeyStroke(VK_ENTER, 0));
        ActionUtility.bindAction(forward, this, cmdForward);
        ActionUtility.bindAction(backward, this, cmdBackward);
        addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent e) {
                validate();
            }
        });
    }

    /* @see net.argius.stew.ui.window.ActionUtility.ActionCommandListener#actionCommandPerform(java.awt.event.ActionEvent, java.lang.String) */
    public void actionCommandPerform(ActionEvent e, String cmd) {
        if (cmd.equals(cmdClose)) {
            setVisible(false);
        } else if (cmd.equals(cmdSearch)) {
            startSearch();
        } else if (cmd.equals(cmdForward)) {
            setSearchBackward(false);
            startSearch();
        } else if (cmd.equals(cmdBackward)) {
            setSearchBackward(true);
            startSearch();
        }
    }

    private static void changeFontSize(Component c, float rate) {
        Font f = c.getFont();
        c.setFont(f.deriveFont(Font.PLAIN, f.getSize() * rate));
    }

    private static Component createFiller(int width) {
        // tH[JX̊֌WJLabel̗p
        char[] a = new char[width];
        Arrays.fill(a, ' ');
        return new JLabel(String.valueOf(a));
    }

    @Override
    public void setVisible(boolean visible) {
        super.setVisible(visible);
        if (visible) {
            final int length = text.getText().length();
            if (length > 0) {
                text.setSelectionStart(0);
                text.setSelectionEnd(length);
            }
            text.requestFocus();
        } else {
            for (TextSearch target : targets) {
                target.reset();
            }
            if (currentTarget instanceof Component) {
                Component c = (Component)currentTarget;
                c.requestFocus();
            }
        }
    }

    @Override
    public void updateUI() {
        super.updateUI();
        if (text != null) {
            // XXX ̏ȂƃReLXgj[ɃtHgfȂ
            SwingUtilities.updateComponentTreeUI(text.getComponentPopupMenu());
        }
    }

    /**
     * Ώۂ̒ǉB
     * @param target ^[Qbg
     */
    void addTarget(TextSearch target) {
        targets.add(target);
    }

    /**
     * Ώۂ̍폜B
     * @param target ^[Qbg
     */
    void removeTarget(TextSearch target) {
        targets.remove(target);
    }

    /**
     * ݂̌Ώۂݒ肷B
     * @param currentTarget ݂̌Ώ
     */
    void setCurrentTarget(TextSearch currentTarget) {
        this.currentTarget = currentTarget;
    }

    /**
     * tɌ邩ǂݒ肷B
     * @param searchBackward tɌ邩ǂ
     */
    void setSearchBackward(boolean searchBackward) {
        this.searchBackward = searchBackward;
    }

    /**
     * JnB
     */
    void startSearch() {
        final String s = text.getText();
        if (s == null || s.length() == 0 || currentTarget == null) {
            return;
        }
        final boolean useRegularExpression = useRegexCheck.isSelected();
        final boolean ignoreCase = ignoreCaseCheck.isSelected();
        Matcher matcher = new Matcher(s, useRegularExpression, ignoreCase);
        matcher.setBackward(searchBackward);
        matcher.setContinuously(true);
        for (TextSearch target : targets) {
            target.reset();
        }
        final boolean found = currentTarget.search(matcher);
        if (!found) {
            Component parent;
            if (currentTarget instanceof Component) {
                parent = ((Component)currentTarget).getParent();
            } else {
                parent = getParent();
            }
            showMessageDialog(parent, res.get("message.notfound", s), null, WARNING_MESSAGE);
        }
    }

}
