/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.ActionEvent.ACTION_PERFORMED;
import static net.argius.stew.ui.window.Menu.Item.*;
import static net.argius.stew.ui.window.Resource.*;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.util.*;
import java.util.List;

import javax.swing.*;

import net.argius.stew.*;

/**
 * EBhE\̃j[o[B
 */
final class Menu extends JMenuBar implements PropertyChangeListener {

    private static final ResourceManager res = ResourceManager.getInstance(Menu.class);

    /**
     * j[ځB
     */
    enum Item {
        NEW,
        CLOSE,
        QUIT,
        CUT,
        COPY,
        PASTE,
        SELECT_ALL,
        FIND,
        TOGGLE_FOCUS,
        CLEAR_MESSAGE,
        STATUS_BAR,
        INFO_TREE,
        COLUMN_NUMBER,
        ALWAYS_ON_TOP,
        REFRESH,
        WIDEN_COLUMN_WIDTH,
        NARROW_COLUMN_WIDTH,
        ADJUST_COLUMN_WIDTH,
        AUTO_ADJUST_MODE,
        AUTO_ADJUST_MODE_NONE,
        AUTO_ADJUST_MODE_HEADER,
        AUTO_ADJUST_MODE_VALUE,
        AUTO_ADJUST_MODE_HEADERANDVALUE,
        EXECUTE,
        BREAK,
        HISTORY_BACK,
        HISTORY_NEXT,
        RETRIEVE_HISTORY,
        ROLLBACK,
        COMMIT,
        CONNECT,
        DISCONNECT,
        POST_PROCESS_MODE,
        POST_PROCESS_MODE_NONE,
        POST_PROCESS_MODE_FOCUS,
        POST_PROCESS_MODE_SHAKE,
        POST_PROCESS_MODE_BLINK,
        ENCRYPTION_KEY,
        EDIT_CONNECTORS,
        SORT,
        IMPORT,
        EXPORT,
        HELP,
        ABOUT;
    }

    private List<JMenuItem> lockingTargets;
    private List<JMenuItem> unlockingTargets;
    private EnumMap<Item, JMenuItem> itemToCompMap;
    private Map<JMenuItem, Item> compToItemMap;

    Menu() {
        this.lockingTargets = new ArrayList<JMenuItem>();
        this.unlockingTargets = new ArrayList<JMenuItem>();
        this.itemToCompMap = new EnumMap<Item, JMenuItem>(Item.class);
        this.compToItemMap = new HashMap<JMenuItem, Item>();
        final boolean autoMnemonic = res.getInt("auto-mnemonic") == 1;
        final ActionListener action = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                fireActionPerformed(e.getSource(),
                                    Item.valueOf(e.getActionCommand().replace('.', '_')));
            }
        };
        for (final String groupId : res.get("groups").split(",", -1)) {
            final String groupKey = "group." + groupId;
            JMenu group = add(buildGroup(groupId, autoMnemonic));
            for (final String itemId : res.get(groupKey + ".items").split(",", -1)) {
                if (itemId.length() == 0) {
                    group.add(new JSeparator());
                    continue;
                }
                JMenuItem m = group.add(buildItem(itemId, autoMnemonic));
                m.addActionListener(action);
                Item item;
                try {
                    item = Item.valueOf(itemId);
                } catch (Exception ex) {
                    assert false : ex.toString();
                    continue;
                }
                itemToCompMap.put(item, m);
                compToItemMap.put(m, item);
                final String shortcutId = "item." + itemId + ".shortcut";
                if (res.containsKey(shortcutId)) {
                    KeyStroke shortcutKey = KeyStroke.getKeyStroke(res.get(shortcutId));
                    if (shortcutKey != null) {
                        setAccelerator(item, shortcutKey);
                    }
                }
                switch (item) {
                    case CLOSE:
                    case QUIT:
                    case CUT:
                    case COPY:
                    case PASTE:
                    case SELECT_ALL:
                    case FIND:
                    case CLEAR_MESSAGE:
                    case REFRESH:
                    case WIDEN_COLUMN_WIDTH:
                    case NARROW_COLUMN_WIDTH:
                    case ADJUST_COLUMN_WIDTH:
                    case AUTO_ADJUST_MODE:
                    case EXECUTE:
                    case HISTORY_BACK:
                    case HISTORY_NEXT:
                    case CONNECT:
                    case DISCONNECT:
                    case SORT:
                    case EXPORT:
                        lockingTargets.add(m);
                        break;
                    case BREAK:
                        unlockingTargets.add(m);
                        break;
                    default:
                }
            }
        }
        for (final Item item : EnumSet.of(AUTO_ADJUST_MODE_NONE,
                                          AUTO_ADJUST_MODE_HEADER,
                                          AUTO_ADJUST_MODE_VALUE,
                                          AUTO_ADJUST_MODE_HEADERANDVALUE,
                                          POST_PROCESS_MODE_NONE,
                                          POST_PROCESS_MODE_FOCUS,
                                          POST_PROCESS_MODE_SHAKE,
                                          POST_PROCESS_MODE_BLINK)) {
            itemToCompMap.get(item).addActionListener(action);
        }
        setEnabledStates(false);
    }

    private static JMenu buildGroup(String groupId, boolean autoMnemonic) {
        final String key = (res.containsKey("group." + groupId) ? "group" : "item") + '.' + groupId;
        final char mn = res.getChar(key + ".mnemonic");
        final String groupString = autoMnemonic ? res.get(key, mn) : res.get(key);
        JMenu group = new JMenu(groupString);
        group.setMnemonic(mn);
        return group;
    }

    private JMenuItem buildItem(String itemId, boolean autoMnemonic) {
        final String itemKey = "item." + itemId;
        final char mnemonic = res.getChar(itemKey + ".mnemonic");
        final JMenuItem m;
        if (res.isTrue(itemKey + ".checkbox")) {
            m = new JCheckBoxMenuItem();
        } else if (res.isTrue(itemKey + ".subgroup")) {
            m = buildGroup(itemId, autoMnemonic);
            ButtonGroup buttonGroup = new ButtonGroup();
            boolean selected = false;
            for (final String id : res.get(itemKey + ".items").split(",", -1)) {
                final JMenuItem sub = buildItem(itemId + '.' + id, autoMnemonic);
                m.add(sub);
                buttonGroup.add(sub);
                if (!selected) {
                    sub.setSelected(true);
                    selected = true;
                }
                Item subItem = Item.valueOf(itemId + '_' + id);
                itemToCompMap.put(subItem, sub);
                compToItemMap.put(sub, subItem);
            }
        } else {
            m = new JMenuItem();
        }
        m.setText(autoMnemonic ? res.get(itemKey, mnemonic) : res.get(itemKey));
        m.setMnemonic(mnemonic);
        m.setActionCommand(itemId);
        m.setIcon(getImageIcon(String.format("menu-%s.png", itemId)));
        m.setDisabledIcon(getImageIcon(String.format("menu-disabled-%s.png", itemId)));
        return m;
    }

    /* @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent) */
    public void propertyChange(PropertyChangeEvent e) {
        final String propertyName = e.getPropertyName();
        final Object source = e.getSource();
        if (source instanceof JLabel && propertyName.equals("ancestor")) {
            itemToCompMap.get(STATUS_BAR).setSelected(((JLabel)source).isVisible());
        } else if (source instanceof ResultSetTable && propertyName.equals("showNumber")) {
            itemToCompMap.get(COLUMN_NUMBER).setSelected((Boolean)e.getNewValue());
        } else if (source instanceof DatabaseInfoTree) {
            itemToCompMap.get(INFO_TREE).setSelected(((Component)source).isEnabled());
        } else if (source instanceof JFrame && propertyName.equals("alwaysOnTop")) {
            itemToCompMap.get(ALWAYS_ON_TOP).setSelected((Boolean)e.getNewValue());
        } else if (source instanceof WindowOutputProcessor && propertyName.equals("autoAdjustMode")) {
            itemToCompMap.get(Item.valueOf(e.getNewValue().toString())).setSelected(true);
        } else if (source instanceof WindowOutputProcessor
                   && propertyName.equals("postProcessMode")) {
            itemToCompMap.get(Item.valueOf(e.getNewValue().toString())).setSelected(true);
        }
    }

    /**
     * V[gJbgL[̐ݒB
     * @param item
     * @param keyStroke
     */
    void setAccelerator(Item item, KeyStroke keyStroke) {
        final int shortcutKey = getMenuShortcutKeyMask();
        final boolean notCtrlKey = (shortcutKey & InputEvent.CTRL_DOWN_MASK) != 0;
        int modifiers = keyStroke.getModifiers();
        if (notCtrlKey && (modifiers & InputEvent.CTRL_DOWN_MASK) != 0) {
            modifiers ^= InputEvent.CTRL_DOWN_MASK;
            modifiers &= shortcutKey;
        }
        final KeyStroke ks = KeyStroke.getKeyStroke(keyStroke.getKeyCode(), modifiers);
        itemToCompMap.get(item).setAccelerator(ks);
    }

    /**
     * eڂ̗LԂݒ肷B
     * @param commandStarted R}hJnǂ
     */
    void setEnabledStates(boolean commandStarted) {
        final boolean lockingTargetsState = !commandStarted;
        for (JMenuItem item : lockingTargets) {
            item.setEnabled(lockingTargetsState);
        }
        final boolean unlockingTargetsState = commandStarted;
        for (JMenuItem item : unlockingTargets) {
            item.setEnabled(unlockingTargetsState);
        }
    }

    /**
     * ActionListener̒ǉB
     * @param listener ActionListener
     */
    void addActionListener(ActionListener listener) {
        assert listener != null;
        listenerList.add(ActionListener.class, listener);
    }

    /**
     * ActionListener̍폜B
     * @param listener ActionListener
     */
    void removeActionListener(ActionListener listener) {
        assert listener != null;
        listenerList.remove(ActionListener.class, listener);
    }

    /**
     * j[IꂽƂʒmB
     * @param source \[XIuWFNg
     * @param item j[
     */
    void fireActionPerformed(Object source, Item item) {
        try {
            ActionEvent event = new ActionEvent(source, ACTION_PERFORMED, item.toString());
            for (ActionListener listener : listenerList.getListeners(ActionListener.class)) {
                listener.actionPerformed(event);
            }
        } catch (Exception ex) {
            WindowOutputProcessor.showErrorDialog(getParent(), ex);
        }
    }

}
