/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.InputEvent.*;
import static java.awt.event.KeyEvent.*;
import static javax.swing.JOptionPane.*;
import static javax.swing.JSplitPane.VERTICAL_SPLIT;
import static javax.swing.KeyStroke.getKeyStroke;
import static javax.swing.ScrollPaneConstants.*;
import static net.argius.stew.Iteration.join;
import static net.argius.stew.ui.window.Resource.*;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.dnd.*;
import java.awt.event.*;
import java.beans.*;
import java.io.*;
import java.lang.Thread.*;
import java.nio.channels.*;
import java.sql.*;
import java.util.*;
import java.util.List;
import java.util.Timer;
import java.util.Map.*;
import java.util.concurrent.*;
import java.util.regex.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.text.*;
import javax.swing.text.Highlighter.*;

import net.argius.logging.*;
import net.argius.stew.*;
import net.argius.stew.io.*;
import net.argius.stew.ui.*;
import net.argius.stew.ui.window.Menu.*;

/**
 * EBhE[hNB
 */
public final class WindowLauncher implements Launcher {

    private static final Logger log = LoggerFactory.getLogger(WindowLauncher.class);

    private static final List<WindowLauncher> instances = Collections.synchronizedList(new ArrayList<WindowLauncher>());
    private static final ImageIcon ICON = Resource.getImageIcon("stew.png");

    JComponent focused;

    private final WindowOutputProcessor op;
    private final Menu menu;
    private final JPanel panel1;
    private final JPanel panel2;
    private final JSplitPane split1;
    private final JSplitPane split2;
    private final ResultSetTable resultSetTable;
    private final ConsoleTextArea textArea;
    private final DatabaseInfoTree infoTree;
    private final TextSearchPanel textSearchPanel;
    private final JLabel statusBar;
    private final List<String> historyList;
    private final ExecutorService executorService;

    private Environment env;
    private Map<JComponent, TextSearch> textSearchMap;
    private int historyIndex;

    WindowLauncher() {
        // [CX^X]
        instances.add(this);
        final JSplitPane split1 = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        final DatabaseInfoTree infoTree = new DatabaseInfoTree();
        final ResultSetTable resultSetTable = new ResultSetTable();
        final ConsoleTextArea textArea = new ConsoleTextArea();
        this.op = new WindowOutputProcessor(resultSetTable, textArea);
        this.menu = new Menu();
        this.panel1 = new JPanel(new BorderLayout());
        this.panel2 = new JPanel(new BorderLayout());
        this.split1 = split1;
        this.split2 = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
        this.resultSetTable = resultSetTable;
        this.textArea = textArea;
        this.infoTree = infoTree;
        this.textSearchPanel = new TextSearchPanel(op);
        this.statusBar = new JLabel(" ");
        this.historyList = new LinkedList<String>();
        this.historyIndex = 0;
        this.executorService = Executors.newScheduledThreadPool(3);
        // [R|[lg̐ݒ]
        // OutputProcessor as frame
        op.setTitle(Resource.getString(".title"));
        op.setIconImage(ICON.getImage());
        op.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        // splitpane (infotree and sub-splitpane)
        split1.setResizeWeight(0.6f);
        split1.setDividerSize(4);
        // splitpane (table and textarea)
        split2.setOrientation(VERTICAL_SPLIT);
        split2.setDividerSize(6);
        split2.setResizeWeight(0.6f);
        // text area
        textArea.setMargin(new Insets(4, 8, 4, 4));
        textArea.setLineWrap(true);
        textArea.setWrapStyleWord(false);
        // text search
        this.textSearchMap = new LinkedHashMap<JComponent, TextSearch>();
        textSearchMap.put(resultSetTable, createTextSearch(resultSetTable));
        textSearchMap.put(textArea, createTextSearch(textArea));
        for (Entry<JComponent, TextSearch> entry : textSearchMap.entrySet()) {
            final JComponent c = entry.getKey();
            c.addFocusListener(new FocusAdapter() {

                @Override
                public void focusGained(FocusEvent e) {
                    focused = c;
                }

            });
            textSearchPanel.addTarget(entry.getValue());
        }
        // status bar
        statusBar.setForeground(Color.BLUE);
        // [CAEg]
        /*
         * split2 = ʃe[u + ̓GA 
         * +----------------------------+
         * | split2                     |
         * | +------------------------+ |
         * | | scroll(resultSetTable) | |
         * | +------------------------+ |
         * | +------------------------+ |
         * | | panel2                 | |
         * | | +--------------------+ | |
         * | | | scroll(textAarea)  | | |
         * | | +--------------------+ | |
         * | | | textSearchPanel    | | |
         * | | +--------------------+ | |
         * | +------------------------+ |
         * +----------------------------+
         * DBc[\
         * +-----------------------------------+
         * | panel1                            |
         * | +-------------------------------+ |
         * | | split1                        | |
         * | | +------------+ +------------+ | |
         * | | | scroll     | | split2     | | |
         * | | | (infoTree) | |            | | |
         * | | +------------+ +------------+ | |
         * | +-------------------------------+ |
         * +-----------------------------------+
         * | status bar                        |
         * +-----------------------------------+
         * DBc[\
         * +-----------------------------------+
         * | panel1                            |
         * | +-------------------------------+ |
         * | | split2                        | |
         * | +-------------------------------+ |
         * +-----------------------------------+
         * | status bar                        |
         * +-----------------------------------+
         */
        panel2.add(new JScrollPane(textArea, VERTICAL_SCROLLBAR_ALWAYS, HORIZONTAL_SCROLLBAR_NEVER),
                   BorderLayout.CENTER);
        panel2.add(textSearchPanel, BorderLayout.SOUTH);
        split2.setTopComponent(new JScrollPane(resultSetTable));
        split2.setBottomComponent(panel2);
        op.add(panel1, BorderLayout.CENTER);
        op.add(statusBar, BorderLayout.PAGE_END);
        op.setJMenuBar(menu);
        // [ۑꂽԂ̕]
        op.addPropertyChangeListener(menu);
        infoTree.addPropertyChangeListener(menu);
        resultSetTable.addPropertyChangeListener(menu);
        statusBar.addPropertyChangeListener(menu);
        loadConfiguration();
        op.removePropertyChangeListener(menu);
        infoTree.removePropertyChangeListener(menu);
        resultSetTable.removePropertyChangeListener(menu);
        // XXX LR[hsƐݒ̕ŃXe[^Xo[̕łȂ
        // statusBar.removePropertyChangeListener(menu);
        // [Cxg̐ݒ]
        op.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosing(WindowEvent e) {
                requestClose();
            }

        });
        bindEvents(infoTree);
        bindEvents(resultSetTable);
        bindEvents(textArea);
        menu.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                performMenuEvent(Item.valueOf(e.getActionCommand()), (Component)e.getSource());
            }

        });
    }

    /* @see net.argius.stew.ui.Launcher#launch(net.argius.stew.Environment) */
    public void launch(Environment env) {
        this.env = env;
        op.setEnvironment(env);
        op.setVisible(true);
        op.output(new Prompt(env));
        textArea.requestFocus();
    }

    /**
     * c[yC̕\/\ݒ肷B
     * @param show \/\
     */
    void setInfoTreePane(boolean show) {
        if (show) {
            split1.removeAll();
            split1.setTopComponent(new JScrollPane(infoTree));
            split1.setBottomComponent(split2);
            panel1.removeAll();
            panel1.add(split1, BorderLayout.CENTER);
            infoTree.setEnabled(true);
            if (env != null) {
                try {
                    infoTree.refreshRoot(env);
                } catch (SQLException ex) {
                    log.error("", ex);
                    op.showErrorDialog(ex);
                }
            }
        } else {
            infoTree.clear();
            infoTree.setEnabled(false);
            panel1.removeAll();
            panel1.add(split2, BorderLayout.CENTER);
        }
        SwingUtilities.updateComponentTreeUI(op);
    }

    /**
     * ݒ̕B
     */
    private void loadConfiguration() {
        Configuration cnf = Configuration.load(LocalSystem.getDirectory());
        op.setSize(cnf.getSize());
        op.setLocation(cnf.getLocation());
        split2.setDividerLocation(cnf.getDividerLocation());
        statusBar.setVisible(cnf.isShowStatusBar());
        resultSetTable.setShowColumnNumber(cnf.isShowTableColumnNumber());
        split1.setDividerLocation(cnf.getDividerLocation0());
        op.setAlwaysOnTop(cnf.isAlwaysOnTop());
        op.setAutoAdjustMode(Item.valueOf(cnf.getAutoAdjustMode()));
        setInfoTreePane(cnf.isShowInfoTree());
        changeFont("monospaced", Font.PLAIN, 1.0d);
    }

    /**
     * ݒ̕ۑB
     */
    private void saveConfiguration() {
        Configuration cnf = Configuration.load(LocalSystem.getDirectory());
        if ((op.getExtendedState() & Frame.MAXIMIZED_BOTH) == 0) {
            // ő剻łȂꍇ̂ݕۑ
            cnf.setSize(op.getSize());
            cnf.setLocation(op.getLocation());
            cnf.setDividerLocation(split2.getDividerLocation());
            cnf.setDividerLocation0(split1.getDividerLocation());
        }
        cnf.setShowStatusBar(statusBar.isVisible());
        cnf.setShowTableColumnNumber(resultSetTable.isShowColumnNumber());
        cnf.setShowInfoTree(infoTree.isEnabled());
        cnf.setAlwaysOnTop(op.isAlwaysOnTop());
        cnf.setAutoAdjustMode(op.getAutoAdjustMode().name());
        cnf.save();
    }

    /**
     * ԕۑ@\(Bean)B
     */
    @SuppressWarnings("all")
    public static final class Configuration {

        private static final Logger log = LoggerFactory.getLogger(Configuration.class);

        private transient File directory;

        private Dimension size;
        private Point location;
        private int dividerLocation;
        private int dividerLocation0;
        private boolean showStatusBar;
        private boolean showTableColumnNumber;
        private boolean showInfoTree;
        private boolean alwaysOnTop;
        private String autoAdjustMode;

        public Configuration() {
            this.size = new Dimension(640, 480);
            this.location = new Point(200, 200);
            this.dividerLocation = -1;
            this.dividerLocation0 = -1;
            this.showStatusBar = false;
            this.showTableColumnNumber = false;
            this.showInfoTree = false;
            this.alwaysOnTop = false;
            this.autoAdjustMode = Item.AUTO_ADJUST_MODE_NONE.name();
        }

        void setDirectory(File directory) {
            this.directory = directory;
        }

        void save() {
            final File file = getFile(directory);
            if (log.isDebugEnabled()) {
                log.debug(String.format("save Configuration to: [%s]", file));
            }
            try {
                XMLEncoder encoder = new XMLEncoder(new FileOutputStream(file));
                try {
                    encoder.writeObject(this);
                } finally {
                    encoder.close();
                }
            } catch (Exception ex) {
                log.warn("", ex);
            }
        }

        static Configuration load(File directory) {
            final File file = getFile(directory);
            if (log.isDebugEnabled()) {
                log.debug(String.format("load Configuration from: [%s]", file));
            }
            if (file.exists()) {
                try {
                    XMLDecoder decoder = new XMLDecoder(new FileInputStream(file));
                    try {
                        final Configuration instance = (Configuration)decoder.readObject();
                        instance.setDirectory(directory);
                        return instance;
                    } finally {
                        decoder.close();
                    }
                } catch (Exception ex) {
                    log.warn("", ex);
                }
            }
            final Configuration instance = new Configuration();
            instance.setDirectory(directory);
            return instance;
        }

        private static File getFile(File systemDirectory) {
            final File file = new File(systemDirectory, Configuration.class.getName() + ".xml");
            return file.getAbsoluteFile();
        }

        public Dimension getSize() {
            return size;
        }

        public void setSize(Dimension size) {
            this.size = size;
        }

        public Point getLocation() {
            return location;
        }

        public void setLocation(Point location) {
            this.location = location;
        }

        public int getDividerLocation() {
            return dividerLocation;
        }

        public void setDividerLocation(int dividerLocation) {
            this.dividerLocation = dividerLocation;
        }

        public int getDividerLocation0() {
            return dividerLocation0;
        }

        public void setDividerLocation0(int dividerLocation0) {
            this.dividerLocation0 = dividerLocation0;
        }

        public boolean isShowStatusBar() {
            return showStatusBar;
        }

        public void setShowStatusBar(boolean showStatusBar) {
            this.showStatusBar = showStatusBar;
        }

        public boolean isShowTableColumnNumber() {
            return showTableColumnNumber;
        }

        public void setShowTableColumnNumber(boolean showTableColumnNumber) {
            this.showTableColumnNumber = showTableColumnNumber;
        }

        public boolean isShowInfoTree() {
            return showInfoTree;
        }

        public void setShowInfoTree(boolean showInfoTree) {
            this.showInfoTree = showInfoTree;
        }

        public boolean isAlwaysOnTop() {
            return alwaysOnTop;
        }

        public void setAlwaysOnTop(boolean alwaysOnTop) {
            this.alwaysOnTop = alwaysOnTop;
        }

        public String getAutoAdjustMode() {
            return autoAdjustMode;
        }

        public void setAutoAdjustMode(String autoAdjustMode) {
            this.autoAdjustMode = autoAdjustMode;
        }

    }

    private void changeFont(String family, int style, double sizeRate) {
        FontControlLookAndFeel.change(family, style, sizeRate);
        SwingUtilities.updateComponentTreeUI(op);
        Font newfont = textArea.getFont();
        if (newfont != null) {
            statusBar.setFont(newfont.deriveFont(newfont.getSize() * 0.8f));
        }
    }

    /**
     * DatabaseInfoTreeɃCxg蓖ĂB
     */
    private void bindEvents(final DatabaseInfoTree tree) {
        final ActionUtility actionUtility = ActionUtility.getInstance(infoTree);
        final String keyCopyFullName = "copy-full-name";
        final String keyRefresh = "refresh";
        final String keyGenerateWherePhrase = "generate-where-phrase";
        final String keyGenerateSelectPhrase = "generate-select-phrase";
        final String keyGenerateUpdateStatement = "generate-update-statement";
        final String keyGenerateInsertStatement = "generate-insert-statement";
        final ConsoleTextArea textArea = this.textArea;
        // [ANVCxg]
        tree.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                textArea.replaceSelection(e.getActionCommand());
                textArea.requestFocus();
            }

        });
        // [ReLXgj[]
        actionUtility.setContextMenu(new String[]{keyCopyFullName, keyRefresh,
                                                  keyGenerateWherePhrase, keyGenerateSelectPhrase,
                                                  keyGenerateUpdateStatement,
                                                  keyGenerateInsertStatement},
                                     "FRWSUI".toCharArray());
    }

    /**
     * ResultSetTableɃCxg蓖ĂB
     * @param rstable
     */
    private void bindEvents(final ResultSetTable rstable) {
        // [ANVCxg]
        final ActionUtility actionUtility = ActionUtility.getInstance(rstable);
        final WindowOutputProcessor op = this.op;
        final int shortcutKey = Resource.getMenuShortcutKeyMask();
        final String keyCopyWithEscape = "copyWithEscape";
        final String keyPaste = "paste";
        final String keyCopyColumnNames = "copyColumnNames";
        final String keyClearCells = "clearCells";
        final String keyAddEmptyRow = "addEmptyRow";
        final String keyInsertFromClipboard = "insertFromClipboard";
        final String keyDuplicateRows = "duplicateRows";
        final String keyLinkRowsToDatabase = "linkRowsToDatabase";
        final String keyDeleteRows = "deleteRows";
        final Action cancelTableEditAction = new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                rstable.editingCanceled(new ChangeEvent(e.getSource()));
            }

        };
        // GXP[vtŃRs[
        actionUtility.bindAction(new AbstractAction(keyCopyWithEscape) {

            public void actionPerformed(ActionEvent e) {
                List<String> rows = new ArrayList<String>();
                for (int rowIndex : rstable.getSelectedRows()) {
                    List<Object> row = new ArrayList<Object>();
                    for (int columnIndex : rstable.getSelectedColumns()) {
                        final Object o = rstable.getValueAt(rowIndex, columnIndex);
                        row.add(CsvFormatter.AUTO.format(o == null ? "" : String.valueOf(o)));
                    }
                    rows.add(join(row, TAB));
                }
                setClipboard(join(rows, EOL));
            }

        }, getKeyStroke(VK_C, shortcutKey | ALT_DOWN_MASK));
        // \t
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                try {
                    InputStream is = new ByteArrayInputStream(getClipboard().getBytes());
                    Importer importer = new SmartImporter(is, TAB);
                    try {
                        int[] selectedColumns = rstable.getSelectedColumns();
                        for (int rowIndex : rstable.getSelectedRows()) {
                            Object[] values = importer.nextRow();
                            final int limit = Math.min(selectedColumns.length, values.length);
                            for (int x = 0; x < limit; x++) {
                                rstable.setValueAt(values[x], rowIndex, selectedColumns[x]);
                            }
                        }
                    } finally {
                        importer.close();
                    }
                    rstable.repaint();
                } catch (Exception ex) {
                    handleError(ex);
                } finally {
                    cancelTableEditAction.actionPerformed(e);
                }
            }

        }, keyPaste);
        // 񖼂Rs[
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                List<String> a = new ArrayList<String>();
                ResultSetTableModel m = rstable.getResultSetTableModel();
                for (int i = 0, n = m.getColumnCount(); i < n; i++) {
                    a.add(m.getColumnName(i));
                }
                setClipboard(join(a, TAB));
            }

        }, keyCopyColumnNames);
        // IZ̒l폜(NULLݒ)
        actionUtility.bindAction(new AbstractAction(keyClearCells) {

            public void actionPerformed(ActionEvent e) {
                try {
                    int[] selectedColumns = rstable.getSelectedColumns();
                    for (int rowIndex : rstable.getSelectedRows()) {
                        final int limit = selectedColumns.length;
                        for (int x = 0; x < limit; x++) {
                            rstable.setValueAt(null, rowIndex, selectedColumns[x]);
                        }
                    }
                    rstable.repaint();
                } catch (Exception ex) {
                    handleError(ex);
                } finally {
                    cancelTableEditAction.actionPerformed(e);
                }
            }

        }, getKeyStroke(VK_DELETE, 0));
        // Vs(񃊃N)ǉ
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                final int selectedRow = rstable.getSelectedRow();
                if (selectedRow >= 0) {
                    m.insertUnlinkedRow(selectedRow + 1, new Object[m.getColumnCount()]);
                } else {
                    m.addUnlinkedRow(new Object[m.getColumnCount()]);
                }
            }

        }, keyAddEmptyRow);
        // Nbv{[h}
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                try {
                    Importer importer = new SmartImporter(getClipboardAsReader(), TAB);
                    try {
                        ResultSetTableModel m = rstable.getResultSetTableModel();
                        while (true) {
                            Object[] row = importer.nextRow();
                            if (row.length == 0) {
                                break;
                            }
                            m.addUnlinkedRow(row);
                            m.linkRow(m.getRowCount() - 1);
                        }
                        rstable.repaintRowHeader("model");
                    } finally {
                        importer.close();
                    }
                } catch (Exception ex) {
                    handleError(ex);
                } finally {
                    cancelTableEditAction.actionPerformed(e);
                }
            }

        }, keyInsertFromClipboard);
        // Is𕡐Ēǉ
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                List<?> rows = m.getDataVector();
                int[] selectedRows = rstable.getSelectedRows();
                int index = selectedRows[selectedRows.length - 1];
                for (int rowIndex : selectedRows) {
                    m.insertUnlinkedRow(++index, (Vector<?>)((Vector<?>)rows.get(rowIndex)).clone());
                }
                rstable.repaint();
                rstable.repaintRowHeader("model");
            }

        }, keyDuplicateRows);
        // 񃊃NsDBƃN
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                try {
                    for (int rowIndex : rstable.getSelectedRows()) {
                        m.linkRow(rowIndex);
                    }
                } catch (SQLException ex) {
                    throw new RuntimeException(ex);
                } finally {
                    rstable.repaintRowHeader("unlinkedRowStatus");
                }
            }

        }, keyLinkRowsToDatabase);
        // Is폜
        actionUtility.bindAction(new AbstractAction(keyDeleteRows) {

            public void actionPerformed(ActionEvent e) {
                try {
                    ResultSetTableModel m = rstable.getResultSetTableModel();
                    while (true) {
                        final int selectedRow = rstable.getSelectedRow();
                        if (selectedRow < 0) {
                            break;
                        }
                        if (m.isLinkedRow(selectedRow)) {
                            final boolean removed = m.removeLinkedRow(selectedRow);
                            assert removed;
                        } else {
                            m.removeRow(selectedRow);
                        }
                    }
                } catch (SQLException ex) {
                    throw new RuntimeException(ex);
                } finally {
                    rstable.repaintRowHeader("model");
                }
            }

        }, getKeyStroke(VK_MINUS, shortcutKey | SHIFT_DOWN_MASK));
        // בւ
        /*
         * Java5ł́AꂪƃL[쎞ɃtH[JXOB
         * Java6ł͖ĂǂB
         */
        actionUtility.bindAction(new AbstractAction("sort") {

            public void actionPerformed(ActionEvent e) {
                op.sortTable(rstable.getSelectedColumn());
            }

        }, getKeyStroke(VK_S, ALT_DOWN_MASK));
        // 񕝂̎
        actionUtility.bindAction(new AbstractAction("adjustColumnWidth") {

            public void actionPerformed(ActionEvent e) {
                op.adjustTableColumnWidth();
            }

        }, getKeyStroke(VK_SLASH, shortcutKey));
        // [ReLXgj[]
        String[] actionNames = new String[]{"copy", keyCopyWithEscape, keyPaste,
                                            keyCopyColumnNames, keyClearCells, "selectAll", "",
                                            keyAddEmptyRow, keyInsertFromClipboard,
                                            keyDuplicateRows, keyLinkRowsToDatabase, keyDeleteRows,};
        final List<String> linkActions = Arrays.asList(new String[]{keyPaste, keyClearCells,
                                                                    keyAddEmptyRow,
                                                                    keyDuplicateRows,
                                                                    keyLinkRowsToDatabase,
                                                                    keyDeleteRows,});
        final Map<String, JMenuItem> m = new LinkedHashMap<String, JMenuItem>();
        final JPopupMenu pmenu = actionUtility.createPopupMenu(actionNames,
                                                               "CWPNRA EISLD".toCharArray(),
                                                               m);
        pmenu.addPopupMenuListener(new PopupMenuListener() {

            public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
                ResultSetTableModel resultSetTableModel = rstable.getResultSetTableModel();
                final boolean linkable = resultSetTableModel.isLinkable();
                for (Entry<String, JMenuItem> entry : m.entrySet()) {
                    if (linkActions.contains(entry.getKey())) {
                        entry.getValue().setEnabled(linkable);
                    }
                }
                final boolean updatable = resultSetTableModel.isUpdatable();
                m.get(keyInsertFromClipboard).setEnabled(updatable);
            }

            public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
                // empty
            }

            public void popupMenuCanceled(PopupMenuEvent e) {
                // empty
            }

        });
        rstable.setComponentPopupMenu(pmenu);
        // [wb_]
        // ReLXgj[
        final JTableHeader columnHeader = rstable.getTableHeader();
        final JPopupMenu pmenuCH = actionUtility.createPopupMenu(new String[]{"copy",
                                                                              keyCopyWithEscape,
                                                                              keyCopyColumnNames,
                                                                              keyAddEmptyRow,
                                                                              keyInsertFromClipboard},
                                                                 "CWNEI".toCharArray());
        final Point mousePosition = new Point();
        final JMenuItem sortMenu = new JMenuItem(new AbstractAction(getString("Action.sort", 'S')) {

            public void actionPerformed(ActionEvent e) {
                op.sortTable(rstable.getTableHeader().columnAtPoint(mousePosition));
            }

        });
        columnHeader.addMouseListener(new MouseAdapter() {

            @Override
            public void mousePressed(MouseEvent e) {
                mousePosition.setLocation(e.getPoint());
            }

        });
        sortMenu.setMnemonic('S');
        pmenuCH.insert(sortMenu, 0);
        columnHeader.setComponentPopupMenu(pmenuCH);
        // [swb_]
        // ReLXgj[
        final JTable rowHeader = rstable.getRowHeader();
        rowHeader.setComponentPopupMenu(pmenu);
    }

    /**
     * ConsoleTextAreaɃCxg蓖ĂB
     * @param textArea
     */
    private void bindEvents(final ConsoleTextArea textArea) {
        // [ANVCxg]
        final ActionUtility actionUtility = ActionUtility.getInstance(textArea);
        textArea.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                executeCommand(e.getActionCommand());
            }

        });
        // [hbvCxg]
        textArea.setDropTarget(new DropTarget(textArea, new DropTargetAdapter() {

            public void drop(DropTargetDropEvent dtde) {
                final List<File> fileList;
                Transferable t = dtde.getTransferable();
                if (t.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
                    dtde.acceptDrop(DnDConstants.ACTION_COPY_OR_MOVE);
                    try {
                        @SuppressWarnings("unchecked")
                        List<File> list = (List<File>)t.getTransferData(DataFlavor.javaFileListFlavor);
                        fileList = list;
                    } catch (UnsupportedFlavorException ex) {
                        throw new RuntimeException(ex);
                    } catch (IOException ex) {
                        throw new RuntimeException(ex);
                    }
                } else {
                    throw new IllegalStateException("Transferable=" + t);
                }
                JPopupMenu menu = new JPopupMenu();
                JMenuItem item1 = new JMenuItem(new AbstractAction(getString("Action.paste-path",
                                                                             'P')) {

                    public void actionPerformed(ActionEvent e) {
                        final String s = join(fileList, " ");
                        final int p = textArea.getCaretPosition();
                        if (textArea.isEditablePosition(p)) {
                            textArea.insert(s, p);
                        } else {
                            textArea.append(s, false);
                        }
                    }

                });
                item1.setMnemonic('P');
                menu.add(item1);
                JMenuItem item2 = new JMenuItem(new AbstractAction(getString("Action.paste-file-content",
                                                                             'F')) {

                    public void actionPerformed(ActionEvent e) {
                        try {
                            for (File file : fileList) {
                                if (file.length() > 0) {
                                    final String s = convertContentToString(file);
                                    final int p = textArea.getCaretPosition();
                                    if (textArea.isEditablePosition(p)) {
                                        textArea.insert(s, p);
                                    } else {
                                        textArea.append(s, false);
                                    }
                                }
                            }
                        } catch (IOException ex) {
                            throw new RuntimeException(ex);
                        }
                    }

                    private String convertContentToString(File file) throws IOException {
                        ByteArrayOutputStream bos = new ByteArrayOutputStream();
                        FileInputStream fis = new FileInputStream(file);
                        try {
                            fis.getChannel().transferTo(0, file.length(), Channels.newChannel(bos));
                        } finally {
                            fis.close();
                        }
                        return bos.toString();
                    }

                });
                item2.setMnemonic('F');
                menu.add(item2);
                Point p = dtde.getLocation();
                menu.show(textArea, p.x, p.y);
            }

        }));
        // [̑]
        // ReLXgj[+AhD
        actionUtility.setActionForTextComponent();
    }

    /**
     * ResultSetTablepTextSearch̐B
     * @param table
     * @return TextSearch
     */
    private static TextSearch createTextSearch(final ResultSetTable table) {
        return new TextSearch() {

            public boolean search(Matcher matcher) {
                final int rowCount = table.getRowCount();
                if (rowCount <= 0) {
                    return false;
                }
                final int columnCount = table.getColumnCount();
                final boolean backward = matcher.isBackward();
                final int amount = backward ? -1 : 1;
                final int rowStart = backward ? rowCount - 1 : 0;
                final int rowEnd = backward ? 0 : rowCount - 1;
                final int columnStart = backward ? columnCount - 1 : 0;
                final int columnEnd = backward ? 0 : columnCount - 1;
                int row = rowStart;
                int column = columnStart;
                if (table.getSelectedColumnCount() > 0) {
                    column = table.getSelectedColumn();
                    row = table.getSelectedRow() + amount;
                    if (backward) {
                        if (row < 0) {
                            --column;
                            if (column < 0) {
                                return false;
                            }
                            row = rowStart;
                        }
                    } else {
                        if (row >= rowCount) {
                            ++column;
                            if (column >= columnCount) {
                                return false;
                            }
                            row = rowStart;
                        }
                    }
                }
                final TableModel m = table.getModel();
                for (; backward ? column >= columnEnd : column <= columnEnd; column += amount) {
                    for (; backward ? row >= rowEnd : row <= rowEnd; row += amount) {
                        if (matcher.find(String.valueOf(m.getValueAt(row, column)))) {
                            table.changeSelection(row, column, false, false);
                            table.requestFocus();
                            return true;
                        }
                    }
                    row = rowStart;
                }
                return false;
            }

            public void reset() {
                // empty
            }

        };
    }

    /**
     * ConsoleTextAreapTextSearch̐B
     * @param textArea
     * @return TextSearch
     */
    private static TextSearch createTextSearch(final ConsoleTextArea textArea) {
        return new TextSearch() {

            public boolean search(Matcher matcher) {
                removeHighlights();
                try {
                    Highlighter highlighter = textArea.getHighlighter();
                    HighlightPainter painter = matcher.getHighlightPainter();
                    String text = textArea.getText();
                    int start = 0;
                    boolean matched = false;
                    while (matcher.find(text, start)) {
                        matched = true;
                        int matchedIndex = matcher.getStart();
                        highlighter.addHighlight(matchedIndex, matcher.getEnd(), painter);
                        start = matchedIndex + 1;
                    }
                    textArea.addKeyListener(new KeyAdapter() {

                        @Override
                        public void keyTyped(KeyEvent e) {
                            textArea.removeKeyListener(this);
                            removeHighlights();
                        }

                    });
                    return matched;
                } catch (BadLocationException ex) {
                    throw new RuntimeException(ex);
                }
            }

            public void reset() {
                removeHighlights();
            }

            void removeHighlights() {
                for (Highlight highlight : textArea.getHighlighter().getHighlights()) {
                    textArea.getHighlighter().removeHighlight(highlight);
                }
            }

        };
    }

    /**
     * Nbv{[h̓e𕶎Xg[ƂĎ擾B
     * @return Xg[
     * @throws IOException
     */
    static Reader getClipboardAsReader() throws IOException {
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        try {
            return DataFlavor.stringFlavor.getReaderForText(clipboard.getContents(null));
        } catch (UnsupportedFlavorException ex) {
            if (log.isDebugEnabled()) {
                log.debug("", ex);
            }
            throw new IOException(ex.toString());
        }
    }

    /**
     * Nbv{[h當擾B
     * @return 
     * @throws IOException 
     */
    static String getClipboard() throws IOException {
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        final Object o;
        try {
            o = clipboard.getData(DataFlavor.stringFlavor);
        } catch (UnsupportedFlavorException ex) {
            if (log.isDebugEnabled()) {
                log.debug("", ex);
            }
            throw new IOException(ex.toString());
        }
        if (log.isTraceEnabled()) {
            log.trace(String.format("get clipboard:[%s]", o));
        }
        return (String)o;
    }

    /**
     * Nbv{[hɕݒ肷B
     * @param s 
     */
    static void setClipboard(String s) {
        if (log.isTraceEnabled()) {
            log.trace(String.format("set clipboard:[%s]", s));
        }
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        StringSelection sselection = new StringSelection(s);
        clipboard.setContents(sselection, sselection);
    }

    void handleError(Throwable th) {
        log.error("", th);
        op.showErrorDialog(th);
    }

    /**
     * j[Cxg̏B
     * @param item j[
     * @param source MR|[lg
     */
    void performMenuEvent(Item item, Component source) {
        try {
            resultSetTable.editingCanceled(new ChangeEvent(source));
        } catch (Exception ex) {
            handleError(ex);
        }
        switch (item) {
            case NEW:
                invoke();
                break;
            case CLOSE:
                requestClose();
                break;
            case QUIT:
                requestExit();
                break;
            case CUT:
            case COPY:
            case PASTE:
            case SELECT_ALL: {
                final KeyStroke accelerator = ((JMenuItem)source).getAccelerator();
                ActionListener action = focused.getActionForKeyStroke(accelerator);
                if (action != null) {
                    action.actionPerformed(new ActionEvent(focused, 1001, ""));
                }
            }
                break;
            case FIND:
                textSearchPanel.setCurrentTarget(textSearchMap.get(focused));
                textSearchPanel.setVisible(true);
                break;
            case TOGGLE_FOCUS:
                if (textArea.isFocusOwner()) {
                    resultSetTable.requestFocus();
                } else {
                    textArea.requestFocus();
                }
                break;
            case CLEAR_MESSAGE:
                textArea.clear();
                executeCommand(EMPTY_STRING);
                break;
            case STATUS_BAR: {
                statusBar.setVisible(((JCheckBoxMenuItem)source).isSelected());
            }
                break;
            case COLUMN_NUMBER:
                resultSetTable.setShowColumnNumber(((JCheckBoxMenuItem)source).isSelected());
                split2.getTopComponent().repaint();
                break;
            case INFO_TREE:
                setInfoTreePane(((JCheckBoxMenuItem)source).isSelected());
                op.validate();
                op.repaint();
                break;
            case ALWAYS_ON_TOP:
                op.setAlwaysOnTop(((JCheckBoxMenuItem)source).isSelected());
                break;
            case WIDEN_COLUMN_WIDTH:
                op.changeTableColumnWidth(1.5f);
                break;
            case NARROW_COLUMN_WIDTH:
                op.changeTableColumnWidth(1 / 1.5f);
                break;
            case ADJUST_COLUMN_WIDTH:
                op.adjustTableColumnWidth();
                break;
            case AUTO_ADJUST_MODE_NONE:
            case AUTO_ADJUST_MODE_HEADER:
            case AUTO_ADJUST_MODE_VALUE:
            case AUTO_ADJUST_MODE_HEADERANDVALUE:
                op.setAutoAdjustMode(item);
                break;
            case REFRESH: {
                if (resultSetTable.getModel() instanceof ResultSetTableModel) {
                    ResultSetTableModel m = resultSetTable.getResultSetTableModel();
                    if (m.isSameConnection(env.getCurrentConnection())) {
                        final String s = m.getCommandString();
                        if (s != null && s.length() > 0) {
                            executeCommand(s);
                        }
                    }
                }
            }
                break;
            case EXECUTE:
                executeCommand(textArea.getEditableText());
                break;
            case BREAK:
                env.getOutputProcessor().close();
                env.setOutputProcessor(new WindowOutputProcessor.Bypass(op));
                op.output(getString("i.cancelled"));
                doPostProcess();
                break;
            case HISTORY_BACK:
                retrieveHistory(-1);
                break;
            case HISTORY_NEXT:
                retrieveHistory(+1);
                break;
            case ROLLBACK:
                if (confirmCommitable()
                    && showConfirmDialog(op, getString("i.confirm-rollback"), null, OK_CANCEL_OPTION) == OK_OPTION) {
                    executeCommand("rollback");
                }
                break;
            case COMMIT:
                if (confirmCommitable()
                    && showConfirmDialog(op, getString("i.confirm-commit"), null, OK_CANCEL_OPTION) == OK_OPTION) {
                    executeCommand("commit");
                }
                break;
            case CONNECT: {
                if (env.getConnectorMap().isEmpty()) {
                    showMessageDialog(op, getString("w.no-connector"));
                    return;
                }
                Object[] a = ConnectorEntry.toList(env.getConnectorMap().values()).toArray();
                final String m = getString("i.choose-connection");
                Object value = showInputDialog(op, m, null, PLAIN_MESSAGE, null, a, a[0]);
                if (value != null) {
                    ConnectorEntry c = (ConnectorEntry)value;
                    executeCommand("connect " + c.getId());
                }
            }
                break;
            case DISCONNECT:
                executeCommand("disconnect");
                break;
            case ENCRYPTION_KEY: {
                JPasswordField password = new JPasswordField(20);
                Object[] a = {getString("i.input-encryption-key"), password};
                if (showConfirmDialog(op, a, null, OK_CANCEL_OPTION) == OK_OPTION) {
                    CipherPassword.setSecretKey(String.valueOf(password.getPassword()));
                }
            }
                break;
            case EDIT_CONNECTORS: {
                env.updateConnectorMap();
                if (env.getCurrentConnector() != null) {
                    showMessageDialog(op, getString("i.reconnect-after-edited-current-connector"));
                }
                ConnectorMapEditDialog dialog = new ConnectorMapEditDialog(op, env);
                dialog.pack();
                dialog.setModal(true);
                dialog.setLocationRelativeTo(op);
                dialog.setVisible(true);
                env.updateConnectorMap();
            }
                break;
            case SORT:
                op.sortTable(resultSetTable.getSelectedColumn());
                break;
            case EXPORT:
                try {
                    op.exportTableContent();
                } catch (IOException ex) {
                    throw new RuntimeException(ex);
                }
                break;
            case HELP:
                showMessageDialog(op, getString("i.help-see-release-package"), null, PLAIN_MESSAGE);
                break;
            case ABOUT:
                ImageIcon icon = new ImageIcon();
                if (op.getIconImage() != null) {
                    icon.setImage(op.getIconImage());
                }
                showMessageDialog(op, getString(".about", getVersion()), null, PLAIN_MESSAGE, icon);
                break;
            default:
                log.warn(String.format("ignored [%s] (%s)", item, source));
        }
    }

    /**
     * NB 
     */
    static void invoke() {
        WindowLauncher instance = new WindowLauncher();
        final Environment env = new Environment();
        env.setOutputProcessor(new WindowOutputProcessor.Bypass(instance.op));
        instance.launch(env);
        // JX^L[oCh̔f
        try {
            File keyBindConf = new File(LocalSystem.getDirectory(), "keybind.conf");
            if (keyBindConf.exists()) {
                List<String> a = new ArrayList<String>();
                Scanner scanner = new Scanner(keyBindConf);
                try {
                    while (scanner.hasNextLine()) {
                        a.add(scanner.nextLine());
                    }
                } finally {
                    scanner.close();
                }
                changeKeyBinds(a);
            }
        } catch (Exception ex) {
            instance.handleError(ex);
        }
    }

    /**
     * L[oChύXB 
     * @param keyBindInfos L[oCh(Item=KeyStroke)
     */
    static void changeKeyBinds(List<String> keyBindInfos) {
        final Pattern p = Pattern.compile("\\s*([^=\\s]+)\\s*=(.*)");
        List<String> errorInfos = new ArrayList<String>();
        for (String s : keyBindInfos) {
            if (s.matches("\\s*#.*")) {
                continue;
            }
            Matcher m = p.matcher(s);
            if (m.matches()) {
                try {
                    Item item = Item.valueOf(m.group(1));
                    KeyStroke keyStroke = getKeyStroke(m.group(2));
                    for (WindowLauncher instance : instances) {
                        instance.menu.setAccelerator(item, keyStroke);
                    }
                } catch (Exception ex) {
                    errorInfos.add(s);
                }
            } else {
                errorInfos.add(s);
            }
        }
        if (!errorInfos.isEmpty()) {
            throw new IllegalArgumentException(errorInfos.toString());
        }
    }

    /**
     * IB
     */
    static void exit() {
        for (WindowLauncher instance : new ArrayList<WindowLauncher>(instances)) {
            try {
                instance.close();
            } catch (Exception ex) {
                log.warn("error occurred when closing all instances", ex);
            }
        }
    }

    /**
     * LauncherB
     */
    void close() {
        instances.remove(this);
        try {
            env.release();
            saveConfiguration();
            executorService.shutdown();
        } finally {
            op.dispose();
        }
    }

    /**
     * YES-NOIvV̊mF_CAO\YESꂽǂmFB
     * @param message bZ[W
     * @return YESꂽǂ
     */
    private boolean confirmYes(String message) {
        return showConfirmDialog(op, message, "", YES_NO_OPTION) == YES_OPTION;
    }

    /**
     * R~bg\ȏԂǂmFB
     * @return R~bg\ǂ
     */
    private boolean confirmCommitable() {
        if (env.getCurrentConnection() == null) {
            showMessageDialog(op, getString("w.not-connect"), null, OK_OPTION);
            return false;
        }
        if (env.getCurrentConnector().isReadOnly()) {
            showMessageDialog(op, getString("w.connector-readonly"), null, OK_OPTION);
            return false;
        }
        return true;
    }

    /**
     * oB
     * @param destination (+/-)
     */
    private void retrieveHistory(int destination) {
        if (historyList.isEmpty()) {
            return;
        }
        historyIndex += destination;
        if (historyIndex >= historyList.size()) {
            historyIndex = 0;
        } else if (historyIndex < 0) {
            historyIndex = historyList.size() - 1;
        }
        textArea.replace(historyList.get(historyIndex));
        final int endPosition = textArea.getEndPosition();
        textArea.setSelectionStart(endPosition);
        textArea.moveCaretPosition(endPosition);
        textArea.requestFocus();
    }

    /**
     * vB
     */
    void requestClose() {
        if (instances.size() == 1) {
            requestExit();
        } else if (env.getCurrentConnection() == null || confirmYes(getString("i.confirm-close"))) {
            close();
        }
    }

    /**
     * IvB
     */
    private void requestExit() {
        if (confirmYes(getString("i.confirm-quit"))) {
            exit();
        }
    }

    /**
     * R}hsB
     * @param commandString R}h
     */
    void executeCommand(String commandString) {
        assert commandString != null;
        textArea.replace("");
        op.output(commandString);
        if (commandString.trim().length() == 0) {
            doPostProcess();
        } else {
            final String cmd = commandString;
            final Environment env = this.env;
            final DatabaseInfoTree infoTree = this.infoTree;
            final JLabel statusBar = this.statusBar;
            final OutputProcessor opref = env.getOutputProcessor();
            try {
                doPreProcess();
                executorService.execute(new Runnable() {

                    public void run() {
                        Connector c = env.getCurrentConnector();
                        long time = System.currentTimeMillis();
                        if (!Command.invoke(env, cmd)) {
                            exit();
                        }
                        if (infoTree.isEnabled()) {
                            try {
                                if (env.getCurrentConnector() != c) {
                                    infoTree.clear();
                                    if (env.getCurrentConnector() != null) {
                                        infoTree.refreshRoot(env);
                                    }
                                }
                            } catch (Throwable th) {
                                handleError(th);
                            }
                        }
                        if (env.getOutputProcessor() == opref) {
                            time = System.currentTimeMillis() - time;
                            statusBar.setText(getString("i.statusbar-message", time / 1000f, cmd));
                            doPostProcess();
                        }
                    }

                });
            } catch (Exception ex) {
                throw new RuntimeException(ex);
            } finally {
                historyIndex = historyList.size();
            }
            if (historyList.contains(commandString)) {
                historyList.remove(commandString);
            }
            historyList.add(commandString);
        }
        historyIndex = historyList.size();
    }

    void doPreProcess() {
        ((Menu)op.getJMenuBar()).setEnabledStates(true);
        resultSetTable.setEnabled(false);
        textArea.setEnabled(false);
        op.repaint();
    }

    void doPostProcess() {
        ((Menu)op.getJMenuBar()).setEnabledStates(false);
        resultSetTable.setEnabled(true);
        textArea.setEnabled(true);
        op.output(new Prompt(env));
        textArea.requestFocus();
    }

    static void wakeup() {
        for (WindowLauncher instance : new ArrayList<WindowLauncher>(instances)) {
            try {
                SwingUtilities.updateComponentTreeUI(instance.op);
            } catch (Exception ex) {
                log.warn("", ex);
            }
        }
        if (log.isInfoEnabled()) {
            log.info("wake up");
        }
    }

    /**
     * Gg|CgB
     * @param args
     */
    public static void main(String... args) {
        final int residentCycle = LocalSystem.getPropertyAsInt("net.argius.stew.ui.window.resident");
        if (residentCycle > 0) {
            final long msec = residentCycle * 60000L;
            Timer timer = new Timer(true);
            timer.scheduleAtFixedRate(new TimerTask() {

                @Override
                public void run() {
                    EventQueue.invokeLater(new Runnable() {

                        public void run() {
                            wakeup();
                        }

                    });
                }

            }, msec, msec);
        }
        final Logger log = WindowLauncher.log;
        EventQueue.invokeLater(new Runnable() {

            public void run() {
                Thread.setDefaultUncaughtExceptionHandler(new UncaughtExceptionHandler() {

                    public void uncaughtException(Thread t, Throwable e) {
                        log.fatal(t, e);
                    }

                });
                invoke();
            }

        });
    }

}
