/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew;

import static java.io.File.pathSeparator;

import java.io.*;
import java.net.*;
import java.sql.*;
import java.util.*;
import java.util.zip.*;

import net.argius.logging.*;

/**
 * RlN^pDriverManagerB
 */
final class ConnectorDriverManager {

    private static final Logger log = LoggerFactory.getLogger(ConnectorDriverManager.class);

    static final Set<File> driverFiles = Collections.synchronizedSet(new LinkedHashSet<File>());
    static final Set<Driver> drivers = Collections.synchronizedSet(new LinkedHashSet<Driver>());

    private ConnectorDriverManager() {
        // empty
    }

    /**
     * hCo̎擾B
     * @param url JDBCURL
     * @param driverClassName hCoNX
     * @param classpath NXpX
     * @return hCo
     * @throws SQLException
     */
    static Driver getDriver(String url, String driverClassName, String classpath) throws SQLException {
        assert !isBlank(url);
        final boolean hasClasspath = !isBlank(classpath);
        if (!hasClasspath) {
            for (Driver driver : new ArrayList<Driver>(drivers)) {
                if (driver.acceptsURL(url)) {
                    return driver;
                }
            }
        }
        List<File> jars = new ArrayList<File>();
        ClassLoader cl;
        if (hasClasspath) {
            List<URL> urls = new ArrayList<URL>();
            for (String path : classpath.split(pathSeparator)) {
                final File file = new File(path);
                if (isJarFile(file)) {
                    jars.add(file);
                }
                try {
                    urls.add(file.toURL());
                } catch (MalformedURLException ex) {
                    log.warn(ex);
                }
            }
            cl = new URLClassLoader(urls.toArray(new URL[urls.size()]));
        } else {
            jars.addAll(getJarFiles("."));
            jars.addAll(driverFiles);
            List<URL> urls = new ArrayList<URL>();
            for (File file : jars) {
                try {
                    urls.add(file.toURL());
                } catch (MalformedURLException ex) {
                    log.warn(ex);
                }
            }
            cl = new URLClassLoader(urls.toArray(new URL[urls.size()]),
                                    ClassLoader.getSystemClassLoader());
        }
        driverFiles.addAll(jars);
        final boolean hasDriverClassName = !isBlank(driverClassName);
        if (hasDriverClassName) {
            Driver driver = DynamicLoader.newInstance(driverClassName, cl);
            assert driver != null;
            return driver;
        }
        final String jdbcDrivers = System.getProperty("jdbc.drivers");
        if (!isBlank(jdbcDrivers)) {
            for (String jdbcDriver : jdbcDrivers.split(":")) {
                try {
                    Driver driver = DynamicLoader.newInstance(jdbcDriver, cl);
                    if (driver != null) {
                        if (!hasClasspath) {
                            drivers.add(driver);
                        }
                        return driver;
                    }
                } catch (DynamicLoadingException ex) {
                    log.warn(ex);
                }
            }
        }
        for (File jar : jars) {
            try {
                Driver driver = getDriver(jar, url, cl);
                if (driver != null) {
                    if (!hasClasspath) {
                        drivers.add(driver);
                    }
                    return driver;
                }
            } catch (IOException ex) {
                log.warn(ex);
            }
        }
        for (String path : System.getProperty("java.class.path", "").split(pathSeparator)) {
            if (isJarFile(path)) {
                Driver driver;
                try {
                    driver = getDriver(new File(path), url, cl);
                    if (driver != null) {
                        drivers.add(driver);
                        return driver;
                    }
                } catch (IOException ex) {
                    log.warn(ex);
                }
            }
        }
        throw new SQLException("driver not found");
    }

    private static Driver getDriver(File jar, String url, ClassLoader cl) throws IOException {
        ZipFile zipFile = new ZipFile(jar);
        try {
            for (ZipEntry entry : Iteration.asIterable(zipFile.entries())) {
                final String name = entry.getName();
                if (name.endsWith(".class")) {
                    final String fqcn = name.replaceFirst("\\.class", "").replace('/', '.');
                    try {
                        Class<?> c = DynamicLoader.loadClass(fqcn, cl);
                        if (Driver.class.isAssignableFrom(c)) {
                            Driver driver = (Driver)c.newInstance();
                            if (driver.acceptsURL(url)) {
                                return driver;
                            }
                        }
                    } catch (Exception ex) {
                        if (log.isTraceEnabled()) {
                            log.trace(ex);
                        }
                    }
                }
            }
        } finally {
            zipFile.close();
        }
        return null;
    }

    private static List<File> getJarFiles(String path) {
        File root = new File(path);
        File[] files = root.listFiles();
        if (files == null) {
            return Collections.emptyList();
        }
        List<File> jars = new ArrayList<File>();
        for (File file : files) {
            if (isJarFile(file)) {
                jars.add(file);
            }
        }
        return jars;
    }

    private static boolean isJarFile(File file) {
        return isJarFile(file.getPath());
    }

    private static boolean isJarFile(String path) {
        return path.matches("(?i).+\\.(jar|zip)");
    }

    private static boolean isBlank(String s) {
        return s == null || s.trim().length() == 0;
    }

}
