/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.EventQueue.invokeLater;
import static javax.swing.JOptionPane.*;
import static net.argius.stew.Iteration.asIterable;
import static net.argius.stew.ui.window.Resource.*;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.sql.*;
import java.util.*;
import java.util.List;

import javax.swing.*;
import javax.swing.table.*;

import net.argius.stew.*;
import net.argius.stew.io.*;
import net.argius.stew.ui.*;
import net.argius.stew.ui.window.Menu.*;

/**
 * EBhE[h̏o͐B
 */
final class WindowOutputProcessor extends JFrame implements OutputProcessor {

    private ResultSetTable resultSetTable;
    private ConsoleTextArea textArea;

    private Environment env;
    private File currentDirectory;
    private Menu.Item autoAdjustMode;
    private int lastSortedIndex;
    private boolean lastSortedIsReverse;

    WindowOutputProcessor(ResultSetTable resultSetTable, ConsoleTextArea textArea) {
        this.resultSetTable = resultSetTable;
        this.textArea = textArea;
        this.lastSortedIndex = -1;
        this.lastSortedIsReverse = false;
    }

    /* @see net.argius.stew.OutputProcessor#output(java.lang.Object) */
    public void output(final Object o) {
        try {
            if (o instanceof ResultSet) {
                outputResult((ResultSet)o, new ColumnOrder(), "");
                return;
            } else if (o instanceof ResultSetReference) {
                final ResultSetReference ref = (ResultSetReference)o;
                ResultSet rs = ref.getResultSet();
                ref.setRecordCount(outputResult(rs, ref.getOrder(), ref.getCommandString()));
                return;
            }
        } catch (SQLException ex) {
            throw new RuntimeException(ex);
        }
        final ConsoleTextArea textArea = this.textArea;
        final String message;
        if (o instanceof Prompt) {
            message = replaceEOL(o.toString());
        } else {
            message = replaceEOL(String.format("%s%n", o));
        }
        invokeLater(new Runnable() {

            public void run() {
                textArea.output(message);
            }

        });
    }

    /* @see net.argius.stew.OutputProcessor#close() */
    public void close() {
        dispose();
    }

    void setEnvironment(Environment env) {
        this.env = env;
    }

    /**
     * ʂo͂B
     * @param rs
     * @param order
     * @param commandString 
     * @return o͂ꂽ(ʌł͂Ȃ)
     * @throws SQLException
     */
    int outputResult(ResultSet rs, ColumnOrder order, String commandString) throws SQLException {
        /*
         * :̃\bh͔AWTXbhs邽
         *      GUȊinvokeLaterōsƁB
         */
        final OutputProcessor opref = env.getOutputProcessor();
        final JTable table = resultSetTable;
        final JTableHeader header = table.getTableHeader();
        invokeLater(new Runnable() {

            public void run() {
                table.setVisible(false);
                header.setVisible(false);
                ((DefaultTableModel)table.getModel()).setRowCount(0);
            }

        });
        final boolean needsOrderChange = order.size() > 0;
        ResultSetMetaData meta = rs.getMetaData();
        final int columnCount = (needsOrderChange) ? order.size() : meta.getColumnCount();
        final Vector<String> columnNames = new Vector<String>(columnCount);
        if (needsOrderChange) {
            for (int i = 0; i < order.size(); i++) {
                columnNames.add(order.getName(i));
            }
        } else {
            for (int index = 1; index <= columnCount; index++) {
                columnNames.add(meta.getColumnName(index));
            }
        }
        final ResultSetTableModel m = new ResultSetTableModel(rs,
                                                              columnNames.toArray(),
                                                              commandString);
        Vector<Object> v = new Vector<Object>(columnCount);
        ValueTransporter transfer = ValueTransporter.getInstance("");
        final int limit = LocalSystem.getPropertyAsInt(PropertyKey.ROW_COUNT_LIMIT,
                                                       Integer.MAX_VALUE);
        int rowCount = 0;
        while (rs.next()) {
            if (rowCount >= limit) {
                invokeLater(new Runnable() {

                    public void run() {
                        output(getString("w.exceeded-limit", limit));
                    }

                });
                break;
            }
            ++rowCount;
            v.clear();
            for (int i = 0; i < columnCount; i++) {
                final int index = needsOrderChange ? order.getOrder(i) : i + 1;
                v.add(transfer.getObject(rs, index));
            }
            m.addRow((Vector<?>)v.clone());
            if (env.getOutputProcessor() != opref) {
                throw new SQLException("interrupted");
            }
        }
        invokeLater(new Runnable() {

            public void run() {
                table.setModel(m);
                Container p = table.getParent();
                if (p != null && p.getParent() instanceof JScrollPane) {
                    JScrollPane scrollPane = (JScrollPane)p.getParent();
                    ImageIcon icon = getImageIcon(String.format("updatable-%s.png", m.isUpdatable()));
                    scrollPane.setCorner(ScrollPaneConstants.UPPER_LEFT_CORNER,
                                         new JLabel(icon, SwingConstants.CENTER));
                }
                adjustTableColumnWidth();
                header.setVisible(true);
                table.doLayout();
                table.setVisible(true);
            }

        });
        return m.getRowCount();
    }

    /**
     * autoAdjustMode̎擾B
     * @return autoAdjustMode
     */
    Menu.Item getAutoAdjustMode() {
        return autoAdjustMode;
    }

    /**
     * autoAdjustMode̐ݒB
     * @param autoAdjustMode autoAdjustMode
     */
    void setAutoAdjustMode(Menu.Item autoAdjustMode) {
        final Item oldValue = this.autoAdjustMode;
        this.autoAdjustMode = autoAdjustMode;
        firePropertyChange("autoAdjustMode", oldValue, autoAdjustMode);
    }

    /**
     * 񕝂ύXB
     * @param rate {
     */
    void changeTableColumnWidth(double rate) {
        for (TableColumn column : asIterable(resultSetTable.getColumnModel().getColumns())) {
            column.setPreferredWidth((int)(column.getWidth() * rate));
        }
    }

    /**
     * 񕝂̎sB
     * @param columnAdjustMode 񕝎[h
     */
    void adjustTableColumnWidth() {
        if (autoAdjustMode == null) {
            return;
        }
        final boolean byHeader;
        final boolean byValue;
        switch (autoAdjustMode) {
            case AUTO_ADJUST_MODE_HEADER:
                byHeader = true;
                byValue = false;
                break;
            case AUTO_ADJUST_MODE_VALUE:
                byHeader = false;
                byValue = true;
                break;
            case AUTO_ADJUST_MODE_HEADERANDVALUE:
                byHeader = true;
                byValue = true;
                break;
            case AUTO_ADJUST_MODE_NONE:
                byHeader = false;
                byValue = false;
                break;
            default:
                throw new IllegalStateException("autoAdjustMode=" + autoAdjustMode);
        }
        final int rowCount = resultSetTable.getRowCount();
        if (!byHeader && byValue && rowCount == 0) {
            return;
        }
        final float max = resultSetTable.getParent().getWidth() * 0.8f;
        TableColumnModel columnModel = resultSetTable.getColumnModel();
        JTableHeader header = resultSetTable.getTableHeader();
        for (int columnIndex = 0, n = resultSetTable.getColumnCount(); columnIndex < n; columnIndex++) {
            float size = 0f;
            if (byHeader) {
                // wb_̕ɂ钲
                TableColumn column = columnModel.getColumn(columnIndex);
                TableCellRenderer renderer = column.getHeaderRenderer();
                if (renderer == null) {
                    renderer = header.getDefaultRenderer();
                }
                if (renderer != null) {
                    Component c = renderer.getTableCellRendererComponent(resultSetTable,
                                                                         column.getHeaderValue(),
                                                                         false,
                                                                         false,
                                                                         0,
                                                                         columnIndex);
                    size = c.getPreferredSize().width * 1.5f;
                }
            }
            if (byValue) {
                // l̕ɂ钲
                for (int rowIndex = 0; rowIndex < rowCount; rowIndex++) {
                    TableCellRenderer renderer = resultSetTable.getCellRenderer(rowIndex,
                                                                                columnIndex);
                    if (renderer == null) {
                        continue;
                    }
                    Object value = resultSetTable.getValueAt(rowIndex, columnIndex);
                    Component c = renderer.getTableCellRendererComponent(resultSetTable,
                                                                         value,
                                                                         false,
                                                                         false,
                                                                         rowIndex,
                                                                         columnIndex);
                    size = Math.max(size, c.getPreferredSize().width);
                    if (size >= max) {
                        break;
                    }
                }
            }
            int width = Math.round(size > max ? max : size) + 1;
            columnModel.getColumn(columnIndex).setPreferredWidth(width);
        }
    }

    /**
     * e[u\[gB
     * @param index \[g
     */
    void sortTable(final int index) {
        final boolean reverse;
        if (lastSortedIndex == index) {
            reverse = (lastSortedIsReverse == false);
        } else {
            lastSortedIndex = index;
            reverse = false;
        }
        lastSortedIsReverse = reverse;
        resultSetTable.getResultSetTableModel().sort(index, reverse);
        resultSetTable.repaint();
        resultSetTable.repaintRowHeader("unlinkedRowStatus");
    }

    void exportTableContent() throws IOException {
        if (currentDirectory == null) {
            synchronized (this) {
                if (currentDirectory == null) {
                    currentDirectory = env.getCurrentDirectory();
                }
            }
        }
        JFileChooser fileChooser = new JFileChooser(currentDirectory);
        fileChooser.setDialogTitle(getString("Action.export"));
        fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
        fileChooser.showSaveDialog(resultSetTable);
        final File file = fileChooser.getSelectedFile();
        if (file == null) {
            return;
        }
        synchronized (this) {
            currentDirectory = file.getParentFile();
        }
        if (file.exists()) {
            if (showConfirmDialog(resultSetTable,
                                  getString("i.confirm-overwrite", file),
                                  EMPTY_STRING,
                                  YES_NO_OPTION) != YES_OPTION) {
                return;
            }
        }
        Exporter exporter = Exporter.getExporter(file);
        try {
            TableColumnModel columnModel = resultSetTable.getTableHeader().getColumnModel();
            List<Object> headerValues = new ArrayList<Object>();
            for (TableColumn column : asIterable(columnModel.getColumns())) {
                headerValues.add(column.getHeaderValue());
            }
            exporter.addHeader(headerValues.toArray());
            DefaultTableModel m = (DefaultTableModel)resultSetTable.getModel();
            @SuppressWarnings("unchecked")
            Vector<Vector<Object>> rows = m.getDataVector();
            for (Vector<Object> row : rows) {
                exporter.addRow(row.toArray());
            }
        } finally {
            exporter.close();
        }
        JOptionPane.showMessageDialog(resultSetTable, getString("i.exported"));
    }

    /**
     * bZ[W_CAO\B
     * @param message bZ[W
     * @param title _CAÕ^Cg
     */
    void showInformationMessageDialog(String message, String title) {
        showInformationMessageDialog(this, message, title);
    }

    /**
     * bZ[W_CAO\B
     * @param parent eR|[lg
     * @param message bZ[W
     * @param title _CAÕ^Cg
     */
    static void showInformationMessageDialog(final Component parent, String message, String title) {
        JTextArea textArea = new JTextArea(message, 6, 60);
        setupReadOnlyTextArea(textArea);
        showMessageDialog(parent, new JScrollPane(textArea), title, INFORMATION_MESSAGE);
    }

    /**
     * G[_CAO\B
     * @param th O
     */
    void showErrorDialog(Throwable th) {
        showErrorDialog(this, th);
    }

    /**
     * G[_CAO\B
     * @param parent eR|[lg
     * @param th O
     */
    static void showErrorDialog(final Component parent, final Throwable th) {
        JPanel p = new JPanel(new BorderLayout());
        final String title = getString("e.error");
        if (th == null) {
            p.add(new JLabel(getString("e.error-no-detail")), BorderLayout.CENTER);
        } else {
            Writer buffer = new StringWriter();
            PrintWriter out = new PrintWriter(buffer);
            th.printStackTrace(out);
            final String text = replaceEOL(buffer.toString());
            JTextArea textArea = new JTextArea(th.getMessage(), 6, 60);
            setupReadOnlyTextArea(textArea);
            p.add(new JScrollPane(textArea), BorderLayout.CENTER);
            JPanel pp = new JPanel();
            p.add(pp, BorderLayout.LINE_END);
            pp.add(new JButton(new AbstractAction(getString("Action.detail")) {

                public void actionPerformed(ActionEvent e) {
                    JEditorPane ep = new JEditorPane("text/plain", text);
                    ep.setCaretPosition(0);
                    ep.setEditable(false);
                    JScrollPane sp = new JScrollPane(ep);
                    JOptionPane pane = new JOptionPane(sp, PLAIN_MESSAGE, DEFAULT_OPTION);
                    JDialog dialog = pane.createDialog(parent, "");
                    dialog.setResizable(true);
                    if (parent == null) {
                        // eR|[lgꍇ͉ʃTCYTCY߂
                        Rectangle r = GraphicsEnvironment.getLocalGraphicsEnvironment()
                                                         .getMaximumWindowBounds();
                        dialog.setSize(r.width / 3, r.height / 4);
                    } else {
                        Dimension size = parent.getSize();
                        size.width *= 1.5;
                        size.height *= 0.9;
                        dialog.setSize(size);
                    }
                    dialog.setLocationRelativeTo(parent);
                    dialog.setVisible(true);
                }

            }));
        }
        showMessageDialog(parent, p, title, ERROR_MESSAGE);
    }

    /**
     * suB
     * @param s 
     * @return ҏWꂽ
     */
    private static String replaceEOL(String s) {
        return s.replaceAll("\\\r\\\n?", EOL);
    }

    static void setupReadOnlyTextArea(JTextArea textArea) {
        textArea.setEditable(false);
        textArea.setWrapStyleWord(false);
        textArea.setLineWrap(true);
        textArea.setOpaque(false);
        textArea.setMargin(new Insets(4, 4, 4, 4));
    }

    /**
     * f̂߂̏o͐oCpXB
     */
    static final class Bypass implements OutputProcessor {

        private OutputProcessor op;
        private volatile boolean closed;

        Bypass(OutputProcessor op) {
            this.op = op;
        }

        /* @see net.argius.stew.OutputProcessor#output(java.lang.Object) */
        public void output(Object object) {
            if (!closed) {
                op.output(object);
            }
        }

        /* @see net.argius.stew.OutputProcessor#close() */
        public void close() {
            closed = true;
        }

    }

}
