/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.KeyEvent.VK_ESCAPE;
import static javax.swing.JOptionPane.*;
import static net.argius.stew.ui.window.Resource.getString;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import java.security.*;
import java.sql.*;
import java.util.*;
import java.util.List;
import java.util.Map.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

import net.argius.stew.*;

/**
 * RlN^ҏW_CAOB
 */
final class ConnectorEditDialog extends JDialog {

    private static final int TEXT_SIZE = 32;
    private static final Insets TEXT_MARGIN = new Insets(1, 3, 1, 0);
    private static final Color COLOR_ESSENTIAL = new Color(0xFFF099);

    private final JTextField tId;
    private final JTextField tName;
    private final JTextField tClasspath;
    private final JTextField tDriver;
    private final JTextField tUrl;
    private final JTextField tUser;
    private final JPasswordField tPassword;
    private final JComboBox cPasswordClass;
    private final JCheckBox cReadOnly;
    private final JCheckBox cUsesAutoRollback;

    private List<ChangeListener> listenerList;
    private File currentDirectory;

    /**
     * RXgN^B
     * @param owner e_CAO
     * @param connector RlN^
     */
    ConnectorEditDialog(JDialog owner, Connector connector) {
        // [CX^X]
        super(owner);
        this.listenerList = new ArrayList<ChangeListener>();
        setTitle(getMessage("title"));
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        // [R|[lg̐ݒ]
        final FlexiblePanel p = new FlexiblePanel();
        add(p);
        // setup and layout text-fields
        String id = connector.getId();
        tId = new JTextField(id, TEXT_SIZE);
        tId.setEditable(false);
        tId.setMargin(TEXT_MARGIN);
        p.addComponent(new JLabel(getMessage("connector.id")), false);
        p.addComponent(tId, true);
        tName = createJTextField(connector.getName(), false);
        p.addComponent(new JLabel(getMessage("connector.name")), false);
        p.addComponent(tName, false);
        p.addComponent(new JButton(new AbstractAction(getMessage("button.refer-to-others")) {

            public void actionPerformed(ActionEvent e) {
                referToOtherConnectors();
            }

        }), true);
        tClasspath = createJTextField(connector.getClasspath(), false);
        p.addComponent(new JLabel(getMessage("connector.classpath")), false);
        p.addComponent(tClasspath, false);
        p.addComponent(new JButton(new AbstractAction(getMessage("button.search.file")) {

            public void actionPerformed(ActionEvent e) {
                chooseClasspath();
            }

        }), true);
        tDriver = createJTextField(connector.getDriver(), true);
        p.addComponent(new JLabel(getMessage("connector.driver")), false);
        p.addComponent(tDriver, false);
        p.addComponent(new JButton(new AbstractAction(getMessage("button.search.driver")) {

            public void actionPerformed(ActionEvent e) {
                chooseDriverClass();
            }

        }), true);
        tUrl = createJTextField(connector.getUrl(), true);
        p.addComponent(new JLabel(getMessage("connector.url")), false);
        p.addComponent(tUrl, true);
        tUser = createJTextField(connector.getUser(), true);
        p.addComponent(new JLabel(getMessage("connector.user")), false);
        p.addComponent(tUser, true);
        tPassword = new JPasswordField(TEXT_SIZE);
        tPassword.setBackground(COLOR_ESSENTIAL);
        tPassword.setMargin(TEXT_MARGIN);
        Password password = connector.getPassword();
        if (id.length() > 0 && password.hasPassword()) {
            tPassword.setText(password.getRowString());
        }
        p.addComponent(new JLabel(getMessage("connector.password")), false);
        p.addComponent(tPassword, true);
        PasswordItem[] items = {new PasswordItem(PlainTextPassword.class),
                                new PasswordItem(PbePassword.class)};
        cPasswordClass = new JComboBox(items);
        cPasswordClass.setEditable(true);
        for (int i = 0; i < items.length; i++) {
            if (items[i].getPasswordClass() == password.getClass()) {
                cPasswordClass.setSelectedIndex(i);
                break;
            }
        }
        p.addComponent(new JLabel(getMessage("connector.encryption")), false);
        p.addComponent(cPasswordClass, true);
        cReadOnly = new JCheckBox(getMessage("connector.readonly"), connector.isReadOnly());
        p.addComponent(cReadOnly, true);
        cUsesAutoRollback = new JCheckBox(getMessage("connector.autorollback"),
                                          connector.usesAutoRollback());
        p.addComponent(cUsesAutoRollback, true);
        // setup and layout buttons
        JPanel p2 = new JPanel(new GridLayout(1, 3, 16, 0));
        p2.add(new JButton(new AbstractAction(getMessage("button.register")) {

            public void actionPerformed(ActionEvent e) {
                requestClose(true);
            }

        }));
        p2.add(new JButton(new AbstractAction(getMessage("button.try.connect")) {

            public void actionPerformed(ActionEvent e) {
                tryToConnect();
            }

        }));
        p2.add(new JButton(new AbstractAction(getMessage("button.cancel")) {

            public void actionPerformed(ActionEvent e) {
                requestClose(false);
            }

        }));
        p.c.gridwidth = GridBagConstraints.REMAINDER;
        p.c.insets = new Insets(12, 0, 12, 0);
        p.c.anchor = GridBagConstraints.CENTER;
        p.addComponent(p2, false);

        pack();
        // [Cxg̐ݒ]
        final String closeKey = "cancel-and-close";
        getRootPane().getActionMap().put(closeKey, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                requestClose(false);
            }

        });
        getRootPane().getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT)
                     .put(KeyStroke.getKeyStroke(VK_ESCAPE, 0), closeKey);
        addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosing(WindowEvent e) {
                requestClose(false);
            }

        });
    }

    /**
     * JTextField̐B
     * @param value l
     * @param isEssential K{ڂ̏ꍇ <code>true</code>
     * @return JTextField
     */
    private JTextField createJTextField(String value, boolean isEssential) {
        final JTextField text = new JTextField(TEXT_SIZE);
        text.setMargin(TEXT_MARGIN);
        if (value != null) {
            text.setText(value);
            text.setCaretPosition(0);
        }
        if (isEssential) {
            text.setBackground(COLOR_ESSENTIAL);
        }
        ActionUtility.getInstance(text).setActionForTextComponent();
        return text;
    }

    /**
     * ύXXio^B
     * @param listener ύXXi
     */
    void addChangeListener(ChangeListener listener) {
        listenerList.add(listener);
    }

    /**
     * ͒lRlN^𐶐B
     * @return RlN^
     */
    private Connector createConnector() {
        final String id = tId.getText();
        Properties props = new Properties();
        props.setProperty("name", tName.getText());
        props.setProperty("driver", tDriver.getText());
        props.setProperty("classpath", tClasspath.getText());
        props.setProperty("url", tUrl.getText());
        props.setProperty("user", tUser.getText());
        props.setProperty("readonly", Boolean.toString(cReadOnly.isSelected()));
        props.setProperty("rollback", Boolean.toString(cUsesAutoRollback.isSelected()));
        Object item = cPasswordClass.getSelectedItem();
        if (item != null && item instanceof PasswordItem) {
            PasswordItem passwordItem = (PasswordItem)item;
            if (passwordItem.getPasswordClass().equals(PlainTextPassword.class)) {
                props.setProperty("password.class", passwordItem.getName());
            }
        }
        Connector connector = new Connector(id, props);
        connector.getPassword().setRowString(String.valueOf(tPassword.getPassword()));
        return connector;
    }

    /**
     * ̐ڑݒQƁEݒ肷B
     */
    void referToOtherConnectors() {
        ConnectorMap m;
        try {
            m = ConnectorConfiguration.load();
        } catch (IOException ex) {
            WindowOutputProcessor.showErrorDialog(this, ex);
            return;
        }
        String[] names = {"id", "name", "classpath", "driver", "url", "user"};
        Vector<Vector<String>> data = new Vector<Vector<String>>();
        for (Entry<String, Connector> entry : m.entrySet()) {
            Vector<String> a = new Vector<String>(names.length);
            Properties p = entry.getValue().toProperties();
            p.setProperty("id", entry.getKey());
            for (String name : names) {
                a.add(p.getProperty(name));
            }
            data.add(a);
        }
        JTable t = new JTable();
        t.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        t.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        t.setDefaultEditor(Object.class, null);
        Vector<String> headers = new Vector<String>();
        for (String name : names) {
            headers.add(getMessage("connector." + name));
        }
        t.setModel(new DefaultTableModel(data, headers));
        JPanel p = new JPanel(new BorderLayout());
        p.add(new JLabel(getMessage("dialog.refer-to-others.message")), BorderLayout.PAGE_START);
        p.add(new JScrollPane(t), BorderLayout.CENTER);
        if (showConfirmDialog(this, p, null, OK_CANCEL_OPTION) != OK_OPTION) {
            return;
        }
        final Properties selected = m.get(t.getValueAt(t.getSelectedRow(), 0)).toProperties();
        Map<String, JTextField> pairs = new HashMap<String, JTextField>();
        pairs.put("name", tName);
        pairs.put("classpath", tClasspath);
        pairs.put("driver", tDriver);
        pairs.put("url", tUrl);
        pairs.put("user", tUser);
        for (Entry<String, JTextField> entry : pairs.entrySet()) {
            JTextField text = entry.getValue();
            if (text.getText().trim().length() == 0) {
                text.setText(selected.getProperty(entry.getKey()));
            }
        }
    }

    /**
     * NXpXIEݒ肷B
     */
    void chooseClasspath() {
        if (currentDirectory == null) {
            synchronized (this) {
                if (currentDirectory == null) {
                    File file = new File(tClasspath.getText().split(File.pathSeparator)[0]);
                    if (file.getParentFile().isDirectory()) {
                        currentDirectory = file.getParentFile();
                    }
                }
            }
        }
        JFileChooser fileChooser = new JFileChooser();
        fileChooser.setCurrentDirectory(currentDirectory);
        fileChooser.setDialogTitle(getMessage("dialog.search.file.header"));
        fileChooser.setApproveButtonText(getMessage("dialog.search.file.button"));
        fileChooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
        fileChooser.showDialog(this, null);
        File file = fileChooser.getSelectedFile();
        if (file != null) {
            tClasspath.setText(file.getPath());
            synchronized (this) {
                currentDirectory = file.getParentFile();
            }
        }
    }

    /**
     * JDBChCoNXIEݒ肷B
     */
    void chooseDriverClass() {
        String text = tClasspath.getText();
        if (text.trim().length() == 0) {
            String message = getMessage("confirm.searchsystemclasspath");
            if (showConfirmDialog(this, message, null, OK_CANCEL_OPTION) != OK_OPTION) {
                return;
            }
            text = LocalSystem.getProperty("java.class.path");
        }
        final Set<String> classes = new LinkedHashSet<String>();
        for (String path : text.split(File.pathSeparator)) {
            File file = new File(path);
            final URL[] urls;
            try {
                urls = new URL[]{file.toURL()};
            } catch (MalformedURLException ex) {
                continue;
            }
            ClassFinder finder = new ClassFinder(path) {

                @Override
                public void filter(Class<?> c) {
                    if (Driver.class.isAssignableFrom(c)) {
                        classes.add(c.getName());
                    }
                }

            };
            URLClassLoader cl = AccessController.doPrivileged(new PrivilegedAction<URLClassLoader>() {

                public URLClassLoader run() {
                    return new URLClassLoader(urls);
                }

            });
            finder.setClassLoader(cl);
            finder.setFailMode(true);
            finder.find(file);
        }
        if (classes.isEmpty()) {
            showMessageDialog(this, getMessage("search.driver.classnotfound"), null, ERROR_MESSAGE);
        } else {
            final String message = getMessage("selectDriverClass");
            Object[] a = classes.toArray();
            Object value = showInputDialog(this, message, "", PLAIN_MESSAGE, null, a, a[0]);
            if (value != null) {
                tDriver.setText((String)value);
                tDriver.setCaretPosition(0);
            }
        }
    }

    /**
     * ڑB
     */
    void tryToConnect() {
        try {
            Connector connector = createConnector();
            Connection conn = connector.getConnection();
            try {
                DatabaseMetaData dbmeta = conn.getMetaData();
                final String message = getMessage("try.connect",
                                                  dbmeta.getDatabaseProductName(),
                                                  dbmeta.getDatabaseProductVersion());
                WindowOutputProcessor.showInformationMessageDialog(this, message, "");
            } finally {
                conn.close();
            }
        } catch (Throwable th) {
            WindowOutputProcessor.showErrorDialog(this, th);
        }
    }

    /**
     * 鏈vB
     * @param withSaving ۑ𔺂ꍇ <code>true</code>
     */
    void requestClose(boolean withSaving) {
        if (withSaving) {
            Connector connector = createConnector();
            ChangeEvent event = new ChangeEvent(connector);
            for (ChangeListener listener : listenerList) {
                listener.stateChanged(event);
            }
        } else {
            if (showConfirmDialog(this,
                                  getString("i.confirm-without-register"),
                                  null,
                                  OK_CANCEL_OPTION) != OK_OPTION) {
                return;
            }
        }
        dispose();
    }

    /**
     * (O)bZ[W̎擾B
     * @param key L[
     * @return bZ[W
     */
    private static String getMessage(String key, Object... args) {
        return Resource.getString("ConnectorEditDialog." + key, args);
    }

    /**
     * pX[hځB
     */
    private static final class PasswordItem {

        final Class<? extends Password> passwordClass;

        PasswordItem(Class<? extends Password> passwordClass) {
            this.passwordClass = passwordClass;
        }

        Class<? extends Password> getPasswordClass() {
            return passwordClass;
        }

        String getName() {
            return toString();
        }

        @Override
        public String toString() {
            return passwordClass.getName();
        }

    }

}
