/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static net.argius.stew.Iteration.join;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.sql.*;
import java.util.*;
import java.util.List;
import java.util.Map.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;

import net.argius.logging.*;
import net.argius.stew.*;

/**
 * f[^x[X\c[B
 */
final class DatabaseInfoTree extends JTree {

    private static final Logger log = LoggerFactory.getLogger(DatabaseInfoTree.class);

    private Connector currentConnector;

    /**
     * RXgN^B
     */
    DatabaseInfoTree() {
        setRootVisible(false);
        setShowsRootHandles(false);
        setScrollsOnExpand(true);
        setCellRenderer(new Renderer());
        setModel(new DefaultTreeModel(null));
        // [Cxg]
        // }EXCxg
        addMouseListener(new MouseAdapter() {

            @Override
            public void mousePressed(MouseEvent e) {
                if (SwingUtilities.isRightMouseButton(e)) {
                    addSelectionPath(getPathForLocation(e.getX(), e.getY()));
                }
            }

        });
        // ANVCxg
        ActionUtility actionUtility = ActionUtility.getInstance(this);
        final String keyCopyFullName = "copy-full-name";
        final String keyRefresh = "refresh";
        final String keyGenerateSelectPhrase = "generate-select-phrase";
        final String keyGenerateWherePhrase = "generate-where-phrase";
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                TreePath[] paths = getSelectionPaths();
                if (paths == null) {
                    return;
                }
                List<String> names = new ArrayList<String>(paths.length);
                for (TreePath path : paths) {
                    InfoNode node = (InfoNode)path.getLastPathComponent();
                    names.add(node.getNodeFullName());
                }
                String s = join(names, String.format("%n"));
                Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
                StringSelection sselection = new StringSelection(s);
                clipboard.setContents(sselection, sselection);
            }

        }, keyCopyFullName);
        actionUtility.bindAction(new AbstractAction(keyRefresh) {

            public void actionPerformed(ActionEvent e) {
                // empty
            }

        });
        actionUtility.bindAction(new AbstractAction(keyGenerateSelectPhrase) {

            public void actionPerformed(ActionEvent e) {
                TreePath[] paths = getSelectionPaths();
                if (paths == null) {
                    return;
                }
                List<ColumnNode> columns = new ArrayList<ColumnNode>();
                Set<String> tableNames = collectTableName(paths, columns);
                if (tableNames.isEmpty()) {
                    return;
                }
                List<String> columnNames = new ArrayList<String>();
                final boolean one = tableNames.size() == 1;
                for (ColumnNode node : columns) {
                    columnNames.add(one ? node.getName() : node.getNodeFullName());
                }
                final String columnString = (columnNames.isEmpty()) ? "*" : join(columnNames, ", ");
                final String tableString = join(tableNames, ", ");
                fireActionPerformed(String.format("SELECT %s FROM %s WHERE ",
                                                  columnString,
                                                  tableString));
            }

        });
        actionUtility.bindAction(new AbstractAction(keyGenerateWherePhrase) {

            public void actionPerformed(ActionEvent e) {
                TreePath[] paths = getSelectionPaths();
                if (paths == null) {
                    return;
                }
                List<ColumnNode> columns = new ArrayList<ColumnNode>();
                Set<String> tableNames = collectTableName(paths, columns);
                if (tableNames.isEmpty()) {
                    return;
                }
                fireActionPerformed(generateEquivalentJoinClause(columns));
            }

        });
    }

    static Set<String> collectTableName(TreePath[] paths, List<ColumnNode> out) {
        Set<String> a = new LinkedHashSet<String>();
        for (TreePath path : paths) {
            InfoNode node = (InfoNode)path.getLastPathComponent();
            if (node instanceof TableNode) {
                a.add(node.getNodeFullName());
            } else if (node instanceof ColumnNode) {
                ColumnNode column = (ColumnNode)node;
                a.add(column.getTableName());
                out.add(column);
            }
        }
        return a;
    }

    void addActionListener(ActionListener listener) {
        listenerList.add(ActionListener.class, listener);
    }

    void removeActionListener(ActionListener listener) {
        listenerList.remove(ActionListener.class, listener);
    }

    void fireActionPerformed(String cmd) {
        ActionEvent event = new ActionEvent(this, ActionEvent.ACTION_PERFORMED, cmd);
        for (ActionListener listener : listenerList.getListeners(ActionListener.class)) {
            listener.actionPerformed(event);
        }
    }

    /**
     * RootvfĐݒ肷B
     * @param env Environment
     * @throws SQLException
     */
    void refreshRoot(Environment env) throws SQLException {
        Connector c = env.getCurrentConnector();
        if (c == null) {
            if (log.isDebugEnabled()) {
                log.debug("not connected");
            }
            currentConnector = null;
            return;
        }
        if (c == currentConnector) {
            if (log.isDebugEnabled()) {
                log.debug("not changed");
            }
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("updating");
        }
        // f
        ConnectorNode connectorNode = new ConnectorNode(c.getName());
        DefaultTreeModel model = new DefaultTreeModel(connectorNode);
        setModel(model);
        final DefaultTreeSelectionModel m = new DefaultTreeSelectionModel();
        m.setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
        setSelectionModel(m);
        // m[h
        final DatabaseMetaData dbmeta = env.getCurrentConnection().getMetaData();
        final Set<InfoNode> createdStatusSet = new HashSet<InfoNode>();
        expandNode(connectorNode, dbmeta);
        createdStatusSet.add(connectorNode);
        expandPath(new TreePath(connectorNode.getFirstChild()));
        // Cxg
        addTreeWillExpandListener(new TreeWillExpandListener() {

            public void treeWillExpand(TreeExpansionEvent event) throws ExpandVetoException {
                TreePath path = event.getPath();
                final Object lastPathComponent = path.getLastPathComponent();
                if (!createdStatusSet.contains(lastPathComponent)) {
                    InfoNode node = (InfoNode)lastPathComponent;
                    createdStatusSet.add(node);
                    try {
                        expandNode(node, dbmeta);
                    } catch (SQLException ex) {
                        throw new RuntimeException(ex);
                    }
                }
            }

            public void treeWillCollapse(TreeExpansionEvent event) throws ExpandVetoException {
                // ignore
            }

        });
        // \
        model.reload();
        setRootVisible(true);
        this.currentConnector = c;
    }

    /**
     * qm[hݒ肵ēWJ܂B
     * @param parent em[h
     * @param dbmeta DB^
     * @throws SQLException
     */
    void expandNode(InfoNode parent, DatabaseMetaData dbmeta) throws SQLException {
        final DefaultTreeModel model = (DefaultTreeModel)getModel();
        for (InfoNode child : parent.createChildren(dbmeta)) {
            model.insertNodeInto(child, parent, parent.getChildCount());
        }
    }

    /**
     * NAB
     */
    void clear() {
        for (TreeWillExpandListener listener : getListeners(TreeWillExpandListener.class).clone()) {
            removeTreeWillExpandListener(listener);
        }
        setModel(null);
        currentConnector = null;
    }

    /**
     * 𐶐B
     * @param nodes m[hXg
     * @return 
     */
    static String generateEquivalentJoinClause(List<ColumnNode> nodes) {
        if (nodes.isEmpty()) {
            return "";
        }
        ListMap tm = new ListMap();
        ListMap cm = new ListMap();
        for (ColumnNode node : nodes) {
            final String tableName = node.getTableName();
            final String columnName = node.getName();
            tm.add(tableName, columnName);
            cm.add(columnName, String.format("%s.%s", tableName, columnName));
        }
        StringBuilder buffer = new StringBuilder();
        if (tm.size() == 1) {
            for (ColumnNode node : nodes) {
                buffer.append(String.format("%s=? AND ", node.getName()));
            }
        } else {
            final String tableName = nodes.get(0).getTableName();
            for (String c : tm.get(tableName)) {
                buffer.append(String.format("%s.%s=? AND ", tableName, c));
            }
            for (Entry<String, List<String>> entry : cm.entrySet()) {
                if (!entry.getKey().equals(tableName) && entry.getValue().size() == 1) {
                    buffer.append(String.format("%s=? AND ", entry.getValue().get(0)));
                }
            }
            for (Entry<String, List<String>> entry : cm.entrySet()) {
                Object[] a = entry.getValue().toArray();
                final int n = a.length;
                for (int i = 0; i < n; i++) {
                    for (int j = i + 1; j < n; j++) {
                        buffer.append(String.format("%s=%s AND ", a[i], a[j]));
                    }
                }
            }
        }
        return buffer.toString();
    }

    /**
     * Xg}bvB
     */
    private static final class ListMap extends LinkedHashMap<String, List<String>> {

        ListMap() {
            // empty
        }

        void add(String key, String value) {
            if (get(key) == null) {
                put(key, new ArrayList<String>());
            }
            get(key).add(value);
        }

    }

    /**
     * vf̕`揈(Renderer)B
     */
    private static class Renderer extends DefaultTreeCellRenderer {

        Renderer() {
            // empty
        }

        @Override
        public Component getTreeCellRendererComponent(JTree tree,
                                                      Object value,
                                                      boolean sel,
                                                      boolean expanded,
                                                      boolean leaf,
                                                      int row,
                                                      boolean hasFocus) {
            super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            if (value instanceof InfoNode) {
                setIcon(Resource.getImageIcon(((InfoNode)value).getIconName()));
            }
            return this;
        }

    }

    /**
     * m[hB
     */
    private static class InfoNode extends DefaultMutableTreeNode {

        InfoNode(Object userObject) {
            super(userObject, true);
        }

        @Override
        public boolean isLeaf() {
            return false;
        }

        String getIconName() {
            final String className = getClass().getName();
            final String nodeType = className.replaceFirst(".+?([^\\$]+)Node$", "$1");
            return "node-" + nodeType.toLowerCase() + ".png";
        }

        protected List<InfoNode> createChildren(DatabaseMetaData dbmeta) throws SQLException {
            return Collections.emptyList();
        }

        protected String getNodeFullName() {
            return String.valueOf(userObject);
        }

    }

    /**
     * RlN^̃m[hB
     */
    private static class ConnectorNode extends InfoNode {

        ConnectorNode(String name) {
            super(name);
        }

        @Override
        protected List<InfoNode> createChildren(DatabaseMetaData dbmeta) throws SQLException {
            List<InfoNode> a = new ArrayList<InfoNode>();
            if (dbmeta.supportsCatalogsInDataManipulation()) {
                ResultSet rs = dbmeta.getCatalogs();
                try {
                    while (rs.next()) {
                        a.add(new CatalogNode(rs.getString(1)));
                    }
                } finally {
                    rs.close();
                }
            } else if (dbmeta.supportsSchemasInDataManipulation()) {
                ResultSet rs = dbmeta.getSchemas();
                try {
                    while (rs.next()) {
                        a.add(new SchemaNode(null, rs.getString(1)));
                    }
                } finally {
                    rs.close();
                }
            } else {
                ResultSet rs = dbmeta.getTableTypes();
                try {
                    while (rs.next()) {
                        a.add(new TableTypeNode(null, null, rs.getString(1)));
                    }
                } finally {
                    rs.close();
                }
            }
            return a;
        }

    }

    /**
     * J^Õm[hB
     */
    private static final class CatalogNode extends InfoNode {

        private final String name;

        CatalogNode(String name) {
            super(name);
            this.name = name;
        }

        @Override
        protected List<InfoNode> createChildren(DatabaseMetaData dbmeta) throws SQLException {
            List<InfoNode> a = new ArrayList<InfoNode>();
            if (dbmeta.supportsSchemasInDataManipulation()) {
                ResultSet rs = dbmeta.getSchemas();
                try {
                    while (rs.next()) {
                        a.add(new SchemaNode(name, rs.getString(1)));
                    }
                } finally {
                    rs.close();
                }
            } else {
                ResultSet rs = dbmeta.getTableTypes();
                try {
                    while (rs.next()) {
                        a.add(new TableTypeNode(name, null, rs.getString(1)));
                    }
                } finally {
                    rs.close();
                }
            }
            return a;
        }

    }

    /**
     * XL[}̃m[hB
     */
    private static final class SchemaNode extends InfoNode {

        private final String catalog;
        private final String schema;

        SchemaNode(String catalog, String schema) {
            super(schema);
            this.catalog = catalog;
            this.schema = schema;
        }

        @Override
        protected List<InfoNode> createChildren(DatabaseMetaData dbmeta) throws SQLException {
            List<InfoNode> a = new ArrayList<InfoNode>();
            ResultSet rs = dbmeta.getTableTypes();
            try {
                while (rs.next()) {
                    a.add(new TableTypeNode(catalog, schema, rs.getString(1)));
                }
            } finally {
                rs.close();
            }
            return a;
        }

    }

    /**
     * e[uʂ̃m[hB
     */
    private static final class TableTypeNode extends InfoNode {

        private static final String ICON_NAME_FORMAT = "node-tabletype-%s.png";

        private final String catalog;
        private final String schema;
        private final String tableType;

        TableTypeNode(String catalog, String schema, String tableType) {
            super(tableType);
            this.catalog = catalog;
            this.schema = schema;
            this.tableType = tableType;
        }

        @Override
        protected List<InfoNode> createChildren(DatabaseMetaData dbmeta) throws SQLException {
            List<InfoNode> a = new ArrayList<InfoNode>();
            ResultSet rs = dbmeta.getTables(catalog, schema, null, new String[]{tableType});
            try {
                while (rs.next()) {
                    a.add(new TableNode(catalog, schema, rs.getString(3)));
                }
            } finally {
                rs.close();
            }
            return a;
        }

        @Override
        String getIconName() {
            final String name = String.format(ICON_NAME_FORMAT, getUserObject());
            if (getClass().getResource("icon/" + name) == null) {
                return String.format(ICON_NAME_FORMAT, "");
            }
            return name;
        }

    }

    /**
     * e[ũm[hB
     */
    private static final class TableNode extends InfoNode {

        private final String catalog;
        private final String schema;
        private final String name;

        TableNode(String catalog, String schema, String name) {
            super(name);
            this.catalog = catalog;
            this.schema = schema;
            this.name = name;
        }

        @Override
        protected List<InfoNode> createChildren(DatabaseMetaData dbmeta) throws SQLException {
            List<InfoNode> a = new ArrayList<InfoNode>();
            ResultSet rs = dbmeta.getColumns(catalog, schema, name, null);
            try {
                while (rs.next()) {
                    a.add(new ColumnNode(name,
                                         rs.getString(4),
                                         rs.getString(6),
                                         rs.getInt(7),
                                         rs.getString(18)));
                }
            } finally {
                rs.close();
            }
            return a;
        }

        @Override
        public boolean isLeaf() {
            return false;
        }

        @Override
        protected String getNodeFullName() {
            List<String> a = new ArrayList<String>();
            if (catalog != null) {
                a.add(catalog);
            }
            if (schema != null) {
                a.add(schema);
            }
            a.add(name);
            return Iteration.join(a, ".");
        }

    }

    /**
     * ̃m[hB
     */
    static final class ColumnNode extends InfoNode {

        private String name;
        private String table;

        ColumnNode(String table, String name, String type, int size, String nulls) {
            super(format(name, type, size, nulls));
            setAllowsChildren(false);
            this.name = name;
            this.table = table;
        }

        String getName() {
            return name;
        }

        String getTableName() {
            return table;
        }

        private static String format(String name, String type, int size, String nulls) {
            final String nonNull = "NO".equals(nulls) ? " NOT NULL" : "";
            return String.format("%s [%s(%d)%s]", name, type, size, nonNull);
        }

        @Override
        public boolean isLeaf() {
            return false;
        }

        @Override
        protected String getNodeFullName() {
            return String.format("%s.%s", table, name);
        }

    }

}
