/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.InputEvent.*;
import static java.awt.event.KeyEvent.*;
import static javax.swing.JOptionPane.*;
import static javax.swing.JSplitPane.VERTICAL_SPLIT;
import static javax.swing.KeyStroke.getKeyStroke;
import static javax.swing.ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS;
import static net.argius.stew.Iteration.join;
import static net.argius.stew.ui.window.Resource.*;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.dnd.*;
import java.awt.event.*;
import java.beans.*;
import java.io.*;
import java.sql.*;
import java.util.*;
import java.util.List;
import java.util.Timer;
import java.util.concurrent.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.text.*;
import javax.swing.text.Highlighter.*;
import javax.swing.tree.*;

import net.argius.logging.*;
import net.argius.stew.*;
import net.argius.stew.io.*;
import net.argius.stew.ui.*;
import net.argius.stew.ui.window.Menu.*;

/**
 * EBhE[hNB
 */
public final class WindowLauncher {

    private static final Logger log = LoggerFactory.getLogger(WindowLauncher.class);

    private static final List<WindowLauncher> instances = Collections.synchronizedList(new ArrayList<WindowLauncher>());
    private static final ImageIcon ICON = Resource.getImageIcon("stew.png");

    JComponent focused;

    private final Environment env;
    private final WindowOutputProcessor op;
    private final JSplitPane sp0;
    private final JSplitPane sp;
    private final ResultSetTable resultSetTable;
    private final ConsoleTextArea textArea;
    private final DatabaseInfoTree infoTree;
    private final TextSearchPanel textSearchPanel;
    private final JLabel statusBar;
    private final List<String> historyList;
    private final ExecutorService executorService;

    private Map<JComponent, TextSearch> textSearchMap;
    private int historyIndex;

    WindowLauncher() {
        // [CX^X]
        instances.add(this);
        final ResultSetTable resultSetTable = new ResultSetTable();
        final ConsoleTextArea textArea = new ConsoleTextArea();
        this.env = new Environment();
        this.op = new WindowOutputProcessor(env, resultSetTable, textArea);
        this.sp0 = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        this.sp = new JSplitPane();
        this.resultSetTable = resultSetTable;
        this.textArea = textArea;
        this.infoTree = new DatabaseInfoTree();
        this.textSearchPanel = new TextSearchPanel(op);
        this.statusBar = new JLabel(" ");
        this.historyList = new LinkedList<String>();
        this.historyIndex = 0;
        this.executorService = Executors.newScheduledThreadPool(3);
        // [R|[lg̐ݒ]
        // OutputProcessor as frame
        op.setTitle(Resource.getString(".title"));
        op.setIconImage(ICON.getImage());
        op.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        // menu
        Menu menu = new Menu();
        // splitpane (table and textarea)
        sp.setOrientation(VERTICAL_SPLIT);
        sp.setDividerSize(6);
        sp.setResizeWeight(0.6f);
        // rstable
        JScrollPane topPane = new JScrollPane(resultSetTable);
        // text area
        textArea.setMargin(new Insets(4, 8, 4, 4));
        textArea.setLineWrap(true);
        textArea.setWrapStyleWord(false);
        // text search
        this.textSearchMap = new LinkedHashMap<JComponent, TextSearch>();
        textSearchMap.put(resultSetTable, createTextSearch(resultSetTable));
        textSearchMap.put(textArea, createTextSearch(textArea));
        for (TextSearch textSearch : textSearchMap.values()) {
            textSearchPanel.addTarget(textSearch);
        }
        // status bar
        statusBar.setForeground(Color.BLUE);
        // [CAEg]
        JScrollPane sptextarea = new JScrollPane(textArea);
        sptextarea.setVerticalScrollBarPolicy(VERTICAL_SCROLLBAR_ALWAYS);
        final JPanel p = new JPanel(new BorderLayout());
        p.add(sptextarea, BorderLayout.CENTER);
        p.add(textSearchPanel, BorderLayout.SOUTH);
        sp.setTopComponent(topPane);
        sp.setBottomComponent(p);
        op.add(statusBar, BorderLayout.PAGE_END);
        op.setJMenuBar(menu);
        sp0.setTopComponent(new JScrollPane(infoTree));
        sp0.setBottomComponent(sp);
        op.add(sp0, BorderLayout.CENTER);
        setInfoTreePane(false);
        // [ۑꂽԂ̕]
        op.addPropertyChangeListener(menu);
        infoTree.addPropertyChangeListener(menu);
        resultSetTable.addPropertyChangeListener(menu);
        statusBar.addPropertyChangeListener(menu);
        loadConfiguration();
        op.removePropertyChangeListener(menu);
        infoTree.removePropertyChangeListener(menu);
        resultSetTable.removePropertyChangeListener(menu);
        // XXX LR[hsƐݒ̕ŃXe[^Xo[̕łȂ
        // statusBar.removePropertyChangeListener(menu);
        // [Cxg̐ݒ]
        op.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosing(WindowEvent e) {
                requestClose();
            }

        });
        infoTree.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                textArea.replace(e.getActionCommand());
            }

        });
        bindEvents(resultSetTable, topPane);
        bindEvents(textArea);
        menu.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                performMenuEvent(Item.valueOf(e.getActionCommand()), (Component)e.getSource());
            }

        });
        // [Jn]
        env.setOutputProcessor(new WindowOutputProcessor.Bypass(op));
        op.output(new Prompt(env));
        textArea.requestFocus();
    }

    /**
     * ݒ̕B
     */
    private void loadConfiguration() {
        Configuration cnf = Configuration.load(LocalSystem.getDirectory());
        op.setSize(cnf.getSize());
        op.setLocation(cnf.getLocation());
        sp.setDividerLocation(cnf.getDividerLocation());
        statusBar.setVisible(cnf.isShowStatusBar());
        resultSetTable.setShowColumnNumber(cnf.isShowTableColumnNumber());
        setInfoTreePane(cnf.isShowInfoTree());
        op.setAlwaysOnTop(cnf.isAlwaysOnTop());
        op.setAutoAdjustMode(Item.valueOf(cnf.getAutoAdjustMode()));
        changeFont("monospaced", Font.PLAIN, 1.0d);
    }

    /**
     * ݒ̕ۑB
     */
    private void saveConfiguration() {
        Configuration cnf = Configuration.load(LocalSystem.getDirectory());
        if ((op.getExtendedState() & Frame.MAXIMIZED_BOTH) == 0) {
            cnf.setSize(op.getSize());
            cnf.setLocation(op.getLocation());
            cnf.setDividerLocation(sp.getDividerLocation());
        }
        cnf.setShowStatusBar(statusBar.isVisible());
        cnf.setShowTableColumnNumber(resultSetTable.isShowColumnNumber());
        cnf.setShowInfoTree(infoTree.isEnabled());
        cnf.setAlwaysOnTop(op.isAlwaysOnTop());
        cnf.setAutoAdjustMode(op.getAutoAdjustMode().name());
        cnf.save();
    }

    /**
     * ԕۑ@\(Bean)B
     */
    @SuppressWarnings("all")
    public static final class Configuration {

        private static final Logger log = LoggerFactory.getLogger(Configuration.class);

        private transient File directory;

        private Dimension size;
        private Point location;
        private int dividerLocation;
        private boolean showStatusBar;
        private boolean showTableColumnNumber;
        private boolean showInfoTree;
        private boolean alwaysOnTop;
        private String autoAdjustMode;

        public Configuration() {
            this.size = new Dimension(640, 480);
            this.location = new Point(200, 200);
            this.dividerLocation = -1;
            this.showStatusBar = false;
            this.showTableColumnNumber = false;
            this.showInfoTree = false;
            this.alwaysOnTop = false;
            this.autoAdjustMode = Item.AUTO_ADJUST_MODE_NONE.name();
        }

        void setDirectory(File directory) {
            this.directory = directory;
        }

        void save() {
            final File file = getFile(directory);
            if (log.isDebugEnabled()) {
                log.debug(String.format("save Configuration to: [%s]", file));
            }
            try {
                XMLEncoder encoder = new XMLEncoder(new FileOutputStream(file));
                try {
                    encoder.writeObject(this);
                } finally {
                    encoder.close();
                }
            } catch (Exception ex) {
                log.warn("", ex);
            }
        }

        static Configuration load(File directory) {
            final File file = getFile(directory);
            if (log.isDebugEnabled()) {
                log.debug(String.format("load Configuration from: [%s]", file));
            }
            if (file.exists()) {
                try {
                    XMLDecoder decoder = new XMLDecoder(new FileInputStream(file));
                    try {
                        final Configuration instance = (Configuration)decoder.readObject();
                        instance.setDirectory(directory);
                        return instance;
                    } finally {
                        decoder.close();
                    }
                } catch (Exception ex) {
                    log.warn("", ex);
                }
            }
            final Configuration instance = new Configuration();
            instance.setDirectory(directory);
            return instance;
        }

        private static File getFile(File systemDirectory) {
            final File file = new File(systemDirectory, Configuration.class.getName() + ".xml");
            return file.getAbsoluteFile();
        }

        public Dimension getSize() {
            return size;
        }

        public void setSize(Dimension size) {
            this.size = size;
        }

        public Point getLocation() {
            return location;
        }

        public void setLocation(Point location) {
            this.location = location;
        }

        public int getDividerLocation() {
            return dividerLocation;
        }

        public void setDividerLocation(int dividerLocation) {
            this.dividerLocation = dividerLocation;
        }

        public boolean isShowStatusBar() {
            return showStatusBar;
        }

        public void setShowStatusBar(boolean showStatusBar) {
            this.showStatusBar = showStatusBar;
        }

        public boolean isShowTableColumnNumber() {
            return showTableColumnNumber;
        }

        public void setShowTableColumnNumber(boolean showTableColumnNumber) {
            this.showTableColumnNumber = showTableColumnNumber;
        }

        public boolean isShowInfoTree() {
            return showInfoTree;
        }

        public void setShowInfoTree(boolean showInfoTree) {
            this.showInfoTree = showInfoTree;
        }

        public boolean isAlwaysOnTop() {
            return alwaysOnTop;
        }

        public void setAlwaysOnTop(boolean alwaysOnTop) {
            this.alwaysOnTop = alwaysOnTop;
        }

        public String getAutoAdjustMode() {
            return autoAdjustMode;
        }

        public void setAutoAdjustMode(String autoAdjustMode) {
            this.autoAdjustMode = autoAdjustMode;
        }

    }

    private void changeFont(String family, int style, double sizeRate) {
        FontControlLookAndFeel.change(family, style, sizeRate);
        SwingUtilities.updateComponentTreeUI(op);
        Font newfont = textArea.getFont();
        if (newfont != null) {
            statusBar.setFont(newfont.deriveFont(newfont.getSize() * 0.8f));
        }
    }

    /**
     * ResultSetTableɃCxg蓖ĂB
     * @param rstable
     * @param topPane
     */
    private void bindEvents(final ResultSetTable rstable, JScrollPane topPane) {
        // [ANVCxg]
        final int shortcutKey = Resource.getMenuShortcutKeyMask();
        final String keyCopyWithEscape = "copyWithEscape";
        final String keyPaste = "paste";
        final String keyCopyColumnNames = "copyColumnNames";
        final String keyClearCells = "clearCells";
        final String keyAddEmptyRow = "addEmptyRow";
        final String keyInsertFromClipboard = "insertFromClipboard";
        final String keyDuplicateRows = "duplicateRows";
        final String keyLinkRowsToDatabase = "linkRowsToDatabase";
        final String keyDeleteRows = "deleteRows";
        final ActionUtility actionUtility = new ActionUtility(rstable);
        final Action cancelTableEditAction = new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                rstable.editingCanceled(new ChangeEvent(e.getSource()));
            }

        };
        // GXP[vtŃRs[
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                List<String> rows = new ArrayList<String>();
                for (int rowIndex : rstable.getSelectedRows()) {
                    List<Object> row = new ArrayList<Object>();
                    for (int columnIndex : rstable.getSelectedColumns()) {
                        final Object o = rstable.getValueAt(rowIndex, columnIndex);
                        row.add(CsvFormatter.AUTO.format(o == null ? "" : String.valueOf(o)));
                    }
                    rows.add(join(row, TAB));
                }
                setClipboard(join(rows, EOL));
            }

        }, keyCopyWithEscape, getKeyStroke(VK_C, shortcutKey | ALT_DOWN_MASK));
        // \t
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                try {
                    InputStream is = new ByteArrayInputStream(getClipboard().getBytes());
                    Importer importer = new SmartImporter(is, TAB);
                    try {
                        int[] selectedColumns = rstable.getSelectedColumns();
                        for (int rowIndex : rstable.getSelectedRows()) {
                            Object[] values = importer.nextRow();
                            final int limit = Math.min(selectedColumns.length, values.length);
                            for (int x = 0; x < limit; x++) {
                                rstable.setValueAt(values[x], rowIndex, selectedColumns[x]);
                            }
                        }
                    } finally {
                        importer.close();
                    }
                    rstable.repaint();
                } catch (Exception ex) {
                    handleError(ex);
                } finally {
                    cancelTableEditAction.actionPerformed(e);
                }
            }

        }, keyPaste);
        // 񖼂Rs[
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                List<String> a = new ArrayList<String>();
                ResultSetTableModel m = rstable.getResultSetTableModel();
                for (int i = 0, n = m.getColumnCount(); i < n; i++) {
                    a.add(m.getColumnName(i));
                }
                setClipboard(join(a, TAB));
            }

        }, keyCopyColumnNames);
        // IZ̒l폜(NULLݒ)
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                try {
                    int[] selectedColumns = rstable.getSelectedColumns();
                    for (int rowIndex : rstable.getSelectedRows()) {
                        final int limit = selectedColumns.length;
                        for (int x = 0; x < limit; x++) {
                            rstable.setValueAt(null, rowIndex, selectedColumns[x]);
                        }
                    }
                    rstable.repaint();
                } catch (Exception ex) {
                    handleError(ex);
                } finally {
                    cancelTableEditAction.actionPerformed(e);
                }
            }

        }, keyClearCells, getKeyStroke(VK_DELETE, 0));
        // Vs(N)ǉ
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                m.addUnlinkedRow(new Object[m.getColumnCount()]);
            }

        }, keyAddEmptyRow);
        // Nbv{[h}
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                try {
                    InputStream is = new ByteArrayInputStream(getClipboard().getBytes());
                    Importer importer = new SmartImporter(is, TAB);
                    try {
                        ResultSetTableModel m = rstable.getResultSetTableModel();
                        while (true) {
                            Object[] row = importer.nextRow();
                            if (row.length == 0) {
                                break;
                            }
                            m.addUnlinkedRow(row);
                            m.linkRow(m.getRowCount() - 1);
                        }
                        rstable.repaintRowHeader("model");
                    } finally {
                        importer.close();
                    }
                } catch (Exception ex) {
                    handleError(ex);
                } finally {
                    cancelTableEditAction.actionPerformed(e);
                }
            }

        }, keyInsertFromClipboard);
        // Is𕡐Ēǉ
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                List<?> rows = m.getDataVector();
                for (int rowIndex : rstable.getSelectedRows()) {
                    m.addUnlinkedRow((Vector<?>)((Vector<?>)rows.get(rowIndex)).clone());
                }
                rstable.repaint();
                rstable.repaintRowHeader("model");
            }

        }, keyDuplicateRows);
        // ۗsm
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                try {
                    for (int rowIndex : rstable.getSelectedRows()) {
                        m.linkRow(rowIndex);
                    }
                } catch (SQLException ex) {
                    throw new RuntimeException(ex);
                } finally {
                    rstable.repaintRowHeader("rowSuspendedStatus");
                }
            }

        }, keyLinkRowsToDatabase);
        // Is폜
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                try {
                    for (int rowIndex : rstable.getSelectedRows()) {
                        if (m.isLinkedRow(rowIndex)) {
                            final boolean removed = m.removeLinkedRow(rowIndex);
                            assert removed;
                        } else {
                            m.removeRow(rowIndex);
                        }
                    }
                } catch (SQLException ex) {
                    throw new RuntimeException(ex);
                } finally {
                    rstable.repaintRowHeader("model");
                }
            }

        }, keyDeleteRows, getKeyStroke(VK_MINUS, shortcutKey | SHIFT_DOWN_MASK));
        // [ReLXgj[]
        final JPopupMenu pmenu = new JPopupMenu();
        final List<JMenuItem> updatables = new ArrayList<JMenuItem>();
        for (String actionName : new String[]{"copy", keyCopyWithEscape, keyPaste,
                                              keyCopyColumnNames, keyClearCells, "selectAll", "",
                                              keyAddEmptyRow, keyInsertFromClipboard,
                                              keyDuplicateRows, keyLinkRowsToDatabase,
                                              keyDeleteRows,}) {
            if (actionName.length() == 0) {
                pmenu.addSeparator();
            } else {
                final String s = getString("Action." + actionName);
                pmenu.add(actionUtility.copyActionTo(actionName, new JMenuItem(s)));
                if (actionName.equals(keyPaste)
                    || actionName.equals(keyClearCells)
                    || actionName.equals(keyAddEmptyRow)
                    || actionName.equals(keyInsertFromClipboard)
                    || actionName.equals(keyDuplicateRows)
                    || actionName.equals(keyLinkRowsToDatabase)
                    || actionName.equals(keyDeleteRows)) {
                    updatables.add(new JMenuItem(s));
                }
            }
        }
        pmenu.addPopupMenuListener(new PopupMenuListener() {

            public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                final boolean updatable = m.isUpdatable();
                for (JMenuItem item : updatables) {
                    item.setEnabled(updatable);
                }
            }

            public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
                // empty
            }

            public void popupMenuCanceled(PopupMenuEvent e) {
                // empty
            }

        });
        rstable.setComponentPopupMenu(pmenu);
        // [wb_]
        bindEvents(rstable.getTableHeader(), actionUtility);
        // [swb_]
        bindEvents(rstable.getRowHeader(), actionUtility);
    }

    /**
     * ColumnHeaderɃCxg蓖ĂB
     * @param columnHeader ColumnHeader
     * @param t ResultSetTable
     */
    private void bindEvents(final JTableHeader columnHeader, final ActionUtility actionUtility) {
        // [ReLXgj[]
        final JPopupMenu pmenu = new JPopupMenu();
        final Point mousePosition = new Point();
        final WindowOutputProcessor op = this.op;
        pmenu.add(new JMenuItem(new AbstractAction("sort") {

            public void actionPerformed(ActionEvent e) {
                op.sortTable(columnHeader.columnAtPoint(mousePosition));
            }

        }));
        for (String actionName : new String[]{"copy", "copyWithEscape", "paste", "clearCells",
                                              "copyColumnNames", "", "addEmptyRow",
                                              "insertFromClipboard",}) {
            if (actionName.length() == 0) {
                pmenu.addSeparator();
            } else {
                pmenu.add(actionUtility.copyActionTo(actionName,
                                                     new JMenuItem(getString("Action." + actionName))));
            }
        }
        columnHeader.setComponentPopupMenu(pmenu);
    }

    /**
     * RowHeaderɃCxg蓖ĂB
     * @param rowHeader RowHeader
     * @param resultSetTable ResultSetTable
     */
    private void bindEvents(final JTable rowHeader, final ActionUtility actionUtility) {
        // [ReLXgj[]
        final JPopupMenu pmenu = new JPopupMenu();
        for (String actionName : new String[]{"copy", "copyWithEscape", "paste", "clearCells",
                                              "copyColumnNames", "", "addEmptyRow",
                                              "insertFromClipboard",}) {
            if (actionName.length() == 0) {
                pmenu.addSeparator();
            } else {
                pmenu.add(actionUtility.copyActionTo(actionName,
                                                     new JMenuItem(getString("Action." + actionName))));
            }
        }
        rowHeader.setComponentPopupMenu(pmenu);
    }

    /**
     * ConsoleTextAreaɃCxg蓖ĂB
     * @param textArea
     */
    private void bindEvents(final ConsoleTextArea textArea) {
        // [ANVCxg]
        final ActionUtility actionUtility = new ActionUtility(textArea);
        final OutputProcessor op = this.op;
        textArea.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                executeCommand(e.getActionCommand());
            }

        });
        // [hbvCxg]
        textArea.setDropTarget(new DropTarget(textArea, new DropTargetAdapter() {

            public void drop(DropTargetDropEvent dtde) {
                final List<File> fileList;
                Transferable t = dtde.getTransferable();
                if (t.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
                    dtde.acceptDrop(DnDConstants.ACTION_COPY_OR_MOVE);
                    try {
                        @SuppressWarnings("unchecked")
                        List<File> list = (List<File>)t.getTransferData(DataFlavor.javaFileListFlavor);
                        fileList = list;
                    } catch (UnsupportedFlavorException ex) {
                        throw new RuntimeException(ex);
                    } catch (IOException ex) {
                        throw new RuntimeException(ex);
                    }
                } else {
                    throw new IllegalStateException("Transferable=" + t);
                }
                JPopupMenu menu = new JPopupMenu();
                menu.add(new JMenuItem(new AbstractAction(getString("dropevent.pastepath")) {

                    public void actionPerformed(ActionEvent e) {
                        op.output(" " + join(fileList, " "));
                    }

                }));
                menu.add(new JMenuItem(new AbstractAction(getString("dropevent.pastefilecontent")) {

                    public void actionPerformed(ActionEvent e) {
                        try {
                            for (File file : fileList) {
                                Scanner scanner = new Scanner(file);
                                try {
                                    while (scanner.hasNextLine()) {
                                        op.output(String.format("%s%n", scanner.nextLine()));
                                    }
                                } finally {
                                    scanner.close();
                                }
                            }
                        } catch (FileNotFoundException ex) {
                            throw new RuntimeException(ex);
                        }
                    }
                }));
                Point p = dtde.getLocation();
                menu.show(textArea, p.x, p.y);
            }

        }));
        // [̑]
        // ReLXgj[+AhD
        actionUtility.setActionForTextComponent();
        // TODO eLXgANVʉɂhomePosition̔肪łȂ
    }

    /**
     * ResultSetTablepTextSearch̐B
     * @param table
     * @return TextSearch
     */
    private TextSearch createTextSearch(final ResultSetTable table) {
        final TextSearch textSearch = new TextSearch() {

            public boolean search(Matcher matcher) {
                final int rowCount = table.getRowCount();
                if (rowCount <= 0) {
                    return false;
                }
                final int columnCount = table.getColumnCount();
                final boolean backward = matcher.isBackward();
                final int amount = backward ? -1 : 1;
                final int rowStart = backward ? rowCount - 1 : 0;
                final int rowEnd = backward ? 0 : rowCount - 1;
                final int columnStart = backward ? columnCount - 1 : 0;
                final int columnEnd = backward ? 0 : columnCount - 1;
                int row = rowStart;
                int column = columnStart;
                if (table.getSelectedColumnCount() > 0) {
                    column = table.getSelectedColumn();
                    row = table.getSelectedRow() + amount;
                    if (backward) {
                        if (row < 0) {
                            --column;
                            if (column < 0) {
                                return false;
                            }
                            row = rowStart;
                        }
                    } else {
                        if (row >= rowCount) {
                            ++column;
                            if (column >= columnCount) {
                                return false;
                            }
                            row = rowStart;
                        }
                    }
                }
                final TableModel m = table.getModel();
                for (; backward ? column >= columnEnd : column <= columnEnd; column += amount) {
                    for (; backward ? row >= rowEnd : row <= rowEnd; row += amount) {
                        if (matcher.find(String.valueOf(m.getValueAt(row, column)))) {
                            table.changeSelection(row, column, false, false);
                            table.requestFocus();
                            return true;
                        }
                    }
                    row = rowStart;
                }
                return false;
            }

            public void reset() {
                // empty
            }

        };
        table.addFocusListener(new FocusAdapter() {

            @Override
            public void focusGained(FocusEvent e) {
                focused = table;
            }

        });
        return textSearch;
    }

    /**
     * ConsoleTextAreapTextSearch̐B
     * @param textArea
     * @return TextSearch
     */
    private TextSearch createTextSearch(final ConsoleTextArea textArea) {
        final TextSearch textSearch = new TextSearch() {

            public boolean search(Matcher matcher) {
                removeHighlights();
                try {
                    Highlighter highlighter = textArea.getHighlighter();
                    HighlightPainter painter = matcher.getHighlightPainter();
                    String text = textArea.getText();
                    int start = 0;
                    boolean matched = false;
                    while (matcher.find(text, start)) {
                        matched = true;
                        int matchedIndex = matcher.getStart();
                        highlighter.addHighlight(matchedIndex, matcher.getEnd(), painter);
                        start = matchedIndex + 1;
                    }
                    textArea.addKeyListener(new KeyAdapter() {

                        @Override
                        public void keyTyped(KeyEvent e) {
                            textArea.removeKeyListener(this);
                            removeHighlights();
                        }

                    });
                    return matched;
                } catch (BadLocationException ex) {
                    throw new RuntimeException(ex);
                }
            }

            public void reset() {
                removeHighlights();
            }

            void removeHighlights() {
                for (Highlight highlight : textArea.getHighlighter().getHighlights()) {
                    textArea.getHighlighter().removeHighlight(highlight);
                }
            }

        };
        textArea.addFocusListener(new FocusAdapter() {

            @Override
            public void focusGained(FocusEvent e) {
                focused = textArea;
            }

        });
        return textSearch;
    }

    /**
     * Nbv{[h當擾B
     * @return 
     * @throws IOException 
     */
    static String getClipboard() throws IOException {
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        String s;
        try {
            Transferable t = clipboard.getContents(null);
            s = (String)t.getTransferData(DataFlavor.stringFlavor);
        } catch (UnsupportedFlavorException ex) {
            IOException exception = new IOException();
            exception.initCause(exception);
            throw exception;
        }
        if (log.isTraceEnabled()) {
            log.trace(String.format("get clipboard:[%s]", s));
        }
        return s;
    }

    /**
     * Nbv{[hɕݒ肷B
     * @param s 
     */
    static void setClipboard(String s) {
        if (log.isTraceEnabled()) {
            log.trace(String.format("set clipboard:[%s]", s));
        }
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        StringSelection sselection = new StringSelection(s);
        clipboard.setContents(sselection, sselection);
    }

    void handleError(Throwable th) {
        log.error("", th);
        op.showErrorDialog(th);
    }

    /**
     * j[Cxg̏B
     * @param item j[
     * @param source MR|[lg
     */
    void performMenuEvent(Item item, Component source) {
        try {
            resultSetTable.editingCanceled(new ChangeEvent(source));
        } catch (Exception ex) {
            handleError(ex);
        }
        switch (item) {
            case NEW:
                invoke();
                break;
            case CLOSE:
                requestClose();
                break;
            case QUIT:
                requestExit();
                break;
            case CUT:
            case COPY:
            case PASTE:
            case SELECT_ALL: {
                final KeyStroke accelerator = ((JMenuItem)source).getAccelerator();
                ActionListener action = focused.getActionForKeyStroke(accelerator);
                if (action != null) {
                    action.actionPerformed(new ActionEvent(focused, 1001, ""));
                }
            }
                break;
            case FIND:
                textSearchPanel.setCurrentTarget(textSearchMap.get(focused));
                textSearchPanel.setVisible(true);
                break;
            case TOGGLE_FOCUS:
                if (textArea.isFocusOwner()) {
                    resultSetTable.requestFocus();
                } else {
                    textArea.requestFocus();
                }
                break;
            case CLEAR_MESSAGE:
                textArea.clear();
                executeCommand(EMPTY_STRING);
                break;
            case STATUS_BAR: {
                statusBar.setVisible(((JCheckBoxMenuItem)source).isSelected());
            }
                break;
            case COLUMN_NUMBER:
                resultSetTable.setShowColumnNumber(((JCheckBoxMenuItem)source).isSelected());
                sp.getTopComponent().repaint();
                break;
            case INFO_TREE:
                setInfoTreePane(((JCheckBoxMenuItem)source).isSelected());
                op.validate();
                op.repaint();
                break;
            case ALWAYS_ON_TOP:
                op.setAlwaysOnTop(((JCheckBoxMenuItem)source).isSelected());
                break;
            case WIDEN:
                op.changeTableColumnWidth(1.5f);
                break;
            case NARROW:
                op.changeTableColumnWidth(1 / 1.5f);
                break;
            case ADJUST_COLUMN_WIDTH:
                op.adjustTableColumnWidth();
                break;
            case AUTO_ADJUST_MODE_NONE:
            case AUTO_ADJUST_MODE_HEADER:
            case AUTO_ADJUST_MODE_VALUE:
            case AUTO_ADJUST_MODE_HEADERANDVALUE:
                op.setAutoAdjustMode(item);
                break;
            case REFRESH: {
                final String s = resultSetTable.getResultSetTableModel().getCommandString();
                if (s != null && s.length() > 0) {
                    executeCommand(s);
                }
            }
                break;
            case EXECUTE:
                executeCommand(textArea.getEditableText());
                break;
            case BREAK:
                // FIXME fenv̂ւȂƎłȂ
                env.getOutputProcessor().close();
                env.setOutputProcessor(new WindowOutputProcessor.Bypass(op));
                op.output(getString("i.cancelled"));
                doPostProcess();
                break;
            case HISTORY_BACK:
                if (historyList.isEmpty()) {
                    return;
                }
                --historyIndex;
                if (historyIndex < 0) {
                    historyIndex = historyList.size() - 1;
                }
                textArea.replace(historyList.get(historyIndex));
                textArea.requestFocus();
                break;
            case HISTORY_NEXT:
                if (historyList.isEmpty()) {
                    return;
                }
                ++historyIndex;
                if (historyIndex >= historyList.size()) {
                    historyIndex = 0;
                }
                textArea.replace(historyList.get(historyIndex));
                textArea.requestFocus();
                break;
            case CONNECT: {
                if (env.getConnectorMap().isEmpty()) {
                    showMessageDialog(op, getString("w.no-connector"));
                    return;
                }
                Object[] a = ConnectorEntry.toList(env.getConnectorMap().values()).toArray();
                final String m = getString("dialog.choosingconnection");
                Object value = showInputDialog(op, m, null, PLAIN_MESSAGE, null, a, a[0]);
                if (value != null) {
                    ConnectorEntry c = (ConnectorEntry)value;
                    executeCommand("connect " + c.getId());
                }
            }
                break;
            case DISCONNECT:
                executeCommand("disconnect");
                break;
            case ENCRYPTION_KEY: {
                JPasswordField password = new JPasswordField(20);
                Object[] a = {getString("WindowOutputProcessor.encryption.message"), password};
                if (showConfirmDialog(op, a, null, OK_CANCEL_OPTION) == OK_OPTION) {
                    CipherPassword.setSecretKey(String.valueOf(password.getPassword()));
                }
            }
                break;
            case EDIT_CONNECTORS: {
                env.updateConnectorMap();
                if (env.getCurrentConnector() != null) {
                    showMessageDialog(op, getString("i.reconnect-after-edited-current-connector"));
                }
                ConnectorMapEditDialog dialog = new ConnectorMapEditDialog(op, env);
                dialog.pack();
                dialog.setModal(true);
                dialog.setLocationRelativeTo(op);
                dialog.setVisible(true);
                env.updateConnectorMap();
            }
                break;
            case SORT:
                op.sortTable(resultSetTable.getSelectedColumn());
                break;
            case EXPORT:
                try {
                    op.exportTableContent();
                } catch (IOException ex) {
                    throw new RuntimeException(ex);
                }
                break;
            case HELP:
                showMessageDialog(op, getString("i.help-see-release-package"), null, PLAIN_MESSAGE);
                break;
            case ABOUT:
                ImageIcon icon = new ImageIcon();
                if (op.getIconImage() != null) {
                    icon.setImage(op.getIconImage());
                }
                showMessageDialog(op, getString(".about", getVersion()), null, PLAIN_MESSAGE, icon);
                break;
            default:
                log.warn(String.format("ignored [%s] (%s)", item, source));
        }
    }

    /**
     * NB 
     */
    static void invoke() {
        WindowLauncher instance = new WindowLauncher();
        instance.op.setVisible(true);
        instance.textArea.requestFocus();
        // JX^L[oCh̔f
        try {
            File keyBindConf = new File(LocalSystem.getDirectory(), "keybind.conf");
            if (keyBindConf.exists()) {
                List<String> a = new ArrayList<String>();
                Scanner scanner = new Scanner(keyBindConf);
                try {
                    while (scanner.hasNextLine()) {
                        a.add(scanner.nextLine());
                    }
                } finally {
                    scanner.close();
                }
                Menu.changeKeyBinds(a);
            }
        } catch (Exception ex) {
            instance.handleError(ex);
        }
    }

    /**
     * IB
     */
    static void exit() {
        for (WindowLauncher instance : new ArrayList<WindowLauncher>(instances)) {
            try {
                instance.close();
            } catch (Exception ex) {
                log.warn("error occurred when closing all instances", ex);
            }
        }
    }

    /**
     * LauncherB
     */
    void close() {
        instances.remove(this);
        try {
            env.release();
            saveConfiguration();
            executorService.shutdown();
        } finally {
            op.dispose();
        }
    }

    /**
     * YES-NOIvV̊mF_CAO\YESꂽǂ𒲂ׂB
     * @param message bZ[W
     * @return YESꂽǂ
     */
    private boolean isConfirmYes(String message) {
        return showConfirmDialog(op, message, "", YES_NO_OPTION) == YES_OPTION;
    }

    /**
     * vB
     */
    void requestClose() {
        if (instances.size() == 1) {
            requestExit();
        } else if (env.getCurrentConnection() == null || isConfirmYes(getString("i.confirm-close"))) {
            close();
        }
    }

    /**
     * IvB
     */
    private void requestExit() {
        if (isConfirmYes(getString("i.confirm-quit"))) {
            exit();
        }
    }

    /**
     * c[yC̕\/\ݒ肷B
     * @param show \/\
     */
    void setInfoTreePane(boolean show) {
        if (show) {
            infoTree.setEnabled(true);
            sp0.setEnabled(true);
            sp0.setDividerSize(4);
            sp0.setResizeWeight(0.6f);
            sp0.setDividerLocation(0.3f);
        } else {
            sp0.setDividerSize(0);
            sp0.setResizeWeight(0.0f);
            sp0.setDividerLocation(0.0f);
            final DefaultTreeModel model = (DefaultTreeModel)infoTree.getModel();
            model.setRoot(null);
            infoTree.setEnabled(false);
            sp0.setEnabled(false);
        }
    }

    /**
     * R}hsB
     * @param commandString R}h
     */
    void executeCommand(String commandString) {
        assert commandString != null;
        textArea.replace(String.format("%s%n", commandString));
        if (commandString.trim().length() == 0) {
            doPostProcess();
        } else {
            final String cmd = commandString;
            final Environment env = this.env;
            final WindowOutputProcessor op = this.op;
            final DatabaseInfoTree infoTree = this.infoTree;
            final JLabel statusBar = this.statusBar;
            final OutputProcessor opref = env.getOutputProcessor();
            try {
                doPreProcess();
                executorService.execute(new Runnable() {

                    public void run() {
                        Connector c = env.getCurrentConnector();
                        op.setCurrentCommand(cmd);
                        long time = System.currentTimeMillis();
                        if (!Command.invoke(env, cmd)) {
                            exit();
                        }
                        if (infoTree.isEnabled()) {
                            if (env.getCurrentConnector() != c) {
                                try {
                                    infoTree.refreshRoot(env);
                                } catch (Throwable th) {
                                    handleError(th);
                                }
                            }
                        }
                        if (env.getOutputProcessor() == opref) {
                            time = System.currentTimeMillis() - time;
                            statusBar.setText(getString("i.statusbar-message", time / 1000f, cmd));
                            doPostProcess();
                        }
                    }

                });
            } catch (Exception ex) {
                throw new RuntimeException(ex);
            }
            if (historyList.contains(commandString)) {
                historyList.remove(commandString);
            }
            historyList.add(commandString);
            historyIndex = historyList.size();
        }
    }

    void doPreProcess() {
        ((Menu)op.getJMenuBar()).setEnabledStates(true);
        resultSetTable.getParent().setEnabled(false);
        textArea.setEnabled(false);
        op.repaint();
    }

    void doPostProcess() {
        ((Menu)op.getJMenuBar()).setEnabledStates(false);
        resultSetTable.setEnabled(true);
        textArea.setEnabled(true);
        op.output(new Prompt(env));
        textArea.requestFocus();
    }

    static void wakeup() {
        for (WindowLauncher instance : new ArrayList<WindowLauncher>(instances)) {
            try {
                instance.op.repaint();
            } catch (Exception ex) {
                log.warn("", ex);
            }
        }
        if (log.isInfoEnabled()) {
            log.info("wake up");
        }
    }


    /**
     * Gg|CgB
     * @param args
     */
    public static void main(String... args) {
        final int residentCycle = LocalSystem.getPropertyAsInt("net.argius.stew.ui.window.resident");
        if (residentCycle > 0) {
            final long msec = residentCycle * 60000L;
            Timer timer = new Timer(true);
            timer.scheduleAtFixedRate(new TimerTask() {

                @Override
                public void run() {
                    EventQueue.invokeLater(new Runnable() {

                        public void run() {
                            wakeup();
                        }

                    });
                }

            }, msec, msec);
        }
        EventQueue.invokeLater(new Runnable() {

            public void run() {
                invoke();
            }

        });
    }

}
