/*
 * ssLog
 *
 * Copyright (c) 2008 benishouga (https://sourceforge.jp/projects/sslib)
 * licensed under the MIT
 */
if(typeof(sslib) == "undefined") { sslib = {}; } (function() { with(sslib) {

// ユーティリティ
var utilVersion = 0.1;
if(!sslib.Util || sslib.Util.version < utilVersion) {
	sslib.Util = {};
	/**
	 * @class ユーティリティクラス
	 */
	Util.version = utilVersion;

	/* ブラウザ判定 */
	/** IEの判定 */
	Util.isIE = !!(window.attachEvent && !window.opera);
	/** Firefoxの判定 */
	Util.isGecko = navigator.userAgent.indexOf('Gecko') > -1 && navigator.userAgent.indexOf('KHTML') == -1;

	/** DocumentBody構築後実行するfunctionリスト */
	Util._observers = [];

	/**
	 * イベント登録
	 * @param elem イベントを登録するエレメント
	 * @param type イベントタイプ exp) "click" "load"
	 * @param func イベント発生時に実行するfunction
	 * @return イベント登録時のハンドラ（Util.removeEventの引数として渡すことでイベント消去が可能）
	 */
	Util.addEvent = function( elem, type, func ) {
		if(elem.addEventListener) {
			elem.addEventListener(type, func, false);
		} else if(elem.attachEvent) {
			elem.attachEvent('on'+type, func);
		} else {
			return false;
		}
		return { element: elem, method : func, name : type };
	};


	/**
	 * イベント削除
	 * @param handler イベントのハンドラ（イベント登録時の戻り値）
	 */
	Util.removeEvent = function(handler) {
		if(handler.element.detachEvent) {
			handler.element.detachEvent('on' + handler.name, handler.method);
		}else{
			handler.element.removeEventListener(handler.name, handler.method, false);
		}
	}


	/**
	 * DOM構築完了イベントにFunctionを登録します。
	 * （document.body定期監視なので少し時間差あり）
	 * @param func 実行するfunction
	 */
	Util.onDocumentBodyLoad = function(func) {
		if( document.body ) {
			// 既にdocument.body構築後の場合、即実行
			func();
		} else {
			Util._observers.push(func);
		}
	};

	/**
	 * document.bodyを監視し、読み込みが完了していた場合、登録されているイベントをすべて実行します。
	 */
	Util._checkDocumentBodyLoad = function() {
		if(document.body) {
			// 2度実行されないためクリア
			clearInterval(Util._documentBodyLoadInterval);
			
			// 登録されているFunctionを全て登録順に実行
			for(var i = 0; i < Util._observers.length; i++) {
				Util._observers[i]();
			}
			// 使い終わったら消去
			Util._observers = null;
		}
	};
	// document.bodyの構築を監視
	Util._documentBodyLoadInterval = setInterval(Util._checkDocumentBodyLoad, 100);

	/**
	 * 数値左0埋め
	 * @param num 数値
	 * @param size 桁数
	 * @return 0埋めされ、指定桁数になった数値
	 */
	Util.fillLeft = function(num, size) {
		var str = "" + num;
		while(str.length < size) {
			str = "0" + str;
		}
		return str;
	};

	/**
	 * 数値右0埋め
	 * @param num 数値
	 * @param size 桁数
	 * @return 0埋めされ、指定桁数になった数値
	 */
	Util.fillRight = function(num, size) {
		var str = "" + num;
		while(str.length < size) {
			str = str + "0";
		}
		return str;
	};

	/**
	 * 文字列を整形します。
	 * 引数は可変長引数として扱われます。
	 * exp) Util.format("a{0}c{1}e{2}", "B", "D", "F"); // 結果：aBcDeF
	 * @param msg メッセージフォーマット
	 * @param param0 {0}に代入される値
	 * @param param1 {1}に代入される値
	 * @return フォーマットされた値
	 */
	Util.format = function(msg, param0, param1) {
		if( arguments.length == 0 ) {
			return "";
		}
		var result = arguments[0];
		for(var i = 1; i < arguments.length; i++) {
		result = Util.replaceAll(result, "{" + (i - 1) + "}", arguments[i]);
		}
		return result;
	};

	/**
	 * 文字列置換
	 * @param text 元の文
	 * @param sText 置換対象文字列
	 * @param rText 置換後文字列
	 */
	Util.replaceAll = function( text, sText, rText ){
		var str = "";
		str = "" + text; // 文字列に変換
		str = str.split( sText );
		str = str.join( rText );
		return str;
	};

	/**
	 * プランを元にTableエレメントを作成します。
	 * @param doc 作成親となるDocumentオブジェクト
	 * @param tablePlan Table設計
	 * @return tableエレメント
	 */
	Util.createTableElement = function(doc, tablePlan){
		// テーブル作成
		var table = doc.createElement("table");
		Util.setAttributes(table, tablePlan);
		var tbody = doc.createElement("tbody");
		table.appendChild(tbody);
		
		for(var i = 0; i < tablePlan.length; i++) {
			// 行作成
			var rowPlan = tablePlan[i];
			var tr = doc.createElement("tr");
			tbody.appendChild(tr);
			Util.setAttributes(tr, rowPlan);
			
			for(var j = 0; j < rowPlan.length; j++) {
				// セル作成
				var cellPlan = rowPlan[j];
				var td = doc.createElement("td");
				tr.appendChild(td);
				Util.setAttributes(td, cellPlan);
				
				if(!cellPlan.element) {
					continue;
				} else if(typeof(cellPlan.element) == "string") {
					td.appendChild(doc.createTextNode(cellPlan.element));
				} else {
					td.appendChild(cellPlan.element);
				}
			}
		}
		return table;
	};

	/**
	 * エレメントの付加情報を設定します。
	 * @param element 付加対象のエレメント
	 * @param plan 付加情報
	 */
	Util.setAttributes = function(element, plan) {
		if(plan.attributes) {
			for(var key in plan.attributes) {
				element[key] = plan.attributes[key];
			}
		}
		
		Util.setStyle(element, plan.style);
	};

	/**
	 * 指定したNodeに子Nodeを追加します。
	 * @param element 親となるNode
	 * @param tag 追加する子Nodeのタグ名
	 * @param style 追加する子Elementのstyle情報
	 * @return 作成された子Node
	 */
	Util.addChildNode = function(element, tag, style) {
		var child = element.ownerDocument.createElement(tag);
		element.appendChild(child);
		Util.setStyle(child, style);
		return child;
	};

	/**
	 * 指定したNodeにTextNode
	 * @param element 親となるNode
	 * @param text 追加するTextNodeの文字列
	 * @param style 追加するTextNodeのstyle情報
	 * @return 作成されたTextNode
	 */
	Util.addTextNode = function(element, text, style) {
		var doc = element.ownerDocument;
		var span = doc.createElement("span");
		element.appendChild(span);
		span.appendChild(doc.createTextNode(text));
		Util.setStyle(span, style);
		return span;
	};

	/**
	 * 指定したNodeの子Elementをすべて削除します。
	 * @param element 子Elementを削除するNode
	 */
	Util.clearChildNode = function(element) {
		while(element.childNodes.length != 0) {
			element.removeChild(element.firstChild);
		}
	};

	/**
	 * 指定したElementにstyleを適用します。
	 * @param element styleが付与されるエレメント
	 * @param style style情報
	 */
	Util.setStyle = function(element, style) {
		if( style ) {
			if(!element.style) {
				element.style = {};
			}
			for(var prop in style) {
				element.style[prop] = style[prop];
			}
		}
	};
} // ユーティリティ


/**
 * @class ログクラス管理クラス
 */
sslib.LogManager = {};

/** デフォルトのLogPlaceId */
LogManager._DEFAULT_ID = "__log_place";

/** ログインスタンスのインスタンスプール */
LogManager._pool = [];

/**
 * ログクラスのインスタンス取得
 * @param id 取得するログインスタンスのID
 */
LogManager.getInstance = function(id) {// TODO:Configを渡して使うように？　AppenderとLayout
	if(!id) {
		id = LogManager._DEFAULT_ID;
	}
	if(!LogManager._pool[id]) {
		LogManager._pool[id] = new Log(id);
	}
	return LogManager._pool[id];
};


/**
 * コンストラクタ
 * @class ログクラス
 */
sslib.Log = function(id) {
	this.id = id;
	this.initialized = false;
	this.level = Log.DEBUG;
	this.iframe = null;
	this.appender = new LayerAppender(this.id);
};

/** ログレベル */
Log.ALL   = { value : Number.MIN_VALUE, label : "" };
Log.TRACE = { value : 100, label : "TRACE" };
Log.DEBUG = { value : 200, label : "DEBUG" };
Log.INFO  = { value : 300, label : "INFO " };
Log.WARN  = { value : 400, label : "WARN " };
Log.ERROR = { value : 500, label : "ERROR" };
Log.FATAL = { value : 600, label : "FATAL" };
Log.OFF   = { value : Number.MAX_VALUE, label : "" };

/** ログレベルに対応する文字列を取得 */
Log.getLogLevelString = function(level) {
	return level.label;
};

/** 出力するログのレベルを設定する */
Log.prototype.setLevel = function(level) {
	this.level = level;
};

/* ログを出力系 */
Log.prototype.trace = function() { this._writeLog(Log.TRACE, arguments); };
Log.prototype.debug = function() { this._writeLog(Log.DEBUG, arguments); };
Log.prototype.info  = function() { this._writeLog(Log.INFO,  arguments); };
Log.prototype.warn  = function() { this._writeLog(Log.WARN,  arguments); };
Log.prototype.error = function() { this._writeLog(Log.ERROR, arguments); };
Log.prototype.fatal = function() { this._writeLog(Log.FATAL, arguments); };

/** 対象ログレベルが有効か判定する */
Log.prototype.isEnabled = function(level) {
	return this.level.value <= level.value;
};
Log.prototype.isTraceEnabled = function() { return this.isEnabled(Log.TRACE); };
Log.prototype.isDebugEnabled = function() { return this.isEnabled(Log.DEBUG); };
Log.prototype.isInfoEnabled  = function() { return this.isEnabled(Log.INFO);  };
Log.prototype.isWarnEnabled  = function() { return this.isEnabled(Log.WARN);  };
Log.prototype.isErrorEnabled = function() { return this.isEnabled(Log.ERROR); };
Log.prototype.isFatalEnabled = function() { return this.isEnabled(Log.FATAL); };

/** ログを出力する */
Log.prototype._writeLog = function(argLevel, argArray) {
	// 表示レベルが足りない場合は即終了
	if(!this.isEnabled(argLevel)) {
		return;
	}
	
	// ログを書き出すためのデータを作成
	var logInfo = { level : argLevel, list : argArray, time : new Date() };
	
	// Appenderにログを出力してもらう
	this.appender.append(logInfo);
};


sslib.Appender = function() {
	this.logHeaderFormat = Appender.LOG_HEADER_FORMAT;
	this.dateTimeFormat = Appender.TIME_STRING_FORMAT;
};

/** エラーメッセージフォーマット */
Appender.ERROR_MESSAGE = 'Error on line {0} of document {1} : {2}';

/** ログヘッダフォーマット */
Appender.LOG_HEADER_FORMAT = '{0} [{1}] : ';

/** 時間文字列フォーマット */
Appender.DATE_TIME_STRING_FORMAT = '{0}-{1}-{2} {3}:{4}:{5}.{6}';
Appender.TIME_STRING_FORMAT = '{3}:{4}:{5}.{6}';

Appender.prototype.append = function() {
	throw new Error("This is abstruct method.");
};


sslib.DivAppender = function() {
	Appender.apply(this, arguments);
	this.initialized = false;
	this.buffer = [];
	this.dumper = null;
	this.iframe = null;
};

DivAppender.prototype = new Appender;

/** 表示Font */
DivAppender.FONT = '"MS Gothic","Osaka-Mono",monospace';

DivAppender.prototype._initialize = function() {
	throw new Error("This is abstruct method.");
};


DivAppender.prototype.append = function(logInfo) {
	// 初期化が行われていない場合は、初期化を行う
	if(!this.initialized) {
		this._initialize();
	}
	
	// 既にログ出力先が存在すれば出力、存在しない場合bufferに追加
	if(this.logDiv) {
		this._appendLog(logInfo);
	} else {
		this.buffer.push(logInfo);
	}
};


/** ログを出力する */
DivAppender.prototype._appendLog = function(logInfo) {
	var doc = this.logDiv.ownerDocument;
	var p = doc.createElement("p");
	this.logDiv.insertBefore(p, this.logDiv.firstChild);
	
	var color = "white";
	if( logInfo.level.value >= Log.ERROR.value ) {
		color = "red";
	} else if( logInfo.level.value >= Log.WARN.value ) {
		color = "orange";
	} else if( logInfo.level.value <= Log.TRACE.value ) {
		color = "#bbb";
	}
	
	Util.setStyle(p, {
		padding : "0px",
		margin : "0px",
		borderBottom : "solid 1px #999",
		fontSize : "11px",
		backgroundColor : "transparent",
		zIndex : 3000,
		paddingLeft : "6ex",
		textIndent : "-6ex",
		color : color
	});
	var time = logInfo.time;
	
	// 時間文字列作成
	var timeString = Util.format(
		this.dateTimeFormat,
		Util.fillLeft(time.getFullYear(), 4),
		Util.fillLeft(time.getMonth(), 2),
		Util.fillLeft(time.getDate(), 2),
		Util.fillLeft(time.getHours(), 2),
		Util.fillLeft(time.getMinutes(), 2),
		Util.fillLeft(time.getSeconds(), 2),
		Util.fillRight(time.getMilliseconds(), 3));
	// ログヘッダ作成
	var logHeader = Util.format(this.logHeaderFormat, Log.getLogLevelString(logInfo.level), timeString);
	// ログヘッダのテキストノードを作成して、追加
	var logHeaderNode = doc.createTextNode(logHeader);
	p.appendChild(logHeaderNode);
	
	// データを出力
	var first = true;
	for(var i = 0; i < logInfo.list.length; i++) {
		if(!first) {
			p.appendChild(doc.createTextNode(" "));
		} else {
			first = false;
		}
		
		if(!this.dumper) {
			this.dumper = new Dumper();
		}
		this.dumper.dump(p, logInfo.list[i]);
	}
};

/** バッファをログ出力する */
DivAppender.prototype._flush = function() {
	for(var i = 0; i < this.buffer.length; i++) {
		this._appendLog(this.buffer[i]);
	}
	this.buffer.length = 0;
};


sslib.LogWindowAppender = function(id) {
	this.id = id;
	this.window = null;
	this.logDiv = null;
};

LogWindowAppender.prototype = new DivAppender;

LogWindowAppender.prototype._initialize = function() {
	/** ログ出力場所を作成するFunctionを返す */
	var __this = this;
	
	var callCreateLogPlace = function () {
		var windowId = "Log" + __this.id;
		__this.window = window.open("", windowId);
		var doc = __this.window.document;
		doc.open("text/html", "replace");
		doc.write('<html><head><title>Log' + __this.id + '</title></head><body style="margin:0px;background-color:black;"></body></html>');
		doc.close();
		__this.logDiv = doc.body;
		
		// bufferを出力する
		__this._flush();
	};
	
	// ウィンドウの再オープン処理を割り込ましておく
	var prevAppendLog = this._appendLog;
	this._appendLog = function() {
		if(__this.window.closed) {
			callCreateLogPlace();
		}
		prevAppendLog.apply(this, arguments);
	};
	
	if(!document.body) {
		Util.onDocumentBodyLoad(callCreateLogPlace);
	} else {
		callCreateLogPlace();
	}
	this.initialized = true;
};

sslib.LayerAppender = function(id) {
	this.id = id;
	this.position = LayerAppender.POSITION_TOP;
	this.height = LayerAppender.HEIGHT;
};


LayerAppender.prototype = new DivAppender;

/** 上に配置 */
LayerAppender.POSITION_TOP = "top";
/** 下に配置 */
LayerAppender.POSITION_BOTTOM = "bottom";

LayerAppender.BUTTON_TABLE_STYLE = {
	backgroundColor : "white",
	cursor : "pointer",
	width : "100%",
	height : "13px",
	border : "solid 1px white",
	borderCollapse : "collapse"
};

/** ログ出力場所のマージン */
LayerAppender.MARGIN = 5;

/** ログ出力場所の高さ */
LayerAppender.HEIGHT = 160;

/** 初期化を行う */
LayerAppender.prototype._initialize = function() {
	/** ログ出力場所を作成するFunctionを返す */
	var __this = this;
	var callCreateLogPlace = function () {
		__this._createLogPlace();
	};

	if(!document.body) {
		Util.onDocumentBodyLoad(callCreateLogPlace);
	} else {
		callCreateLogPlace();
	}
	this.initialized = true;
};

/** 表示位置を調整する */
LayerAppender.prototype.fixScrollTop = function() {
	if(!this.iframe) {
		return;
	}
	
	// bodyのスクロール量を取得
	var y = window.pageYOffset || document.body.scrollTop || document.documentElement.scrollTop;
	
	// 各値によって、表示位置を調整
	if(this.position == LayerAppender.POSITION_BOTTOM) {
		// window表示域の高さを足す
		y += self.innerHeight || document.documentElement.clientHeight || document.body.clientHeight;
		
		// Log表示域の高さを引く
		y -= this.height;
		
		// margin分を引く
		y -= LayerAppender.MARGIN * 2;
		
		// TODO:横スクロールバーの幅を考慮（ちゃんと調整してあげてもいいかも
		y -= 15;
	}
	
	// 必死に設定してみるけど、できないこともあるっぽい？
	this.iframe.style.top = y;
};

/** 表示位置を設定する */
LayerAppender.prototype.setPosition = function(position) {
	this.position = position;
	this.fixScrollTop();
};

LayerAppender.prototype.setSize = function(height) {
	this.height = height;
	this.fixSize();
};

LayerAppender.prototype.fixSize = function() {
	if(this.iframe) {
		this.iframe.style.height = this.height;
	}
};

LayerAppender.prototype.dragStart = function() {
	this.dragFlag = true;
	
	var hoge = this.iframe.offsetHeight;
	this.offset = parseInt(this.iframe.offsetHeight);
	this.funcAreavY = this.nowY;
};

LayerAppender.prototype.dragEnd = function() {
	this.dragFlag = false;
};

LayerAppender.prototype.dragProc = function(evt) {
	this.nowY = evt.screenY;
	
	if(this.dragFlag) {
		var targetSize = this.offset + (this.nowY - this.funcAreavY);
		
		// 最低サイズ
		if( targetSize < 50 ) {
			targetSize = 50;
		}
		
		// 最高サイズ
		var screenHeight = self.innerHeight || document.documentElement.clientHeight || document.body.clientHeight;
		if(screenHeight * 0.9 < targetSize) {
			targetSize = screenHeight * 0.9;
		}
		
		this.setSize(targetSize);
		this.fixSize();
		this.fixScrollTop();
	}
};



/** ログ出力場所を作成する */
LayerAppender.prototype._createLogPlace = function() {
	var __this = this;
	
	// 一番親となるIFrameを作成する
	var iframe = document.createElement("iframe");
	iframe.id = this.id;
	iframe.handler = this;
	iframe.src = "javascript:false;";
	
	Util.setAttributes(iframe, {
		attributes : {
			allowTransparency : "true",
			frameBorder : 0
		},
		style : {
			position : "absolute",
			margin : LayerAppender.MARGIN + "px",
			width : "98%",
			height : this.height + "px",
			border : 0,
			zIndex : 100
		}
	});
	
	// IFrameをBodyに追加
	document.body.insertBefore(iframe, document.body.firstChild);
	
	// IFrameの中身を作成
	var doc = this.ownerDocument = iframe.contentDocument ? iframe.contentDocument : document.frames[iframe.id].document;
	
	doc.open();
	doc.write('<html><head><title>Log:' + this.id + '</title></head><body style="margin:0px;background-color:transparent;"></body></html>');
	doc.close();
	
	// メインのDiv
	var logBaseDiv = Util.addChildNode(doc.body, "div", {
		position : "absolute",
		height : "100%",
		width : "100%",
		fontFamily : DivAppender.FONT,
		zIndex : 200
	});
	
	// 操作ボタンを置くDiv
	var controllerDiv = Util.addChildNode(logBaseDiv, "div", {
		width : "30px",
		height : "100%",
		styleFloat : "right",
		cssFloat : "right",
		overflow : "hidden",
		textAlign: "center",
		zIndex : 250
	});
	// ボタンコントロールの作成
	this._setupContollerDiv(controllerDiv);
	
	// Logの出力先のDiv
	this.logDiv = Util.addChildNode(logBaseDiv, "div", {
		overflow : "auto",
		height : "100%",
		color : "white",
		zIndex : 250
	});
	
	// 背景となるDivを作成
	var opacityValue = 0.6;
	Util.addChildNode(doc.body, "div", {
		filter : 'alpha(opacity=' + (opacityValue * 100) + ')',
		MozOpacity : opacityValue,
		opacity : opacityValue,
		position : "absolute",
		backgroundColor : "black",
		height : "100%",
		width : "100%",
		zIndex : 150
	});
	
	
	// インターバルを使い、表示位置を調整するメソッドを呼ぶ
	setInterval(function(){ __this.fixScrollTop(); },200);
	
	// 完成したら位置やサイズを調整する
	this.fixScrollTop();
	this.fixSize();
	
	this.iframe = iframe;
	
	// bufferを出力する
	this._flush();
};

LayerAppender.prototype._setupContollerDiv = function(controllerDiv) {
	var doc = this.ownerDocument;
	
	var td11 = {style : {width : "50%"}};
	td11.element = this._createTopPositionButton();
	
	var td12 = {style : {width : "50%"}};
	td12.element = this._createCloseButton();
	
	var row1Plan = [td11, td12];
	row1Plan.style = { height : "15px" };
	
	var centerTd = { element : "", attributes : { colSpan : 2 } };
	var row2Plan = [centerTd];
	
	var td31 = {style : {width : "50%"}};
	td31.element = this._createBottomPositionButton();
	
	var td32 = {style : {width : "50%"}};
	td32.element = this._createResizeButton();
	
	var row3Plan = [td31, td32];
	row3Plan.style = { height : "15px" };
	
	var tablePlan = [row1Plan, row2Plan, row3Plan];
	tablePlan.style = {
		width : "100%",
		height : "100%",
		textAlign : "center",
		verticalAlign : "middle",
		border : "solid 0px",
		borderCollapse : "collapse"
	};
	
	var table = Util.createTableElement(doc, tablePlan);
	
	controllerDiv.appendChild(table);
}


LayerAppender.prototype._createTopPositionButton = function() {
	var __this = this;
	var row1Plan = [{}];
	row1Plan.style = {};
	row1Plan.style.height = "2px";
	row1Plan.style.backgroundColor = "gray";
	
	var row2Plan = [{}];
	
	var tablePlan = [row1Plan, row2Plan];
	Util.setStyle(tablePlan, LayerAppender.BUTTON_TABLE_STYLE);
	tablePlan.attributes = {};
	tablePlan.attributes.title = "set position top";
	
	var table = Util.createTableElement(this.ownerDocument, tablePlan);
	
	Util.addEvent(table, "click", function(){__this.setPosition(LayerAppender.POSITION_TOP);});
	
	return table;
};

LayerAppender.prototype._createBottomPositionButton = function() {
	var __this = this;
	var row1Plan = [{}];
	
	var row2Plan = [{}];
	row2Plan.style = {};
	row2Plan.style.height = "2px";
	row2Plan.style.backgroundColor = "gray";
	
	var tablePlan = [row1Plan, row2Plan];
	Util.setStyle(tablePlan, LayerAppender.BUTTON_TABLE_STYLE);
	tablePlan.attributes = {};
	tablePlan.attributes.title = "set position bottom";
	
	var table = Util.createTableElement(this.ownerDocument, tablePlan);
	
	Util.addEvent(table, "click", function(){__this.setPosition(LayerAppender.POSITION_BOTTOM);});
	
	return table;
};


LayerAppender.prototype._createCloseButton = function() {
//	var __this = this;
//	var row1Plan = [{}];
//	
//	var row2Plan = [{}];
//	row2Plan.style = {};
//	row2Plan.style.height = "1px";
//	row2Plan.style.backgroundColor = "gray";
//	
//	var row3Plan = [{}];
//	
//	var tablePlan = [row1Plan, row2Plan, row3Plan];
	
	var tablePlan = [];
	Util.setStyle(tablePlan, LayerAppender.BUTTON_TABLE_STYLE);
	tablePlan.attributes = {};
	tablePlan.attributes.title = "close";
	
	var table = Util.createTableElement(this.ownerDocument, tablePlan);
	
//	Util.addEvent(table, "click", function(){alert("close button onclick")});
	
	return table;
};

LayerAppender.prototype._createResizeButton = function() {
	var __this = this;
	
	var tablePlan = [];
	var cellMax = 4;
	for(var i = 0; i < cellMax; i++) {
		var rowPlan = [];
		for(var j = 0; j < cellMax; j++) {
			var cell = {};
			cell.style = {};
			cell.style.height = (100 / cellMax) + "%";
			cell.style.width = (100 / cellMax) + "%";
			if(cellMax - 1 < j + i) {
				cell.style.backgroundColor = "gray";
			}
			rowPlan.push(cell);
		}
		tablePlan.push(rowPlan);
	}
	
	Util.setStyle(tablePlan, LayerAppender.BUTTON_TABLE_STYLE);
	tablePlan.style.cursor = "se-resize";
	
	tablePlan.attributes = {};
	tablePlan.attributes.title = "resize";
	var table = Util.createTableElement(this.ownerDocument, tablePlan);
	
	Util.addEvent(table, "mousedown", function(){ __this.dragStart(); });
	Util.addEvent(this.ownerDocument, "mouseup", function(){ __this.dragEnd(); });
	Util.addEvent(this.ownerDocument, "mousemove", function(evt) { __this.dragProc(evt); });
	
	return table;
};

/** データを見やすく変換する何か */
sslib.Dumper = function(functionPlace){
	if( typeof(functionPlace) == "undefined" ) {
		this.functionPlace = null;
	} else {
		this.functionPlace = functionPlace;
	}
};

/** デフォルトの展開階層数 */
Dumper.DEFAULT_EXPANDED = 1;

Dumper.LINK_STYLE = {
	cursor:"pointer", 
	textDecoration:"underline"
}

/** データを見やすい形に変換してAppendする */
Dumper.prototype.dump = function(element, obj, expendCount) {
	// 展開深度
	if(typeof(expendCount) == "undefined") {
		expendCount = Dumper.DEFAULT_EXPANDED;
	} else {
		expendCount--;
	}
	
	if(typeof(obj) == "undefined") {
		Util.addTextNode(element, "undefined", { fontWeight : "bold" });
	} else if(typeof(obj) == "unknown") {
		Util.addTextNode(element, "unknown", { fontWeight : "bold" });
	} else if(obj == null) {
		Util.addTextNode(element, "null", { fontWeight : "bold" });
	} else if(typeof(obj) == "boolean") {
		Util.addTextNode(element, obj, { color : "violet" });
	} else if(typeof(obj) == "string"){
		Util.addTextNode(element, '"' + obj + '"', { color : "lightblue" });
	} else if(typeof(obj) == "number") {
		Util.addTextNode(element, obj, { color : "gold" });
	} else if(typeof(obj) == "object") {
		if(obj.constructor === Array) {
			this._dumpArray(element, obj, expendCount);
		} else if(obj.constructor === RegExp) {
			Util.addTextNode(element, obj, { color : "pink" });
		} else {
			this._dumpObject(element, obj, expendCount);
		}
	} else if(typeof(obj) == "function") {
		if(obj.constructor === RegExp) { // RegExpはブラウザによって、こっちに来ちゃう
			Util.addTextNode(element, obj, { color : "pink" });
		} else {
			this._dumpFunction(element, obj);
		}
	}
};

Dumper.prototype._dumpNest = function(element, obj, expendCount) {
	var baseSpan = Util.addChildNode(element, "span");
	
	Util.addTextNode(baseSpan, obj.constructor === Array ? "[ " : "{ ", {color : "lightgreen"});
	
	this._dumpNestInternal(baseSpan, obj, expendCount);
	
	Util.addTextNode(baseSpan, obj.constructor === Array ? " ]" : " }", {color : "lightgreen"});
};

Dumper.prototype._dumpNestInternal = function(element, obj, expendCount) {
	var __this = this;
	/*
	try {
		obj.toString();
	} catch(e) {
		Util.addTextNode(baseSpan, e);
		return;
	}
	*/
	
	var exist = false;
	for(var key in obj) {
		exist = true;
		break;
	}
	
	if(exist) {
		var internal = Util.addChildNode(element, "span");
		
		if(0 < expendCount) {
			var close = Util.addTextNode(internal, "<<", Dumper.LINK_STYLE);
			close.value = obj;
			Util.addTextNode(internal, " ");
			
			var first = true;
			
			for(var key in obj) {
				if(!first) {
					Util.addTextNode(internal, ", ");
				}
				var child = Util.addChildNode(internal, "span");
				
				Util.addTextNode(child, key);
				Util.addTextNode(child, "=");
				
				try {
					// IEの場合、unknownって状態になるobjectがあるらしい
					// そしてそれを引数に指定したらその時点で怒られるらしい
					if(typeof(obj[key]) == "unknown") {
						Util.addTextNode(child, "unknown", { fontWeight : "bold" });
					} else {
						this.dump(child, obj[key], expendCount);
					}
				} catch (e) {
					// FireFoxで参照するだけでエラーになるオブジェクトがあるっぽい
					// どうしようもないのでエラーの内容を結果とする
					Util.addTextNode(child, e, { color : "#E9967A" });
				}
				first = false;
			}
			Util.addEvent(close, "click", function(){
				Util.clearChildNode(internal);
				__this._dumpNestInternal(internal, close.value, 0);
			});
		} else {
			var dot = Util.addTextNode(internal, "...", Dumper.LINK_STYLE);
			dot.value = obj;
			
			Util.addEvent(dot, "click", function(){
				Util.clearChildNode(internal);
				__this._dumpNestInternal(internal, dot.value, 1);
			});
		}
	}
} 


Dumper.prototype._dumpObject = function(element, obj, expendCount) {
	this._dumpNest(element, obj, expendCount);
};

Dumper.prototype._dumpArray = function(element, obj, expendCount) {
	this._dumpNest(element, obj, expendCount);
};

Dumper.prototype._dumpFunction = function(element, obj) {
	var __this = this;
	var doc = element.ownerDocument;
	var span = Util.addChildNode(element, "span", Dumper.LINK_STYLE);
	Util.addTextNode(span, "function()");
	Util.addEvent(span, "click", function(){
		if(__this.functionPlace) {
			__this.functionPlace.goNext(obj);
		} else {
			new Dumper.FunctionPanel(doc).show(obj);
		}
	});
};

/**
 * 
 */
Dumper.FunctionPanel = function(doc) {
	this.ownerDocument = doc;
	this.dumper = new Dumper(this);
	this.parentStack = [];
	this.hideFlag = true;
	this.current = null;
	var __this = this;
	this.functionPlace = this.ownerDocument.createElement("div");
	
	Util.addEvent(this.functionPlace, "mouseover", function() {
		__this.hideFlag = false;
	});
	
	Util.addEvent(this.functionPlace, "mouseout", function() {
		__this.hideFlag = true;
	});
	
	this.mouseDownHandler = Util.addEvent(this.ownerDocument, "mousedown", function() {
		if (__this.hideFlag == true) {
			__this.hide();
		}
	});
	
	this.ownerDocument.body.insertBefore(
		this.functionPlace, this.ownerDocument.body.firstChild);
	
	Util.setStyle(this.functionPlace, {
		position : "absolute",
		zIndex : 1000,
		height : "90%",
		width : "80%",
		top : "1px",
		left : "1px"
	});
	
	var opacityValue = 0.6;
	Util.addChildNode(this.functionPlace, "div", {
		filter : 'alpha(opacity=' + (opacityValue * 100) + ')', // )
		MozOpacity : opacityValue,
		opacity : opacityValue,
		position : "absolute",
		backgroundColor : "black",
		zIndex : 110,
		height : "100%",
		width : "100%",
		top : "1px",
		left : "1px"
	});
	
	var base = Util.addChildNode(this.functionPlace, "div", {
		position : "absolute",
		overflow : "auto",
		color : "white",
		height : "100%",
		width : "100%",
		zIndex : 1200,
		fontFamily : DivAppender.FONT,
		fontSize : "11px",
		padding : "5px"
	});
	
	Util.addChildNode(base, "div");
	
	this.properties = Util.addChildNode(base, "p");
	
	this.funcArea = Util.addChildNode(base, "pre");
};

Dumper.FunctionPanel.prototype.hide = function() {
	if(this.functionPlace) {
		Util.removeEvent(this.mouseDownHandler);
		this.ownerDocument.body.removeChild(this.functionPlace);
		this.mouseDownHandler = null;
		this.functionPlace = null;
	}
};

Dumper.FunctionPanel.prototype.show = function(obj) {
	Util.clearChildNode(this.funcArea);
	Util.clearChildNode(this.properties);
	
	this.current = obj;
	
	if(this.parentStack.length != 0) {
		var __this = this;
		var p = Util.addChildNode(this.properties, "p");
		var parentLink = Util.addChildNode(p, "span", Dumper.LINK_STYLE);
		Util.addTextNode(parentLink, "parent");
		Util.addEvent(parentLink, "click", function(){
			__this.goBack();
		});
	}
	
	// オブジェクトの中身を書き出す
	this.dumper._dumpNest(this.properties, obj, 1);
	
	// functionの内容を書き出す
	Util.addTextNode(this.funcArea, obj);
};

Dumper.FunctionPanel.prototype.goNext = function(obj) {
	this.parentStack.push(this.current);
	this.show(obj);
};

Dumper.FunctionPanel.prototype.goBack = function() {
	this.show(this.parentStack.pop());
};

(function() {
	var _errorLog = LogManager.getInstance();
	// ウィンドウエラーをハンドルする
	if(typeof(console) == "undefined" || typeof(console.log) == "undefined") {
		window.onerror = function(message, url, line) {
			_errorLog.fatal(Util.format(Appender.ERROR_MESSAGE, line, url, message));
			return true;
		};
	} else if(Util.isGecko) {
		window.onerror = function(message, url, line) {
			_errorLog.fatal(Util.format(Appender.ERROR_MESSAGE, line, url, message));
			return false;
		};
	}
})();

}})();

// デフォルトで使えるインスタンスを一つ用意
log = sslib.LogManager.getInstance();



