package net.sqs2.image;

import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.Point2D;

public class ImageTranslator {

	private Point[] masterCorners;
	private Point[] blockCenters;
	private Point[] corners;

	private int masterWidth;
	private int masterHeight;
	private int ox;
	private int oy;
	private int ax;
	private int ay;
	private int bx;
	private int by;
	private int cx;
	private int cy;
	private int dx;
	private int dy;

	public ImageTranslator() {
		super();
	}

	public ImageTranslator(Point[] masterCorners, Point[] imageGuideBlockCenters) {
		this.masterCorners = masterCorners;
		this.blockCenters = imageGuideBlockCenters;
		this.corners = new Point[4];
		parseMaster();
		setCorners();
		setABCD();
	}

	public Point[] getCorners() {
		return this.corners;
	}

	public Point[] getBlockCenters() {
		return this.blockCenters;
	}

	public Point[] getMasterCorners() {
		return this.masterCorners;
	}

	private void parseMaster() {
		this.masterWidth = this.masterCorners[1].x - this.masterCorners[0].x;
		this.masterHeight = this.masterCorners[2].y - this.masterCorners[0].y;
		this.ox = this.masterCorners[0].x;
		this.oy = this.masterCorners[0].y;
	}

	private void setCorners() {
		int cxs = (this.blockCenters[3].x - this.blockCenters[2].x)
				* (this.masterCorners[0].x - this.masterCorners[2].x) / this.masterWidth;
		int cys = (this.blockCenters[3].y - this.blockCenters[2].y)
				* (this.masterCorners[0].x - this.masterCorners[2].x) / this.masterWidth;
		this.corners[0] = this.blockCenters[0];
		this.corners[1] = this.blockCenters[1];
		this.corners[2] = new Point(this.blockCenters[2].x + cxs, this.blockCenters[2].y + cys);
		this.corners[3] = new Point(this.blockCenters[3].x + cxs, this.blockCenters[3].y + cys);
	}

	private void setABCD() {
		this.ax = this.corners[1].x - this.corners[0].x;
		this.ay = this.corners[1].y - this.corners[0].y;
		this.bx = this.corners[2].x - this.corners[0].x;
		this.by = this.corners[2].y - this.corners[0].y;
		this.cx = this.corners[3].x - this.corners[2].x;
		this.cy = this.corners[3].y - this.corners[2].y;
		this.dx = this.corners[3].x - this.corners[1].x;
		this.dy = this.corners[3].y - this.corners[1].y;
	}

	public Point2D getPoint(final float x, final float y, Point2D ret) {
		float s = ((x - this.ox) / this.masterWidth);
		float t = ((y - this.oy) / this.masterHeight);
		float aax = this.corners[0].x + this.ax * s;
		float aay = this.corners[0].y + this.ay * t;
		float bbx = this.corners[0].x + this.bx * s;
		float bby = this.corners[0].y + this.by * t;
		float acx = this.corners[2].x + this.cx * s - aax;
		float acy = this.corners[2].y + this.cy * t - aay;
		float bdx = this.corners[1].x + this.dx * s - bbx;
		float bdy = this.corners[1].y + this.dy * t - bby;

		double b = acy * bdx - acx * bdy;
		if (acx == 0 || b == 0) {
			ret.setLocation(aax, bby);
		} else {
			double q = (acx * (bby - aay) - acy * (bbx - aax)) / b;
			double p = (bbx + bdx * q - aax) / acx;
			ret.setLocation(aax + acx * p, aay + acy * p);
		}
		return ret;
	}

	public Point2D getPoint(final int x, final int y) {
		return this.getPoint((float) x, (float) y, new Point2D.Float());
	}

	public void translateTo(Point2D[] translatedRectCorners, Rectangle rect) {
		translatedRectCorners[0] = getPoint(rect.x, rect.y, translatedRectCorners[0]);
		translatedRectCorners[1] = getPoint(rect.x + rect.width, rect.y, translatedRectCorners[1]);
		translatedRectCorners[2] = getPoint(rect.x, rect.y + rect.height, translatedRectCorners[2]);
		translatedRectCorners[3] = getPoint(rect.x + rect.width, rect.y + rect.height,
				translatedRectCorners[3]);
	}

	public Point2D[] translate(Rectangle rect) {
		Point2D[] translatedRectCorners = new Point2D[4];

		translatedRectCorners[0] = getPoint(rect.x, rect.y, new Point2D.Float());
		translatedRectCorners[1] = getPoint(rect.x + rect.width, rect.y, new Point2D.Float());
		translatedRectCorners[2] = getPoint(rect.x, rect.y + rect.height, new Point2D.Float());
		translatedRectCorners[3] = getPoint(rect.x + rect.width, rect.y + rect.height, new Point2D.Float());
		return translatedRectCorners;
	}

}
