/*
 * 

 ImageFactory.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */
package net.sqs2.image;

import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.awt.image.RenderedImage;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.imageio.ImageIO;

import org.apache.batik.ext.awt.image.codec.FileCacheSeekableStream;
import org.apache.batik.ext.awt.image.codec.tiff.TIFFDecodeParam;
import org.apache.batik.ext.awt.image.codec.tiff.TIFFImageDecoder;
import org.apache.commons.io.output.ByteArrayOutputStream;

public class ImageFactory {

	public static final String ERROR_NOT_PNM = "data format may not be PNM";

	public static final boolean isSupported(final String filename) {
		return ImageUtil.isTiff(filename) || ImageUtil.isPNG(filename) || ImageUtil.isGIF(filename)
				|| ImageUtil.isJPEG(filename) || ImageUtil.isMultipageTiff(filename);
	}

	public static Raster create(final String filename) throws IOException {
		return create(new java.io.File(filename));
	}

	public static BufferedImage createImage(final String filename) throws IOException {
		return createImage(new java.io.File(filename));
	}

	public static final BufferedImage createImage(final File file) throws IOException {
		return createImage(new BufferedInputStream(new FileInputStream(file)), 0, file.getName());
	}

	public static final BufferedImage createImage(final File file, final int index) throws IOException {
		return createImage(new BufferedInputStream(new FileInputStream(file)), index, file.getName());
	}

	public static final int getNumPages(File file) throws IOException {
		String filename = file.getName();
		if (ImageUtil.isTiff(filename)) {
			InputStream stream = null;
			FileCacheSeekableStream seekableStream = null;
			try {
				stream = new BufferedInputStream(new FileInputStream(file));
				seekableStream = new FileCacheSeekableStream(stream);
				TIFFImageDecoder decoder = new TIFFImageDecoder(seekableStream, new TIFFDecodeParam());
				return decoder.getNumPages();
			} catch (IOException ex) {
				ex.printStackTrace();
				throw ex;
			} finally {
				try {
					seekableStream.close();
				} catch (Exception ignore) {
				}
				try {
					stream.close();
				} catch (Exception ignore) {
				}
			}
		} else if (ImageUtil.isSupported(filename)) {
			return 1;
		} else {
			return 0;
		}
	}

	public static final BufferedImage createImage(final byte[] data, final int index, final String suffix) throws IOException {
		return createImage(new ByteArrayInputStream(data), index, suffix);
	}

	public static final BufferedImage createImage(final InputStream stream, final int index, final String suffix) throws IOException {
		boolean isTiff = ImageUtil.isTiff(suffix);
		boolean isMultiPageTiff = ImageUtil.isMultipageTiff(suffix);
		if (isTiff || isMultiPageTiff) {
			FileCacheSeekableStream seekableStream = null;
			try {
				seekableStream = new FileCacheSeekableStream(stream);
				TIFFImageDecoder decoder = new TIFFImageDecoder(seekableStream, new TIFFDecodeParam());
				RenderedImage renderedImage = null;
				if (isMultiPageTiff) {
					renderedImage = decoder.decodeAsRenderedImage(index);
				} else {
					renderedImage = decoder.decodeAsRenderedImage();
				}
				BufferedImage image = new BufferedImage(renderedImage.getColorModel(), renderedImage
						.getData().createCompatibleWritableRaster(), true,
						new java.util.Hashtable<String, Object>());
				image.setData(renderedImage.getData());
				return image;
			} catch (IOException ex) {
				throw ex;
			} catch (Throwable ex) {
				ex.printStackTrace();
				throw new RuntimeException(ex);
			} finally {
				try {
					seekableStream.close();
				} catch (Exception ignore) {
				}
			}
		} else if (ImageUtil.isPBM(suffix)) {
			try {
				return createPBM(stream);
			} finally {
				try {
					stream.close();
				} catch (Exception ignore) {
				}
			}
		} else {
			try {
				return ImageIO.read(stream);
			} finally {
				try {
					stream.close();
				} catch (Exception ignore) {
				}
			}
		}
	}

	public static final Raster create(final File file) throws IOException {
		if (ImageUtil.isTiff(file.getName())) {
			FileCacheSeekableStream stream = null;
			Raster raster = null;
			try {
				stream = new FileCacheSeekableStream(new FileInputStream(file));
				TIFFImageDecoder decoder = new TIFFImageDecoder(stream, new TIFFDecodeParam());
				raster = decoder.decodeAsRaster();
			} finally {
				stream.close();
			}
			return raster;
		} else if (ImageUtil.isPBM(file.getName())) {
			FileInputStream stream = new FileInputStream(file);
			Raster raster = null;
			try {
				raster = createPBM(stream).getData();
			} finally {
				stream.close();
			}
			return raster;
		} else {
			Raster raster = ImageIO.read(file).getData();
			return raster;
		}
	}

	public static final BufferedImage createPBM(final InputStream in) throws IOException {
		return new PBMImageFactory(in).createByPBM();
	}

	public static void writeImage(BufferedImage bufferedImage, String imageType, File imageFile) throws IOException {
		OutputStream out = new BufferedOutputStream(new FileOutputStream(imageFile));
		try {
			ImageIO.write(bufferedImage, imageType, out);
		} finally {
			out.close();
		}
	}

	public static void writeImage(BufferedImage bufferedImage, String imageType, OutputStream out) throws IOException {
		ImageIO.write(bufferedImage, imageType, out);
	}

	public static byte[] writeImage(BufferedImage bufferedImage, String imageType) throws IOException {
		ByteArrayOutputStream out = new ByteArrayOutputStream(1024);
		try {
			ImageIO.write(bufferedImage, imageType, out);
		} finally {
			out.close();
		}
		return out.toByteArray();
	}

}
