/*

 SessionProgressBar.java

 Copyright 2004-2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2006/01/10

 */
package net.sqs2.omr.swing.session;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.io.File;
import java.util.ConcurrentModificationException;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import javax.swing.border.MatteBorder;

import net.sqs2.lang.GroupThreadFactory;
import net.sqs2.omr.model.PageTask;
import net.sqs2.omr.model.PageTaskError;
import net.sqs2.omr.model.TaskNumberCounter;
import net.sqs2.omr.session.model.PageTaskErrorEntry;
import net.sqs2.omr.session.model.PageTaskExecutionProgressModel;
import net.sqs2.omr.session.monitor.MarkReaderSessionMonitorAdapter;

public class PageTaskExecutionProgressBar extends PageTaskExecutionProgressPanel {
	private static final long serialVersionUID = 0L;

	private float indeterminate = -1f;
	private int mousePositionX = -1;

	private transient ExecutorService executor;
	private transient Future<?> indeterminateAnimationFuture = null;

	public PageTaskExecutionProgressBar(PageTaskExecutionProgressModel model) {
		super(model);
		setPreferredSize(new Dimension(300, 20));
		setSize(new Dimension(300, 20));
		setBorder(new MatteBorder(1,1,1,1,Color.gray));
		addMouseMotionListener(new ProgressBarMouseMotionAdapter());
		addMouseListener(new MouseAdapter() {
			@Override
			public void mouseExited(MouseEvent ev) {
				clear();
			}
		});
		addFocusListener(new FocusAdapter() {
			@Override
			public void focusLost(FocusEvent ev) {
				clear();
			}
		});
		this.pageTaskExecutionProgressModel.addSessionMonitor(new MarkReaderSessionMonitorAdapter() {
			@Override
			public void notifySourceInitializeDone(File sourceDirectoryRootFile) {
				//setIndeterminate(true);
			}

			@Override
			public void notifyStoreTask(PageTask pageTask) {
				setIndeterminate(false);
			}
		});
	}

	void clear() {
		PageTaskExecutionProgressBar.this.pageTaskExecutionProgressModel.setStatusMessage("");
		PageTaskExecutionProgressBar.this.mousePositionX = -1;
		update();
	}

	class ProgressBarMouseMotionAdapter implements MouseMotionListener {
		public void mouseDragged(MouseEvent e) {

		}

		public void mouseMoved(MouseEvent e) {
			int x = e.getX();
			int y = e.getY();
			Dimension size = getSize();
			int height = (int) size.getHeight();
			setToolTipText(null);
			try {
				if (PageTaskExecutionProgressBar.this.pageTaskExecutionProgressModel.getNumTargetPages() == 0) {
					return;
				}
				// int unit = width /
				// OMRSessionProgressBar.this.model.getNumTargetPages();
				// unit = (unit == 0)? 1:unit;
				if (0 <= y && y < height) {
					PageTaskExecutionProgressBar.this.mousePositionX = x;
					update();
					return;
				} else {
					clear();
				}
			} catch (ConcurrentModificationException ignore) {
			}
		}
	}

	public boolean isIndeterminate() {
		return (0 <= this.indeterminate);
	}

	public void setIndeterminate(boolean isIndeterminate) {
		if (isIndeterminate) {
			this.indeterminate = 0.0f;
			if (this.executor == null) {
				this.executor = Executors.newSingleThreadExecutor(new GroupThreadFactory(
						"SessionProgressBar", Thread.MIN_PRIORITY, true));
			}
			this.indeterminateAnimationFuture = this.executor.submit(new Runnable() {
				public void run() {
					while (0 <= PageTaskExecutionProgressBar.this.indeterminate) {
						PageTaskExecutionProgressBar.this.indeterminate += 0.033f;
						update();
						try {
							Thread.sleep(33);
						} catch (InterruptedException ignore) {
						}
					}
				}
			});

		} else {
			if (this.indeterminateAnimationFuture != null) {
				this.indeterminateAnimationFuture.cancel(true);
				this.indeterminate = -1.0f;
			}
		}
	}

	@Override
	public void paintComponent(Graphics g) {
		super.paintComponent(g);

		Dimension size = getSize();
		int width = (int) size.getWidth();
		int height = (int) size.getHeight();

		g.setColor(Color.BLACK);
		g.fillRect(0, 0, (int) size.getWidth(), (int) size.getHeight());

		// int FONT_WIDTH = 14;
		int FONT_HEIGHT = 14;
		// int statusMesageWidth = FONT_WIDTH *
		// this.model.getStatusMessage().length();

		if (this.pageTaskExecutionProgressModel == null || 0 <= this.indeterminate
				|| this.pageTaskExecutionProgressModel.getNumTargetPages() == 0) {
			paintIndeterminate(g, size, width);
			g.setColor(Color.WHITE);
			g.drawString(this.pageTaskExecutionProgressModel.getStatusMessage(), 
					4, height / 2 + FONT_HEIGHT / 2);
			return;
		}

		paintProgressBarCore(g, width, height);
		
		paintErrorItems(g, width, height, FONT_HEIGHT);
	}

	private void paintProgressBarCore(Graphics g, int width, int height) {
		g.setColor(this.pageTaskExecutionProgressModel.getColTotal());
		g.fillRect(0, 0, width, height);

		TaskNumberCounter taskNumberCounter = this.pageTaskExecutionProgressModel.createTaskNumberCounter();
		int total = taskNumberCounter.getNumTotal();
		
		float unit = (float) width / total;
		
		g.setColor(this.pageTaskExecutionProgressModel.getColPrepared());
		g.fillRect((int)( getWidth() - unit * taskNumberCounter.getNumPrepared()),
				0, 
				(int)( unit * taskNumberCounter.getNumPrepared()), 
				height);

		g.setColor(this.pageTaskExecutionProgressModel.getColReused());
		g.fillRect(0, 0,
				(int)(unit * taskNumberCounter.getNumReused()),
				height);

		g.setColor(this.pageTaskExecutionProgressModel.getColExternalized());
		g.fillRect((int)(unit * taskNumberCounter.getNumReused()), 0,
				(int)(unit * taskNumberCounter.getNumExternalized()),
				height);
		
		g.setColor(this.pageTaskExecutionProgressModel.getColSubmitted());		
		g.fillRect((int)(unit * (taskNumberCounter.getNumReused() + taskNumberCounter.getNumExternalized())),
				0,
				(int)(unit * taskNumberCounter.getNumSubmitted()),
				height);
		
		g.setColor(this.pageTaskExecutionProgressModel.getColLeasedLocal());
		g.fillRect((int)(unit * (taskNumberCounter.getNumReused() + taskNumberCounter.getNumExternalized() + taskNumberCounter.getNumSubmitted())),
				0, 
				(int)(unit * taskNumberCounter.getNumLocalLeased()), 
				height);
		
		g.setColor(this.pageTaskExecutionProgressModel.getColLeasedRemote());
		g.fillRect((int)(unit * (taskNumberCounter.getNumReused() + taskNumberCounter.getNumExternalized() + taskNumberCounter.getNumSubmitted() + taskNumberCounter.getNumLocalLeased())),
				0,
				(int)(unit * taskNumberCounter.getNumRemoteLeased()),
				height);
	}

	private void paintErrorItems(Graphics g, int width, int height,
			int FONT_HEIGHT) {
		g.setColor(this.pageTaskExecutionProgressModel.getColError());

		int unit = width / this.pageTaskExecutionProgressModel.getNumTargetPages() + 1;

		try {
			for (Map.Entry<String, PageTaskErrorEntry> taskErrorEntry : this.pageTaskExecutionProgressModel
					.getErrorPathToTaskErrorEntryMap().entrySet()) {
				PageTaskErrorEntry e = taskErrorEntry.getValue();
				int errorIndex = e.getIndex();
				PageTaskError taskError = e.getTaskError();
				int x1 = errorIndex * width / this.pageTaskExecutionProgressModel.getNumTargetPages();
				if (x1 <= this.mousePositionX && this.mousePositionX <= x1 + unit) {
					g.setColor(Color.RED);
					g.fillRect(x1, 0, unit, height);
					g.setColor(this.pageTaskExecutionProgressModel.getColError());
					String message = taskError.getSource().getFileResourceID().getRelativePath() + "="
							+ taskError.getLocalizedMessage();
					PageTaskExecutionProgressBar.this.pageTaskExecutionProgressModel.setStatusMessage(message);
				} else {
					g.fillRect(x1, 0, unit, height);
				}
			}
			g.setColor(Color.WHITE);
			g.drawString(this.pageTaskExecutionProgressModel.getStatusMessage(), 0, height / 2 + FONT_HEIGHT / 2);
		} catch (ConcurrentModificationException ignore) {
		}
	}

	private void paintIndeterminate(Graphics g, Dimension size, int width) {
		g.setColor(Color.GRAY);
		g.fillRect(1, 1, (int) size.getWidth() - 1, (int) size.getHeight() - 1);
		g.setColor(Color.LIGHT_GRAY);
		if (((int) this.indeterminate) % 2 == 0) {
			g.fillRect((int) (width * (this.indeterminate % 1)), 0, width / 10, (int) size.getHeight());
		} else {
			g.fillRect(width - (int) (width * (this.indeterminate % 1)), 0, width / 10, (int) size
					.getHeight());
		}
		g.setColor(Color.BLACK);
		g.drawRect(0, 0, (int) size.getWidth() - 1, (int) size.getHeight() - 1);
	}

}
